import { Injectable } from "@angular/core";
import * as braintree from "braintree-web";
import { PageType } from "../../models/braintree-payment-methods.model";
import { FormGroup } from "@angular/forms";
import { CheckoutData } from "../../models/braintree-payment-data.model";
import { BraintreeCheckoutService } from "../checkout/braintree-checkout.service";
import { BraintreeUtilsService } from "../utils/braintree-utils.service";
import {GlobalMessageService, GlobalMessageType} from "@spartacus/core";

@Injectable({
  providedIn: "root",
})
export class UsBankAccountCheckoutService {
  protected deviceData: string;

  constructor(
    protected braintreeUtils: BraintreeUtilsService,
    protected braintreeCheckoutService: BraintreeCheckoutService,
    protected globalMessageService: GlobalMessageService
  ) {}

  tokenizeUsBankAccount(
    checkoutData: CheckoutData,
    usBankAccountForm: FormGroup,
    mandateText: string,
    pageType: PageType
  ) {
    this.prepareRequest(usBankAccountForm);
    this.initializeUsBankAccount(checkoutData, pageType, usBankAccountForm, mandateText);
  }

  private initializeUsBankAccount(
    checkoutData: CheckoutData,
    pageType: PageType,
    usBankAccountForm: FormGroup,
    mandateText: string
  ) {
    this.braintreeUtils.createClientInstance(
      checkoutData.configurationData,
      (client: any, deviceData: string) => {
        this.deviceData = deviceData;
        this.createUsBankAccount(
          client,
          pageType,
          this.prepareRequest(usBankAccountForm),
          mandateText
        );
      }
    );
  }

  private createUsBankAccount(
    client: any,
    pageType: PageType,
    usBankAccountRequest: any,
    mandateText: string
  ) {
    braintree.usBankAccount.create(
      {
        client: client,
      },
      (err, usBankAccountInstance) => {
        if (err) {
          console.error("Error creating Us Bank Account:", err);
          return;
        }

        usBankAccountInstance.tokenize(
          {
            bankDetails: usBankAccountRequest,
            mandateText: mandateText,
          },
          (tokenizeErr, payload) => {
            if (tokenizeErr) {
              this.errorInvalidPaymentDetails();
              console.error("An error occurred:", tokenizeErr.message);
            } else {
              this.braintreeCheckoutService.processUsBankAccount(
                payload,
                usBankAccountRequest,
                usBankAccountRequest.billingAddress,
                this.deviceData,
                pageType
              );
            }
          }
        );
      }
    );
  }

  private prepareRequest(usBankAccountForm: FormGroup): any {
    var usBankAccountRequest: any = {
      accountNumber: usBankAccountForm.value.accountNumber,
      routingNumber: usBankAccountForm.value.routingNumber,
      accountType: usBankAccountForm.value.accountType.type,
      billingAddress: {
        streetAddress: usBankAccountForm.value.streetAddress,
        extendedAddress: usBankAccountForm.value.extendedAddress,
        locality: usBankAccountForm.value.locality,
        region: usBankAccountForm.value.region.isocodeShort,
        postalCode: usBankAccountForm.value.postalCode,
      },
    };
    if (usBankAccountForm.value.ownershipType.type === 'Personal') {
        usBankAccountRequest.ownershipType = 'personal';
        usBankAccountRequest.firstName = usBankAccountForm.value.firstName;
        usBankAccountRequest.lastName = usBankAccountForm.value.lastName;
    }
    if (usBankAccountForm.value.ownershipType.type === 'Business') {
      usBankAccountRequest.ownershipType = 'business';
      usBankAccountRequest.businessName = usBankAccountForm.value.businessName;
  }
    return usBankAccountRequest;
  }

  public errorInvalidPaymentDetails() {
    this.globalMessageService.add(
        {key: 'error.invalidUsBankAccount'},
        GlobalMessageType.MSG_TYPE_ERROR, 5000
    );
  }
}
