import {Injectable} from '@angular/core';
import {BraintreeUtilsService} from '../utils/braintree-utils.service';
import {CheckoutData} from '../../models/braintree-payment-data.model';
import * as braintree from 'braintree-web';
import {
  ActiveCartService,
  Address,
  GlobalMessageService,
  GlobalMessageType,
  OCC_USER_ID_ANONYMOUS,
  Principal, 
  TranslationService
} from '@spartacus/core';
import {
  CheckoutDeliveryFacade,
} from '@spartacus/checkout/root';
import {take} from 'rxjs/operators';
import {BraintreeCheckoutService} from '../checkout/braintree-checkout.service';
import {BraintreePaymentMethodsErrorHandlerService} from '../errorHandling';

@Injectable({
  providedIn: 'root'
})
export class BraintreeLocalPaymentMethodsService {

  protected deliveryAddress: Address;
  protected cartAmount;
  protected userEmail;
  protected lpmWindowOpenFailed;

  constructor(
    protected braintreeUtils: BraintreeUtilsService,
    protected checkoutDeliveryService: CheckoutDeliveryFacade,
    protected activeCartService: ActiveCartService,
    protected braintreeCheckoutService: BraintreeCheckoutService,
    protected braintreeErrorHandlerService: BraintreePaymentMethodsErrorHandlerService,
    protected globalMessageService: GlobalMessageService,
    protected translation: TranslationService
  ) {}

  initializeLocalPaymentMethod(checkoutData: CheckoutData, lpmType: string): void{
    this.setDeliveryAddressValue();
    this.setCartAmountAndUserEmail();
    const parsedUrl = new URL(window.location.href);
    const baseUrl = parsedUrl.origin;

    this.braintreeUtils.createClientInstance(checkoutData.configurationData, (client, deviceData) => {

      braintree.localPayment.create({
        client,
        merchantAccountId: checkoutData.configurationData.currencyMerchantAccountId
      }, (localPaymentErr, paymentInstance)  => {

        if (localPaymentErr) {
          console.error('Error creating local payment:', localPaymentErr);
          return;
        }

        paymentInstance.startPayment({
          paymentType: lpmType,
          amount: this.cartAmount,
          fallback: {
            url: baseUrl + '/localPaymentMethods/fallback?currency=' + checkoutData.configurationData.currency,
            buttonText: 'Complete Payment'
          },
          currencyCode: checkoutData.configurationData.currency,
          email: this.userEmail,
          givenName: this.deliveryAddress.firstName,
          surname: this.deliveryAddress.lastName,
          address: {
            countryCode: this.deliveryAddress.country.isocode,
          },
          shippingAddressRequired: false,
          onPaymentStart: (data, start) => {
            // NOTE: It is critical here to store data.paymentId on your server
            //       so it can be mapped to a webhook sent by Braintree once the
            //       buyer completes their payment. See Start the payment
            //       section for details.

            this.braintreeCheckoutService.savePaymentIdForLPM(data.paymentId, start);
          }
        }, (startPaymentError, payload) => {
          if (startPaymentError) {
            if (startPaymentError.code === 'LOCAL_PAYMENT_POPUP_CLOSED' || startPaymentError.code === 'LOCAL_PAYMENT_WINDOW_CLOSED') {
              console.error('Customer closed Local Payment popup.');
            } else if (startPaymentError.code === 'LOCAL_PAYMENT_START_PAYMENT_FAILED'){
              console.error('LPM failed!', startPaymentError);
              this.handleLPMFailedError(startPaymentError);
            } else if (startPaymentError.code === 'LOCAL_PAYMENT_WINDOW_OPEN_FAILED'){
              console.error('LMP failed!', startPaymentError);
              this.translation.translate('error.lpm_window_open_failed').subscribe(value => this.lpmWindowOpenFailed = value);
              alert(this.lpmWindowOpenFailed);
            } else if (startPaymentError.details?.originalError?.details?.originalError?.error?.message !== undefined) {
              console.error('LMP failed!', startPaymentError);
              this.braintreeErrorHandlerService
              .getErrorMessageByDetailedMessage(startPaymentError.details?.originalError?.details?.originalError?.error?.message)
              .subscribe(message => {
                this.showErrorMessage(message);
              });
            } else {
              console.error('LMP failed!', startPaymentError);
              this.translation.translate('error.lpm_payment_failed')
              .subscribe(message => {
                this.showErrorMessage(message);
              });
            }
          } else {
            // Send the nonce to your server to create a transaction
            this.braintreeCheckoutService.processLocalPaymentMethodResponse(payload, deviceData);
          }
        });
      });
    });

  }

  initializeLPMFallback(checkoutData: CheckoutData, currencyFromFallbackURL: string): void {
    this.braintreeUtils.createClientInstance(checkoutData.configurationData, (client, deviceData) => {
      braintree.localPayment.create({
        client,
        merchantAccountId: checkoutData.configurationData.currencyMerchantAccountId
      }, (localPaymentErr, paymentInstance) => {

        if (paymentInstance.hasTokenizationParams()) {
          paymentInstance.tokenize().then((payload) => {

            // send payload.nonce to your server
            this.braintreeCheckoutService.processLPMFallback(payload, deviceData, currencyFromFallbackURL);

          }).catch((tokenizeError) => {
            // handle tokenization error
          });
        } else {
          // if this page should only be reached when
          // recovering from a mobile app switch,
          // display an error for not having the
          // correct params in the query string
        }
      });
    });

  }

  setDeliveryAddressValue(): void{
    this.checkoutDeliveryService
      .getDeliveryAddress()
      .pipe(take(1))
      .subscribe((address) => {
        this.deliveryAddress = address;
      });
  }

  setCartAmountAndUserEmail(): void{
    this.activeCartService
      .getActive()
      .subscribe(cart => {
        this.cartAmount = cart.totalPrice.value;
        this.userEmail = cart.user.uid;
        this.setUserEmail(cart.user);
      })
      .unsubscribe();
  }

  setUserEmail(user: Principal): void{
    if (user.name === OCC_USER_ID_ANONYMOUS){
      let strings = user.uid.split('|');
      this.userEmail = strings[strings.length - 1];
    }else {
      this.userEmail = user.uid;
    }
  }

  private handleLPMFailedError(error): void {
    if (typeof error !== 'undefined' || error !== 'undefined') {
        let messageText = error.message;
        if (typeof messageText === 'undefined' || messageText === 'undefined') {
          // Undefined error
        } else {
          this.braintreeErrorHandlerService.getErrorMessage(error.code.toLowerCase()).subscribe(res => {
            if (res !== undefined) {
              messageText = res;
            }
            this.globalMessageService.add(
                {
                  key: 'error.lpm_payment_failed',
                  params: {reason: messageText},
                },
                GlobalMessageType.MSG_TYPE_ERROR
            );
          });
        }
      }
    }

  private showErrorMessage(message): void {
    this.globalMessageService.add(
        message,
        GlobalMessageType.MSG_TYPE_ERROR
    );
  }

}
