import {ModalService} from '@spartacus/storefront';
import {BraintreeUtilsService} from '../utils/braintree-utils.service';
import {ActivatedRoute} from '@angular/router';
import {
  BraintreePaymentDetails,
  CheckoutData,
  CreditCardPaymentMethod, PayPalConfiguration,
} from '../../models/braintree-payment-data.model';
import {Injectable} from '@angular/core';
import * as braintree from 'braintree-web';
import {
  ActiveCartService,
  Address,
  Cart,
  GlobalMessageService,
  PaymentDetails,
  GlobalMessageType,
  TranslationService,
  EventService
} from '@spartacus/core';
import {CheckoutDeliveryFacade} from '@spartacus/checkout/root';
import { BraintreeCheckoutService } from '../checkout/braintree-checkout.service';
import {BraintreePaymentDetailsService} from '../../facade/braintree-payment-details.service';
import {PageType, Fields} from '../../models/braintree-payment-methods.model';
import {CheckoutStepService} from '@spartacus/checkout/components';

@Injectable({
  providedIn: 'root',
})
export class BraintreeHostedFieldsCheckoutService {

  protected tokenizeHostedFieldsEvent: any;
  protected tokenizeCVVCheckEvent: any;
  protected checkHostedFieldsValidityCallback: any;
  protected deviceData: string;
  protected billingAddress: Address;
  protected sameAsShipping: boolean;
  protected pageType: PageType;
  protected activatedRoute: ActivatedRoute;

  constructor(
    protected braintreeUtils: BraintreeUtilsService,
    protected checkoutStepService: CheckoutStepService,
    protected braintreeCheckoutService: BraintreeCheckoutService,
    protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
    protected activeCartService: ActiveCartService,
    protected globalMessageService: GlobalMessageService,
    protected modalService: ModalService,
    protected translationService: TranslationService,
    protected checkoutDeliveryService: CheckoutDeliveryFacade,
    protected eventService: EventService
  ) {}

  initializeHostedFields(
      checkoutData: CheckoutData,
      buttonContainer: HTMLElement,
      pageType: PageType
  ): void {
    this.pageType = pageType;

    this.braintreeUtils.createClientInstance(
        checkoutData.configurationData,
        (client: any, deviceData: string) => {
          this.deviceData = deviceData;
          this.createHostedFields(
              client,
              checkoutData.configurationData,
              checkoutData.creditCardPaymentMethod,
              buttonContainer
          );
        }
    );
  }

  initializeCVVCheck(checkoutData: CheckoutData) {
     this.braintreeUtils.createClientInstance(
      checkoutData.configurationData,
      (client: any, deviceData: string) => {
        this.deviceData = deviceData;
        this.createCVVCheck(
            client
        );
      }
    );
  }

  tokenizeCVVVerification(selectedPaymentDetails: PaymentDetails, checkoutData: CheckoutData, activatedRoute: ActivatedRoute) {
    this.activatedRoute = activatedRoute;
    this.tokenizeCVVCheckEvent(selectedPaymentDetails, checkoutData);
  }

  private createCVVCheck(client: any) {
    braintree.hostedFields.create({
      client: client,
      fields: {
          cvv: {
            container: '#cvv'
          }
      }
  }, (err, hostedFieldsInstance) => {

      if (err) {
          console.error(err);
          return;
      }

      //Event triggered on submitting CVV check
      this.tokenizeCVVCheckEvent = (selectedPaymentDetails: PaymentDetails, checkoutData: CheckoutData) => {
        const cvvCheckPromise = () => {
          return new Promise<string> ((resolve, reject) => {
          hostedFieldsInstance.tokenize((err, payload) => {
            if (err) reject(err);
            let nonce = payload.nonce;
            resolve(nonce);
          });
        })
      }

      cvvCheckPromise()
        .then(nonce => {
            if (
              (checkoutData.creditCardPaymentMethod.secure3d || checkoutData.creditCardPaymentMethod.secure3dFallback) &&
              (selectedPaymentDetails.subscriptionId === 'CreditCard'
                  || selectedPaymentDetails.subscriptionId === 'AndroidPayCard'
                  || selectedPaymentDetails.subscriptionId === 'VisaCheckoutCard'
              )) {
            this.braintreePaymentDetailsService.getBraintreePaymentDetailsById(selectedPaymentDetails.id, Fields.Full)
            .subscribe(paymentDetails => {
              if (checkoutData.creditCardPaymentMethod.secure3d || paymentDetails.shouldPerform3dSecure) {
                this.initialise3dSecure(
                    paymentDetails,
                    checkoutData,
                    PageType.BILLING,
                    nonce);
              } else {
                this.braintreeCheckoutService.selectBraintreePaymentMethod(selectedPaymentDetails.id, null, false, this.activatedRoute, nonce);
              }
            });
          } 
        });
      };
    });
  }

  private createHostedFields(
      client: any,
      configurationData: PayPalConfiguration,
      creditCardData: CreditCardPaymentMethod,
      buttonContainer: HTMLElement
  ): void {
    this.translationService.translate('paymentForm.accountHolderName.placeholder').subscribe(accountPlaceHolder => {
      braintree.hostedFields.create(
          {
            client,
            styles: {
              // Styling element state
              ':focus': {
                color: 'blue',
              },
              '.valid': {
                color: 'green',
              },
              '.invalid': {
                color: 'red',
              },
            },
            fields: {
              cardholderName: {
                container: '#cardholderName',
                placeholder: accountPlaceHolder
              },
              number: {
                container: '#cardNumber',
              },
              expirationDate: {
                container: '#expiration-date',
                placeholder: 'MM/YY',
              },
              cvv: {
                container: '#cvv',
              },
            },
          },
          (hostedFieldsErr, hostedFieldsInstance) => {
            const ERROR_FILED_ID = 'FieldError';
            const CONTAINER_BORDER_COLOR_ERROR = 'border-color: var(--cx-color-danger)';

            hostedFieldsInstance.on('empty', function(event) {
              event.fields[event.emittedBy].container.setAttribute('style', CONTAINER_BORDER_COLOR_ERROR);
              document.getElementById(event.emittedBy + ERROR_FILED_ID).setAttribute('style', '');
            });

            hostedFieldsInstance.on('blur', function(event) {
              if (event.fields[event.emittedBy].isEmpty) {
                event.fields[event.emittedBy].container.setAttribute('style', CONTAINER_BORDER_COLOR_ERROR);
                document.getElementById(event.emittedBy + ERROR_FILED_ID).setAttribute('style', '');
              }
            });

            hostedFieldsInstance.on('notEmpty', function(event) {
              event.fields[event.emittedBy].container.setAttribute('style', '');
              document.getElementById(event.emittedBy + ERROR_FILED_ID).setAttribute('style', 'display:none');
            });

            if (hostedFieldsErr) {
              this.braintreeUtils.handleClientError(hostedFieldsErr);
              return;
            }

            // Call back for checking form validity
            this.checkHostedFieldsValidityCallback = () => {
              const state = hostedFieldsInstance.getState();
              let result = true;
              Object.keys(state.fields).forEach((key) => {
                if (!state.fields[key].isValid) {
                  state.fields[key].container.setAttribute('style', CONTAINER_BORDER_COLOR_ERROR);
                  document.getElementById(key + ERROR_FILED_ID).setAttribute('style', '');
                  result = false;

                  this.globalMessageService.add(
                      {
                        key: 'error.invalidCard',
                      },
                      GlobalMessageType.MSG_TYPE_ERROR
                  );
                }
              });

              return result;
            };

            // Add a click event listener to PayPal image
            // tslint:disable-next-line:prefer-const
            this.tokenizeHostedFieldsEvent = () => {
              // initialize paypal authorization
              hostedFieldsInstance.tokenize({
                authenticationInsight: {
                  merchantAccountId: configurationData.currencyMerchantAccountId
                }
              }, (tokenizeErr: any, payload: any) => {
                if (tokenizeErr) {
                  this.braintreeUtils.handleClientError(tokenizeErr);
                } else {
                  if (this.is3DSecureShouldBePerformed(payload, creditCardData, this.pageType)) {
                    (<HTMLInputElement> document.getElementById("bt-hosted-fields-continue-btn")).disabled = true;
                    this.verify3DSecure(
                        client,
                        payload,
                        payload.details.cardholderName,
                        creditCardData.skip3dSecureLiabilityResult
                    );
                  } else {
                    this.braintreeCheckoutService.processHostedFields(
                        payload,
                        payload.details.cardholderName,
                        this.sameAsShipping,
                        this.billingAddress,
                        this.deviceData,
                        false,
                        this.pageType
                    );
                  }
                }
              });
            };
          }
      );
    });

  }

  is3DSecureShouldBePerformed(payload, creditCardData: CreditCardPaymentMethod, pageType: PageType): boolean {
    const environment = payload.authenticationInsight.regulationEnvironment;
    var isSecure3dFallbackEnabled = creditCardData.secure3dFallback
        && (environment === 'psd2' || environment === 'unavailable');
    var is3DSecureTurnOn = creditCardData.secure3d || isSecure3dFallbackEnabled;

    if (pageType === PageType.MY_ACCOUNT) {
        return is3DSecureTurnOn && creditCardData.secure3dOnMyAccount;
    } else if (pageType === PageType.BILLING) {
        return is3DSecureTurnOn;
    }
    return false;
  }

  tokenizeHostedFields(sameAsShipping: boolean, billingAddress: Address): void {
    this.billingAddress = billingAddress;
    this.sameAsShipping = sameAsShipping;
    this.tokenizeHostedFieldsEvent();
  }

  isHostedFieldsValid(): boolean {
    return this.checkHostedFieldsValidityCallback();
  }

  initialise3dSecure(paymentMethod: BraintreePaymentDetails, checkoutData: CheckoutData, pageType: PageType, cvvNonce?: string, activatedRoute?: ActivatedRoute): boolean {
    (<HTMLInputElement> document.getElementById("bt-billing-continue-btn")).disabled = true;
    this.pageType = pageType;
    braintree.client.create(
        {
          authorization: checkoutData.configurationData.client_id,
        },
        (clientErr, clientInstance) => {
          if (clientErr) {
            this.braintreeUtils.handleClientError(clientErr);
            return;
          }
          braintree.threeDSecure.create(
              {
                version: 2,
                client: clientInstance,
              },
              (err, threeDSecure) => {
                threeDSecure.verifyCard(
                    this.get3DSVerificationData(paymentMethod.paymentMethodNonce, paymentMethod.bin, paymentMethod.billingAddress),
                    (error: any, response: any) => {
                      if (error) {
                        this.globalMessageService.add(
                            error.toString(),
                            GlobalMessageType.MSG_TYPE_ERROR
                        );
                        return;
                      } else {
                        // 3DSecure finished, add 3DSecure returned nonce
                        const liabilityShifted = response.liabilityShifted;
                        const liabilityShiftPossible = response.liabilityShiftPossible;
                        // allow process card if 3dSecureLiabilityResult is skipped by merchant
                        if (
                            liabilityShifted ||
                            checkoutData.creditCardPaymentMethod.skip3dSecureLiabilityResult || !liabilityShiftPossible
                        ) {
                          const route = activatedRoute ? activatedRoute : this.activatedRoute;
                          this.braintreeCheckoutService.selectBraintreePaymentMethod(paymentMethod.id, response.nonce, true, route, cvvNonce);
                        } else {
                          this.show3DSecureMessage('error.unsecuredCard');
                        }
                      }
                      (<HTMLInputElement> document.getElementById("bt-billing-continue-btn")).disabled = false;
                    }
                );
              }
          );
        }
    );
    return false;
  }

  private verify3DSecure(
      clientInstance: any,
      paymentResponse: any,
      accountHolderName: string,
      skip3dSecureLiabilityResult: boolean
  ): void {
    braintree.threeDSecure.create(
        {
          version: 2,
          client: clientInstance,
        },
        (err, threeDSecure) => {
          threeDSecure.verifyCard(
              this.get3DSVerificationData(paymentResponse.nonce, paymentResponse.details.bin, null),
              (error, response) => {
                if (error) {
                  (<HTMLInputElement> document.getElementById("bt-hosted-fields-continue-btn")).disabled = false;
                  this.globalMessageService.add(
                      error.toString(),
                      GlobalMessageType.MSG_TYPE_ERROR
                  );
                  return;
                } else {
                  // 3DSecure finished
                  // add 3DSecure returned nonce
                  paymentResponse.nonce = response.nonce;
                  const liabilityShifted = response.liabilityShifted;
                  const liabilityShiftPossible = response.liabilityShiftPossible;
                  // allow process card if 3dSecureLiabilityResult is
                  // skipped by merchant
                  if (liabilityShifted || skip3dSecureLiabilityResult || !liabilityShiftPossible) {
                    paymentResponse.liabilityShifted = liabilityShifted;
                    this.braintreeCheckoutService.processHostedFields(
                        paymentResponse,
                        accountHolderName,
                        this.sameAsShipping,
                        this.billingAddress,
                        this.deviceData,
                        true,
                        this.pageType
                    );
                  } else {
                    (<HTMLInputElement> document.getElementById("bt-hosted-fields-continue-btn")).disabled = false;
                    this.show3DSecureMessage('error.unsecuredCard');
                  }
                }
              }
          );
        }
    );
  }

  private get3DSVerificationData(nonce: string, bin: string, addressOnCard: any): any {
    let parameters: any;
    let address: any;
    let cart: Cart;

    this.checkoutDeliveryService
    .getDeliveryAddress()
    .subscribe((res) => address = res)
    .unsubscribe();

    this.activeCartService
    .getActive()
    .subscribe((res) => cart = res)
    .unsubscribe();

    if (this.pageType === PageType.MY_ACCOUNT) {
        parameters = {
            amount: '0',
            email: null,
            billingAddress: {
                givenName: this.billingAddress.firstName,
                surname: this.billingAddress.lastName,
                phoneNumber: this.billingAddress.phone,
                streetAddress: this.billingAddress.line1,
                extendedAddress: this.billingAddress.line2,
                locality: this.billingAddress.town,
                postalCode: this.billingAddress.postalCode,
                countryCodeAlpha2: this.billingAddress.country?.isocode
            }
        }
    } else {
        parameters = {
            amount: cart.totalPrice.value,
            email: cart.user.uid,
            billingAddress: this.getBillingAddressFor3DS(address, addressOnCard),
            additionalInformation: {
                workPhoneNumber: address.phone,
                shippingGivenName: address.firstName,
                shippingSurname: address.lastName,
                shippingPhone: address.phone,
                shippingAddress: {
                    streetAddress: address.line1,
                    extendedAddress: address.line2,
                    locality: address.town,
                    postalCode: address.postalCode,
                    countryCodeAlpha2: address.country.isocode,
                    region: address.region?.isocodeShort
                }
            }
        }
    }

    let data = {
        amount: parameters.amount,
        nonce,
        bin,
        email: parameters.email,
        billingAddress: parameters.billingAddress,
        additionalInformation: parameters.additionalInformation,

        onLookupComplete: (data, next) => {
            next();
        },
        addFrame: (err: any, iframe: HTMLIFrameElement) => {
          const threeDSContainer = document.getElementById(
              'form-additionals'
          );
          threeDSContainer.appendChild(iframe);
          this.modalService.open(iframe, {centered: true});
        },
        removeFrame: () => {
          this.modalService.closeActiveModal();
        }
    };
    console.log(JSON.stringify(data));
    return data;
  }

  private getBillingAddressFor3DS(address: any, adressOnCard: any): any {
    let billingAddress: any;

    if (adressOnCard !== null) {
      billingAddress = {
        givenName: adressOnCard.firstName,
        surname: adressOnCard.lastName,
        phoneNumber: adressOnCard.phone,
        streetAddress: adressOnCard.line1,
        extendedAddress: adressOnCard.line2,
        locality: adressOnCard.town,
        postalCode: adressOnCard.postalCode,
        countryCodeAlpha2: adressOnCard.country.isocode
      };
      if (adressOnCard.region !== undefined) {
        billingAddress.region = adressOnCard.region.isocodeShort;
      }
    } else if (this.sameAsShipping) {
      billingAddress = {
        givenName: address.firstName,
        surname: address.lastName,
        phoneNumber: address.phone,
        streetAddress: address.line1,
        extendedAddress: address.line2,
        locality: address.town,
        postalCode: address.postalCode,
        countryCodeAlpha2: address.country.isocode
      };
      if (address.region !== undefined) {
        billingAddress.region = address.region.isocodeShort;
      }
    } else if (this.billingAddress !== undefined) {
      billingAddress = {
        givenName: this.billingAddress.firstName,
        surname: this.billingAddress.lastName,
        phoneNumber: this.billingAddress.phone,
        streetAddress: this.billingAddress.line1,
        extendedAddress: this.billingAddress.line2,
        locality: this.billingAddress.town,
        postalCode: this.billingAddress.postalCode,
        countryCodeAlpha2: this.billingAddress.country.isocode
      };
      if (this.billingAddress.region !== undefined) {
        billingAddress.region = this.billingAddress.region.isocodeShort;
      }
    }

    return billingAddress;
  }

  private show3DSecureMessage(message): void {
    this.globalMessageService.add(
        {
          key: message,
        },
        GlobalMessageType.MSG_TYPE_ERROR
    );
  }

}
