import {Injectable} from '@angular/core';
import * as braintree from 'braintree-web';
import {PageType} from '../../models/braintree-payment-methods.model';
import {CheckoutData} from '../../models/braintree-payment-data.model';
import { BraintreeCheckoutService } from '../checkout/braintree-checkout.service';
import {BraintreeUtilsService} from '../utils/braintree-utils.service';

@Injectable({
  providedIn: 'root',
})
export class ApplePayCheckoutService {

  protected deviceData: string;

  constructor(
    protected braintreeUtils: BraintreeUtilsService,
    protected braintreeCheckoutService: BraintreeCheckoutService,
  ) {}

  initalizeApplePay(
    checkoutData: CheckoutData,
    buttonType: PageType,
    button: HTMLElement
  ): void {
    const flow = this.braintreeUtils.getPaypalFlow(checkoutData, buttonType);

    this.braintreeUtils.createClientInstance(
      checkoutData.configurationData,
      (client: any, deviceData: string) => {
        this.deviceData = deviceData;
        this.createApplePay(client, flow, button, buttonType);
      }
    );
  }

  private createApplePay(
    client: any,
    flow: string,
    buttonSelector: HTMLElement,
    buttonType: PageType
  ): void {
    braintree.applePay.create(
      {
        client,
      },
      (applePayErr, applePayInstance) => {
        if (applePayErr) {
          console.error('Error creating applePayInstance:', applePayErr);
          return;
        }
        let createApplePaySession = () => {
          const amount = this.braintreeUtils.getTotalAmountBeforeRendering();
          if (buttonType === PageType.BILLING) {
            var paymentRequest = applePayInstance.createPaymentRequest({
              total: {
                label: flow,
                amount,
              },
              requiredBillingContactFields: ['postalAddress', 'name'],
              requiredShippingContactFields: ['email'],
            });
          } else {
            paymentRequest = applePayInstance.createPaymentRequest({
              requiredBillingContactFields: ['postalAddress', 'name'],
              requiredShippingContactFields: [
                'postalAddress',
                'name',
                'phone',
                'email',
              ],
              total: {
                label: flow,
                amount,
              },
            });
          }

          // @ts-ignore
          const session = new ApplePaySession(1, paymentRequest);

          session.onvalidatemerchant = (event) => {
            applePayInstance.performValidation(
              {
                validationURL: event.validationURL,
                displayName: 'Electronics',
              },
              (error, merchantSession) => {
                if (error) {
                  // Handle errors
                  console.error(error);
                  session.abort();
                  return;
                }
                session.completeMerchantValidation(merchantSession);
              }
            );
          };
          session.onpaymentauthorized = (event: any) => {
            applePayInstance.tokenize(
              {
                token: event.payment.token,
              },
              (error, payload) => {
                if (error) {
                  // Handle errors
                  console.error(error);
                  // @ts-ignore
                  session.completePayment(ApplePaySession.STATUS_FAILURE);
                  return;
                }

                // @ts-ignore
                session.completePayment(ApplePaySession.STATUS_SUCCESS);
                this.braintreeCheckoutService.processApplePayResponse(payload, event, buttonType, this.deviceData);
              }
            );
          };
          session.begin();
        };

        // @ts-ignore
        const promise = ApplePaySession.canMakePaymentsWithActiveCard(
          applePayInstance.merchantIdentifier
        );
        promise.then((canMakePaymentsWithActiveCard) => {
          if (!canMakePaymentsWithActiveCard) {
            this.disableApplePay(buttonSelector);
            createApplePaySession = null;
            return;
          } else {
            this.enableApplePay(buttonSelector);
            buttonSelector.addEventListener('click', () => {
              if (createApplePaySession !== null) {
                createApplePaySession();
              }
            });
          }
        });
      }
    );
  }

  isAvailableApplePay(): boolean {
    return (
      typeof (window as any).ApplePaySession !== 'undefined' &&
      // @ts-ignore
      ApplePaySession.canMakePayments()
    );
  }

  private disableApplePay(buttonSelector: HTMLElement): void {
    buttonSelector.classList.add('disabled-button');
   }

  private enableApplePay(buttonSelector: HTMLElement): void {
    buttonSelector.classList.remove('disabled-button');
  }
}
