import {
  ApplePayPaymentRequest,
  BraintreeCreditCardPaymentDetails,
  BraintreeUsBankAccountPaymentDetails,
  UsBankAccountAddress,
  Fields,
  GooglePayPaymentRequest,
  PageType, SrcPaymentRequest
} from '../../models/braintree-payment-methods.model';
import {Injectable, InjectionToken} from '@angular/core';
import {
  ActiveCartService,
  Address,
  Converter,
  ConverterService,
  EventService, GlobalMessageService, GlobalMessageType,
  OCC_USER_ID_ANONYMOUS,
  RoutingService,
  UserIdService,
} from '@spartacus/core';
import {
  CheckoutDeliveryFacade,
  CheckoutPaymentFacade,
  CheckoutFacade,
} from '@spartacus/checkout/root';
import {PayPalTokenizePayload} from 'braintree-web';
import {BraintreeExpressCheckoutService} from './braintree-express-checkout.service';
import {
  APPLE_PAY_ADDRESS_SERIALIZER,
  GOOGLE_PAY_ADDRESS_SERIALIZER,
  PAYPAL_ADDRESS_SERIALIZER,
  SRC_ADDRESS_SERIALIZER
} from '../../braintree-checkout-store/converters/converters';
import {BraintreePaymentDetailsService} from '../../facade/braintree-payment-details.service';
import {BraintreePaymentMethodsUtilsService} from '../utils/braintree-payment-methods-utils.service';
import {BraintreeMarkCheckoutService} from './braintree-mark-checkout.service';
import {take, takeWhile} from 'rxjs/operators';
import {BraintreeChangedPaymentDetailsEvent} from '../../events/braintree-payment-details.event';
import {ActivatedRoute} from '@angular/router';
import {CheckoutStepService} from '@spartacus/checkout/components';

@Injectable({
  providedIn: 'root'
})
export class BraintreeCheckoutService {

  constructor(
      protected checkoutPaymentService: CheckoutPaymentFacade,
      protected checkoutDeliveryService: CheckoutDeliveryFacade,
      protected converter: ConverterService,
      protected checkoutStepService: CheckoutStepService,
      protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
      protected braintreeExpressCheckoutService: BraintreeExpressCheckoutService,
      protected paymentMethodsUtilsService: BraintreePaymentMethodsUtilsService,
      protected braintreeMarkCheckoutService: BraintreeMarkCheckoutService,
      protected checkoutService: CheckoutFacade,
      protected routingService: RoutingService,
      protected userIdService: UserIdService,
      protected eventService: EventService,
      protected activeCartService: ActiveCartService,
      protected globalMessageService: GlobalMessageService,
  ) {
  }

  processPayPalResponse(
      payload: PayPalTokenizePayload,
      credit: boolean,
      deviceData: string,
      pageType: PageType,
      fundingSource?: string,
      changePaymentMethodAddress?: Address
  ): void {
    const isCredit = PageType.MY_ACCOUNT === pageType ? false : credit;

    if (this.isAnonymousExpressCheckout(pageType)) {
      this.activeCartService.addEmail(payload.details.email);
      this.activeCartService.getAssignedUser()
      .pipe(takeWhile(() => !this.activeCartService.isGuestCart(), true))
      .subscribe(() => {
        if (this.activeCartService.isGuestCart()) {
          this.braintreePaymentDetailsService.savePayPalPaymentDetails(isCredit, this.isShouldBeSavedPayment(pageType),
              payload, pageType, deviceData, Fields.Full, fundingSource);
        }
      });
    } else {
      this.braintreePaymentDetailsService.savePayPalPaymentDetails(isCredit, this.isShouldBeSavedPayment(pageType),
          payload, pageType, deviceData, Fields.Full, fundingSource);
      this.addConfirmationMessage(pageType);
    }

    this.processPaymentDetailsSetting(pageType, payload, PAYPAL_ADDRESS_SERIALIZER, changePaymentMethodAddress);
  }

  processGooglePayResponse(result, googlePayPaymentData, deviceData: string, pageType: PageType): void {
    const googlePayPaymentRequest: GooglePayPaymentRequest = {
      nonce: result.nonce,
      type: result.type,
      email: googlePayPaymentData.email,
      cardType: result.details.cardType,
      lastFour: result.details.lastFour,
      billingAddress: googlePayPaymentData.paymentMethodData.info.billingAddress
    };

    if (this.isAnonymousExpressCheckout(pageType)) {
      this.activeCartService.addEmail(googlePayPaymentRequest.email);
      this.activeCartService.getAssignedUser()
      .pipe(takeWhile(() => !this.activeCartService.isGuestCart(), true))
      .subscribe(() => {
        if (this.activeCartService.isGuestCart()) {
          this.braintreePaymentDetailsService.saveGooglePayPaymentDetails(
              googlePayPaymentRequest,
              this.isShouldBeSavedPayment(pageType),
              deviceData,
              Fields.Full
          );
        }
      });
    } else {
      this.braintreePaymentDetailsService.saveGooglePayPaymentDetails(
          googlePayPaymentRequest,
          this.isShouldBeSavedPayment(pageType),
          deviceData,
          Fields.Full
      );
      this.addConfirmationMessage(pageType);
    }

    this.processPaymentDetailsSetting(pageType, googlePayPaymentData.shippingAddress, GOOGLE_PAY_ADDRESS_SERIALIZER);
  }

  processVenmoResponse(payload, deviceData, pageType: PageType): void {
    let selectedAddressCode = '';
    if (PageType.MY_ACCOUNT === pageType) {
      selectedAddressCode = this.paymentMethodsUtilsService.getBillingAddress().id;
    } else {
      this.checkoutDeliveryService
      .getDeliveryAddress()
      .pipe(take(1))
      .subscribe((address) => {
        selectedAddressCode = address.id;
      });
    }
    this.braintreePaymentDetailsService.saveVenmoPaymentDetails(payload, selectedAddressCode, this.isShouldBeSavedPayment(pageType),
        deviceData, Fields.Full);

    this.braintreePaymentDetailsService.getSelectedPaymentDetails().subscribe(paymentDetails => {
      if (paymentDetails !== undefined) {
        if (PageType.BILLING === pageType) {
          this.braintreeMarkCheckoutService.processCheckout(paymentDetails);
        } else if (PageType.MY_ACCOUNT === pageType) {
          this.paymentMethodsUtilsService.getBackToPaymentMethodsEmitter().emit();
          this.addConfirmationMessage(pageType);
        }
      }
    });
  }

  processLocalPaymentMethodResponse(payload, deviceData): void {

    this.braintreePaymentDetailsService.processLocalPaymentMethod(payload, deviceData, Fields.Full);
  }

  processApplePayResponse(payload: any, event: any, pageType: PageType, deviceData: string): void {
    const applePayPaymentRequest: ApplePayPaymentRequest = {
      nonce: payload.nonce,
      type: payload.type,
      email: event.payment.shippingContact.emailAddress,
      cardType: payload.details.cardType,
      billingContact: event.payment.billingContact
    };
    applePayPaymentRequest.billingContact.countryCode = applePayPaymentRequest.billingContact.countryCode.toUpperCase();

    if (this.isAnonymousExpressCheckout(pageType)) {
      this.activeCartService.addEmail(applePayPaymentRequest.email);
      this.activeCartService.getAssignedUser()
      .pipe(takeWhile(() => !this.activeCartService.isGuestCart(), true))
      .subscribe(() => {
        if (this.activeCartService.isGuestCart()) {
          this.braintreePaymentDetailsService.saveApplePayPaymentDetails(
              applePayPaymentRequest,
              deviceData,
              Fields.Full
          );
        }
      });
    } else {
      this.braintreePaymentDetailsService.saveApplePayPaymentDetails(
          applePayPaymentRequest,
          deviceData,
          Fields.Full
      );
      this.addConfirmationMessage(pageType);
    }

    this.processPaymentDetailsSetting(pageType, event.payment.shippingContact, APPLE_PAY_ADDRESS_SERIALIZER);
  }

  processHostedFields(payload: any, accountHolderName: string, sameAsShipping: boolean, billingAddress: Address,
                      deviceData: string, is3dSecureFlow: boolean, pageType: PageType): void {
    let selectedAddressCode = '';
    const hostedFieldsResponse: BraintreeCreditCardPaymentDetails = {
      tokenizedCardData: payload
    };
    if (pageType === PageType.MY_ACCOUNT) {
      selectedAddressCode = billingAddress.id;
    } else {
      if (sameAsShipping) {
        this.checkoutDeliveryService.getDeliveryAddress().subscribe(address => {
          selectedAddressCode = address.id;
        }).unsubscribe();
      } else {
        hostedFieldsResponse.billingAddress = billingAddress;
        if (billingAddress.region) {
          billingAddress.region.isocode = billingAddress.country.isocode + '-' + billingAddress.region.isocodeShort;
        }
      }
    }
    hostedFieldsResponse.tokenizedCardData.details.cardholderName = accountHolderName;
    this.braintreePaymentDetailsService.saveCreditCardPaymentDetails(
        hostedFieldsResponse,
        selectedAddressCode,
        is3dSecureFlow,
        this.isShouldBeSavedPayment(pageType),
        deviceData,
        Fields.Full
    );
    this.addConfirmationMessageIfPaymentExist(pageType);
    this.processPaymentDetailsSetting(pageType);

  }

  processUsBankAccount(payload: any, usBankAccountForm: any, billingAddress: UsBankAccountAddress, deviceData: string, pageType: PageType): void {
    const usBankAccountResponse: BraintreeUsBankAccountPaymentDetails = {
      tokenizedUsBankAccount: payload,
      routingNumber: usBankAccountForm.routingNumber,
      accountNumber: usBankAccountForm.accountNumber
    };
    usBankAccountResponse.billingAddress = billingAddress;
    if (usBankAccountForm.ownershipType === 'personal') {
    usBankAccountResponse.billingAddress.firstName =  usBankAccountForm.firstName;
    usBankAccountResponse.billingAddress.lastName =  usBankAccountForm.lastName;
    }
    if (usBankAccountForm.ownershipType === 'business') {
    usBankAccountResponse.billingAddress.businessName =  usBankAccountForm.businessName;
    }
    this.braintreePaymentDetailsService.saveUsBankAccountPaymentDetails(
        usBankAccountResponse,
        this.isShouldBeSavedPayment(pageType),
        deviceData,
        Fields.Full
    );
    this.addConfirmationMessageIfPaymentExist(pageType);
    this.processPaymentDetailsSetting(pageType);
  }

  processSrcResponse(payload: any, deviceData: string, pageType: PageType): void {

    const srcPaymentRequest: SrcPaymentRequest = {
      nonce: payload.nonce,
      details: payload.details,
      description: payload.description,
      type: payload.type,
      billingAddress: payload.billingAddress,
      userEmail: payload.userData.userEmail,
      userFullName: payload.userData.userFullName
    };

    if (this.isAnonymousExpressCheckout(pageType)) {
      this.activeCartService.addEmail(srcPaymentRequest.userEmail);
      this.activeCartService.getAssignedUser()
      .pipe(takeWhile(() => !this.activeCartService.isGuestCart(), true))
      .subscribe(() => {
        if (this.activeCartService.isGuestCart()) {
          this.braintreePaymentDetailsService.saveSrcPaymentDetails(
              srcPaymentRequest,
              this.isShouldBeSavedPayment(pageType),
              deviceData,
              Fields.Full
          );
        }
      });
    } else {
      this.braintreePaymentDetailsService.saveSrcPaymentDetails(
          srcPaymentRequest,
          this.isShouldBeSavedPayment(pageType),
          deviceData,
          Fields.Full
      );
      this.addConfirmationMessage(pageType);
    }

    this.processPaymentDetailsSetting(pageType, payload.shippingAddress, SRC_ADDRESS_SERIALIZER);
  }

  processPaymentDetailsSetting<S, T>(pageType: PageType, addressSource?: any, injectionToken?: InjectionToken<Converter<S, T>>, addressForOverride?: Address
  ): void {
    this.braintreePaymentDetailsService.getSelectedPaymentDetails()
    .pipe(takeWhile(paymentDetails => paymentDetails === undefined, true))
    .subscribe(paymentDetails => {
      if (paymentDetails !== undefined) {
        if (PageType.CART === pageType) {
          let address = this.converter.convert(addressSource, injectionToken);
          if (addressForOverride !== undefined) {
            this.overridePayPalShippingAddress(addressForOverride, address);
          }
          this.braintreeExpressCheckoutService.processExpressCheckout(address, paymentDetails);
        } else if (PageType.BILLING === pageType) {
          this.braintreeMarkCheckoutService.processCheckout(paymentDetails);
        } else if (PageType.MY_ACCOUNT === pageType) {
          this.paymentMethodsUtilsService.getBackToPaymentMethodsEmitter().emit();
        }
      }
    });
  }

  isAnonymousExpressCheckout(pageType: PageType): boolean {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();
    return PageType.CART === pageType && OCC_USER_ID_ANONYMOUS === userId && !this.activeCartService.isGuestCart();
  }

  processLPMFallback(payload, deviceData, currencyFromFallbackURL: string): void {
    this.braintreePaymentDetailsService.placeOrderThroughFallback(payload, currencyFromFallbackURL, deviceData, Fields.Full);

    this.checkoutService.getPlaceOrderSuccess().subscribe(res => {
      if (res) {
        this.routingService.go({cxRoute: 'orderConfirmation'});
      }
    });
  }

  savePaymentIdForLPM(paymentId, start): void {
    this.braintreePaymentDetailsService.savePaymentIdForLPM(paymentId).subscribe(res => {
      start();
    });
  }

  selectBraintreePaymentMethod(paymentMethodId: string, paymentMethodNonce: string, is3dSecureFlow: boolean, activatedRoute: ActivatedRoute, cvvNonce?: string): void {
     this.braintreePaymentDetailsService.selectBraintreePaymentMethod(paymentMethodId, paymentMethodNonce, is3dSecureFlow, cvvNonce).subscribe(() => {
           this.checkoutStepService.next(activatedRoute)
           this.eventService.dispatch(new BraintreeChangedPaymentDetailsEvent())
        }
    );
  }

  isShouldBeSavedPayment(pageType: PageType): boolean {
    if (PageType.MY_ACCOUNT === pageType) {
      return true;
    } else {
      const savePaymentInfo = this.paymentMethodsUtilsService.getSavePaymentInfo();
      return savePaymentInfo !== undefined ? savePaymentInfo : false;
    }
  }

  private addConfirmationMessage(pageType: PageType) {
    if (PageType.MY_ACCOUNT === pageType) {
      this.globalMessageService.add(
          {key: 'myAccount.paymentMethodIsSuccessfullyAddedToTheWallet'},
          GlobalMessageType.MSG_TYPE_CONFIRMATION, 15000
      );
    }
  }

  private addConfirmationMessageIfPaymentExist(pageType: PageType): void {
    this.braintreePaymentDetailsService.getSelectedPaymentDetails()
        .pipe(takeWhile(paymentDetails => paymentDetails === undefined, true))
        .subscribe(paymentDetails => {
          if (paymentDetails !== undefined){
            this.addConfirmationMessage(pageType);
          }
        });
  }

  private overridePayPalShippingAddress(source: Address, target: Address): void {
    target.firstName = source.firstName;
    target.lastName = source.lastName;
    target.title = source.title;
    target.phone = source.phone;
  }

}
