import {Injectable} from "@angular/core";
import {
    CheckoutData,
    PayPalConfiguration,
    PayPalConnectComponent,
    PayPalConnectB2BData,
    PayPalConnectB2BRegisterData,
    PayPalConnectData, PayPalButtonStyle, PageType
} from "../models";
import * as braintree from 'braintree-web';
import {BraintreeUtilsService} from "../services/utils";
import {PaypalConnectConnector} from '../connectors';
import {AuthService, UserIdService, GlobalMessageService, GlobalMessageType} from '@spartacus/core';
import {Router} from '@angular/router';
import {from, Observable, of} from 'rxjs';
import * as CONST from "../models/braintree.constants";
import {BraintreePaymentMethodsErrorHandlerService} from "../services/errorHandling";
import {PaypalCheckoutService} from "../services/payment-methods";
import {BraintreePaymentDetailsService} from "./braintree-payment-details.service";

@Injectable({
    providedIn: 'root'
})
export class PaypalConnectService {

    constructor(
        protected braintreeUtilsService: BraintreeUtilsService,
        protected userIdService: UserIdService,
        protected paypalConnectConnector: PaypalConnectConnector,
        protected router: Router,
        protected auth: AuthService,
        protected paypalCheckoutService: PaypalCheckoutService,
        protected globalMessageService: GlobalMessageService,
        protected braintreePaymentDetailsService:BraintreePaymentDetailsService,
        protected braintreeErrorHandlerService: BraintreePaymentMethodsErrorHandlerService

    ) {
    }

    public initializeConnectWithPayPal(componentData: PayPalConnectComponent, configurationData: PayPalConfiguration): void {
        this.braintreeUtilsService.loadSdk(configurationData.payPalConnectScript,
            () => this.renderConnectWithPayPalButton(componentData, configurationData));
    }

    protected renderConnectWithPayPalButton(componentData: PayPalConnectComponent, configurationData: PayPalConfiguration): void {
        try {
            if (typeof (window as any).paypal !== undefined) {
                let authend = '';
                if (configurationData.environment === 'sandbox'){
                    authend = 'sandbox';
                }

                (window as any).paypal.use( ['login'], function (login) {
                    login.render ({
                        "appid": configurationData.client_id,
                        "authend": authend,
                        "scopes": configurationData.payPalConnectScopes,
                        "containerid": componentData.buttonDiv,
                        "responseType": configurationData.payPalConnectResponseType,
                        "theme": componentData.buttonTheme,
                        "buttonType": componentData.buttonType,
                        "buttonShape": componentData.buttonShape.toLocaleLowerCase(),
                        "buttonSize": componentData.buttonSize.toLocaleLowerCase(),
                        "fullPage": componentData.fullPage,
                        "returnurl": configurationData.payPalConnectReturnUrl,
                        "nonce": '1111111'
                    });
                });
            }
        } catch (err) {
            console.log(err.message);
        }
    }

    public exchangeAuthorizationCode(authorizationCode: string): Observable<PayPalConnectData> {
        let userId;
        this.userIdService
        .getUserId()
        .subscribe((occUserId) => (userId = occUserId))
        .unsubscribe();
        if (userId) {
            return this.paypalConnectConnector.exchangeAuthorizationCode(authorizationCode, userId);
        }
        return of(null);
    }

    public login(accessTokenGuid: string): Observable<boolean> {
        let userId;
        this.userIdService
        .getUserId()
        .subscribe((occUserId) => (userId = occUserId))
        .unsubscribe();
        if(userId) {
            this.paypalConnectConnector.login(accessTokenGuid, userId).subscribe(loginData => {
                from(this.auth.loginWithCredentials(loginData.login, loginData.accessTokenGuid)).subscribe(te => {
                    this.userIdService
                    .getUserId()
                    .subscribe((occUserId) => (userId = occUserId))
                    .unsubscribe();
                    this.paypalConnectConnector.afterLogin(accessTokenGuid, userId).subscribe(flag => {})
                });
            });
            return of(true);
        }
        return of(true);
    }

    public register(accessTokenGuid: string): Observable<string> {
        let userId;
        this.userIdService
        .getUserId()
        .subscribe((occUserId) => (userId = occUserId))
        .unsubscribe();
        if(userId) {
            return this.paypalConnectConnector.register(accessTokenGuid, userId);
        }
    }

    public exchangeAuthorizationCodeB2B(authorizationCode: string): Observable<PayPalConnectB2BData> {
        let userId;
        this.userIdService
        .getUserId()
        .subscribe((occUserId) => (userId = occUserId))
        .unsubscribe();
        if (userId) {
            return this.paypalConnectConnector.exchangeAuthorizationCodeB2B(authorizationCode, userId);
        }
        return of(null);
    }

    public registerB2BCustomer(
        accessTokenGuid: string,
        registerData: PayPalConnectB2BRegisterData
    ): Observable<string> {
        let userId;
        this.userIdService
        .getUserId()
        .subscribe((occUserId) => (userId = occUserId))
        .unsubscribe();
        if(userId) {
            registerData.accessTokenGuid = accessTokenGuid;
            return this.paypalConnectConnector.registerB2BCustomer(registerData, userId);
        }
        return of(undefined);
    }

    initializePayPalAutomaticSavePaymentMethod(
        buttonContainer,
        checkoutData: CheckoutData,
        buttonStyle: PayPalButtonStyle,
        pageType: PageType,
        onSuccessCallback,
        shouldSaveAddressInfo: boolean,
        shouldSavePaymentInfo: boolean,
        accessTokenGuid: string
    ): void {
        let payPalOptions;
        payPalOptions = this.paypalCheckoutService.createPayPalOptions(checkoutData, pageType);

        this.braintreeUtilsService.createClientInstance(checkoutData.configurationData, (client, deviceData) => {
            this.createPayPalCheckout(payPalOptions, client, buttonContainer, pageType, checkoutData, buttonStyle, deviceData, onSuccessCallback, shouldSaveAddressInfo, shouldSavePaymentInfo,accessTokenGuid);
        });
    }

    private createPayPalCheckout(
        paypalOptions,
        client,
        payPalButtonContainer,
        pageType: PageType,
        checkoutData: CheckoutData,
        buttonStyle: PayPalButtonStyle,
        deviceData: string,
        onSuccessCallback,
        shouldSaveAddressInfo: boolean,
        shouldSavePaymentInfo: boolean,
        accessTokenGuid: string
    ): void {
        if (payPalButtonContainer != null) {
            const commit = paypalOptions.paypalIntent === CONST.INTENT_SALE && paypalOptions.userAction === 'true';
            const isIntentValid = this.braintreeUtilsService.checkIntentOption(checkoutData.configurationData.intent);
            if (isIntentValid === false) {
                this.globalMessageService.add(
                    {key: 'intent.notSupported'},
                    GlobalMessageType.MSG_TYPE_ERROR);
                return;
            }

            braintree.paypalCheckout.create({
                autoSetDataUserIdToken: checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton,
                client
            }, (paypalCheckoutErr, paypalCheckoutInstance) => {
                paypalOptions.amount = this.braintreeUtilsService.getTotalAmountBeforeRendering()?.toFixed(2);
                this.braintreeUtilsService.loadPayPalSDK(checkoutData, paypalCheckoutInstance,
                    checkoutData.configurationData.intent, commit, paypalOptions.flow, pageType, () => {
                        console.log('PayPal Sdk was loaded');
                        payPalButtonContainer.innerHTML = '';

                        this.renderAutomaticPaymentMethodAddingButtons(paypalCheckoutInstance, payPalButtonContainer, paypalOptions, commit, buttonStyle, checkoutData, deviceData, onSuccessCallback, shouldSaveAddressInfo, shouldSavePaymentInfo, accessTokenGuid);
                    }, paypalOptions.amount);
            });
        }
    }

    private renderAutomaticPaymentMethodAddingButtons(
        paypalCheckoutInstance,
        payPalButtonContainer,
        paypalOptions,
        commit,
        buttonStyle: PayPalButtonStyle,
        checkoutData: CheckoutData,
        deviceData: string,
        onSuccessCallback,
        shouldSaveAddressInfo: boolean,
        shouldSavePaymentInfo: boolean,
        accessTokenGuid: string
        ): void {
        let fundingSource;
        const defaultShippingAddressEditable = paypalOptions.shippingAddressEditable;
        try {
            (window as any).paypalSdk.Buttons({
                style: buttonStyle,
                locale: checkoutData.configurationData.braintreeLocale,
                commit,
                onClick: (button) => {
                    fundingSource = button.fundingSource;
                    paypalOptions.amount = this.braintreeUtilsService.getTotalAmountBeforeRendering()?.toFixed(2);
                    this.updatePaypalOptions(paypalOptions, fundingSource, defaultShippingAddressEditable);
                },
                createBillingAgreement(): any {
                    return paypalCheckoutInstance.createPayment(paypalOptions);
                },
                onApprove: (data) => {
                    return paypalCheckoutInstance.tokenizePayment(data).then((payload) => {
                        this.braintreePaymentDetailsService.savePaymentDetailsAndAddressForNewUser(
                            accessTokenGuid,
                            onSuccessCallback,
                            payload,
                            shouldSaveAddressInfo,
                            shouldSavePaymentInfo,
                            deviceData,
                            fundingSource
                        );
                    });

                },
                onCancel(data): void {
                    console.log('User cancel PayPal flow');
                },

                onError: (err) => {
                    console.error('Error: ' + err, err);
                    this.handlePayPalClientError(err);
                }
            }).render(payPalButtonContainer);
        } catch (err) {
            console.log(err.message);
            this.handlePayPalButtonError(err.message);
        }
    }

    private updatePaypalOptions(paypalOptions, fundingSource, defaultShippingAddressEditable): void {
        if (fundingSource === CONST.PAYPAL_FUNDING_CARD) {
            paypalOptions.shippingAddressEditable = true;
        } else {
            paypalOptions.shippingAddressEditable = defaultShippingAddressEditable;
        }
    }

    private handlePayPalButtonError(errorMsg): void {
        if (typeof errorMsg !== 'undefined' || errorMsg !== 'undefined') {
            this.globalMessageService.add(
                errorMsg,
                GlobalMessageType.MSG_TYPE_ERROR
            );
        }
    }

    private handlePayPalClientError(error): void {
        if (typeof error !== 'undefined' || error !== 'undefined') {
            // skip validation error: use paypal method
            if ('User did not enter a payment method' !== error.message && 'PAYPAL_POPUP_CLOSED' !== error.code) {
                let messageText = error.message;
                if (typeof messageText === 'undefined' || messageText === 'undefined') {
                    // Undefined error
                } else {
                    this.braintreeErrorHandlerService.getErrorMessage(error.code.toLowerCase()).subscribe(res => {
                        if (res !== undefined) {
                            messageText = res;
                        }
                        this.globalMessageService.add(
                            {
                                key: 'error.provider',
                                params: {reason: messageText},
                            },
                            GlobalMessageType.MSG_TYPE_ERROR
                        );
                    });
                }
            }
        }
    }

}
