import {Observable, of} from 'rxjs';
import {catchError, map, mergeMap} from 'rxjs/operators';
import {Actions, Effect, ofType} from '@ngrx/effects';
import {BraintreePaymentDetailsConnector} from '../../connectors/braintree-payment-details.connector';
import {
  CartActions,
  EventService,
  normalizeHttpError,
  Occ,
  OCC_USER_ID_ANONYMOUS,
  SiteContextActions,
  UserActions,
  withdrawOn
} from '@spartacus/core';
import {
    CheckoutActions,
    CheckoutDeliveryConnector,
} from '@spartacus/checkout/core';
import {BraintreePaymentDetailsActions, BraintreeCheckoutActions} from '../actions';
import {Injectable} from '@angular/core';
import {DeliveryModeWasDeletedEvent} from '../../events';
import PaymentDetails = Occ.PaymentDetails;

@Injectable()
export class BraintreeCheckoutEffect {

  private contextChange$ = this.actions$.pipe(
    ofType(
      SiteContextActions.CURRENCY_CHANGE,
      SiteContextActions.LANGUAGE_CHANGE,
    )
  );


    @Effect()
  createVenmoPaymentDetails$: Observable<
    | UserActions.LoadUserPaymentMethods
    | CheckoutActions.CreatePaymentDetailsSuccess
    | BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess
    | CheckoutActions.CreatePaymentDetailsFail> = this.actions$.pipe(
    ofType(BraintreePaymentDetailsActions.CREATE_VENMO_PAYMENT_DETAILS),
    map((action: any) => action.payload),
    mergeMap((payload) => {
      // get information for creating a subscription directly with payment provider
      const venmoPaymentDetails: Observable<PaymentDetails> = this.checkoutPaymentConnector
        .saveVenmoPaymentDetails(payload.userId, payload.selectedAddressCode, payload.venmoPayment, payload.cartId,
          payload.shouldBeSaved, payload.deviceData, payload.fields);
      return this.processPaymentCreationResult(venmoPaymentDetails, payload);
    }),
    withdrawOn(this.contextChange$)
  );

  @Effect()
  createPayPalPaymentDetails$: Observable<
    | UserActions.LoadUserPaymentMethods
    | CheckoutActions.CreatePaymentDetailsSuccess
    | BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess
    | CheckoutActions.CreatePaymentDetailsFail> = this.actions$.pipe(
    ofType(BraintreePaymentDetailsActions.CREATE_PAYPAL_PAYMENT_DETAILS),
    map((action: any) => action.payload),
    mergeMap((payload) => {
      // get information for creating a subscription directly with payment provider
      const payPalPaymentDetails: Observable<PaymentDetails> = this.checkoutPaymentConnector
        .savePayPalPaymentDetails(payload.credit, payload.shouldBeSaved, payload.payPalRequest, payload.pageType,
          payload.userId, payload.cartId, payload.deviceData, payload.fields, payload.fundingSource);
      return this.processPaymentCreationResult(payPalPaymentDetails, payload);
    }),
    withdrawOn(this.contextChange$)
  );


  @Effect()
  createGooglePayPaymentDetails$: Observable<
    | UserActions.LoadUserPaymentMethods
    | CheckoutActions.CreatePaymentDetailsSuccess
    | BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess
    | CheckoutActions.CreatePaymentDetailsFail> = this.actions$.pipe(
    ofType(BraintreePaymentDetailsActions.CREATE_GOOGLE_PAY_PAYMENT_DETAILS),
    map((action: any) => action.payload),
    mergeMap((payload) => {
      // get information for creating a subscription directly with payment provider
      const googlePayPaymentDetails: Observable<PaymentDetails> = this.checkoutPaymentConnector
        .saveGooglePayPaymentDetails(payload.userId, payload.googlePayRequest, payload.shouldBeSaved, payload.cartId,
          payload.deviceData, payload.fields);
      return this.processPaymentCreationResult(googlePayPaymentDetails, payload);
    }),
    withdrawOn(this.contextChange$)
  );

  @Effect()
  createSrcPaymentDetails$: Observable<
      | UserActions.LoadUserPaymentMethods
      | CheckoutActions.CreatePaymentDetailsSuccess
      | BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess
      | CheckoutActions.CreatePaymentDetailsFail> = this.actions$.pipe(
      ofType(BraintreePaymentDetailsActions.CREATE_SRC_PAYMENT_DETAILS),
      map((action: any) => action.payload),
      mergeMap((payload) => {

        const srcPaymentDetails: Observable<PaymentDetails> = this.checkoutPaymentConnector
        .saveSrcPaymentDetails(payload.userId, payload.srcRequest, payload.shouldBeSaved, payload.cartId,
            payload.deviceData, payload.fields);
        return this.processPaymentCreationResult(srcPaymentDetails, payload);
      }),
      withdrawOn(this.contextChange$)
  );


  @Effect()
  createCreditCardPaymentDetails$: Observable<
    | UserActions.LoadUserPaymentMethods
    | CheckoutActions.CreatePaymentDetailsSuccess
    | BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess
    | CheckoutActions.CreatePaymentDetailsFail> = this.actions$.pipe(
    ofType(BraintreePaymentDetailsActions.CREATE_CREDIT_CARD_PAYMENT_DETAILS),
    map((action: any) => action.payload),
    mergeMap((payload) => {
      // get information for creating a subscription directly with payment provider
      const creditCardPaymentDetails: Observable<PaymentDetails> = this.checkoutPaymentConnector
        .saveCreditCardPaymentDetails(payload.userId, payload.paymentDetails, payload.selectedAddressCode, payload.is3dSecureFlow,
          payload.cartId, payload.shouldBeSaved, payload.deviceData, payload.fields);
      return this.processPaymentCreationResult(creditCardPaymentDetails, payload);
    }),
    withdrawOn(this.contextChange$)
  );

  @Effect()
  createUsBankAccountPaymentDetails$: Observable<
    | UserActions.LoadUserPaymentMethods
    | CheckoutActions.CreatePaymentDetailsSuccess
    | BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess
    | CheckoutActions.CreatePaymentDetailsFail> = this.actions$.pipe(
    ofType(BraintreePaymentDetailsActions.CREATE_US_BANK_ACCOUNT_PAYMENT_DETAILS),
    map((action: any) => action.payload),
    mergeMap((payload) => {
      // get information for creating a subscription directly with payment provider
      const usBankAccountPaymentDetails: Observable<PaymentDetails> = this.checkoutPaymentConnector
        .saveUsBankAccountPaymentDetails(payload.userId, payload.paymentDetails, payload.shouldBeSaved, payload.deviceData, payload.fields);
      return this.processPaymentCreationResult(usBankAccountPaymentDetails, payload);
    }),
    withdrawOn(this.contextChange$)
  );


  @Effect()
  createApplePayPaymentDetails$: Observable<
    | UserActions.LoadUserPaymentMethods
    | CheckoutActions.CreatePaymentDetailsSuccess
    | BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess
    | CheckoutActions.CreatePaymentDetailsFail> = this.actions$.pipe(
    ofType(BraintreePaymentDetailsActions.CREATE_APPLE_PAY_PAYMENT_DETAILS),
    map((action: any) => action.payload),
    mergeMap((payload) => {
      // get information for creating a subscription directly with payment provider
      const applePayPaymentDetails: Observable<PaymentDetails> = this.checkoutPaymentConnector
        .saveApplePayPaymentDetails(payload.userId, payload.applePayRequest, payload.cartId, payload.deviceData,
          payload.fields);
      return this.processPaymentCreationResult(applePayPaymentDetails, payload);
    }),
    withdrawOn(this.contextChange$)
  );

  @Effect()
  processLocalPaymentMethod: Observable<
    | CheckoutActions.PlaceOrderSuccess
    | CartActions.RemoveCart
    | CheckoutActions.PlaceOrderFail> = this.actions$.pipe(
    ofType(BraintreePaymentDetailsActions.PROCESS_LOCAL_PAYMENT_METHOD),
    map((action: any) => action.payload),
    mergeMap((payload) => {
      // get information for creating a subscription directly with payment provider
  return this.checkoutPaymentConnector
    .saveLocalPaymentMethodPaymentDetails(payload.userId, payload.cartId, payload.localPaymentRequest,
      payload.deviceData, payload.fields) .pipe(mergeMap((data) => {
          return [new CartActions.RemoveCart({ cartId: payload.cartId }),
            new CheckoutActions.PlaceOrderSuccess(data)];
      }),
      catchError((error) =>
        of(new CheckoutActions.PlaceOrderFail(normalizeHttpError(error)))
      )
    );

    }),
    withdrawOn(this.contextChange$)
  );

  @Effect()
  placeOrderThroughFallback$: Observable<
    | CheckoutActions.PlaceOrderSuccess
    | CartActions.RemoveCart
    | CheckoutActions.PlaceOrderFail> = this.actions$.pipe(
      ofType(BraintreePaymentDetailsActions.PLACE_ORDER_THROUGH_FALLBACK),
    map((action: any) => action.payload),
    mergeMap((payload) => {
      return this.checkoutPaymentConnector.processLPMFallback(payload.localPaymentRequest, payload.currencyFromFallbackURL, payload.deviceData, payload.fields)
        .pipe(mergeMap((data) => {
          if (payload.cartId){
            return [new CartActions.RemoveCart({ cartId: payload.cartId }),
                new CheckoutActions.PlaceOrderSuccess(data)];
          } else {
            return [new CheckoutActions.PlaceOrderSuccess(data)];
          }
        }),
      catchError((error) =>
            of(new CheckoutActions.PlaceOrderFail(normalizeHttpError(error)))
          )
        );

    }),
    withdrawOn(this.contextChange$)
  );

  @Effect()
  sendDeliveryModeWasDeletedEvent: Observable<
      | CheckoutActions.ClearCheckoutDeliveryModeSuccess
      | BraintreeCheckoutActions.SentDeliveryModeWasDeletedSuccess> = this.actions$.pipe(
      ofType(CheckoutActions.CLEAR_CHECKOUT_DELIVERY_MODE_SUCCESS),
      map((action) => {
        this.eventService.dispatch(new DeliveryModeWasDeletedEvent());
        return new BraintreeCheckoutActions.SentDeliveryModeWasDeletedSuccess();
      })
  );

  @Effect()
  addDeliveryAddress$: Observable<
      | UserActions.LoadUserAddresses
      | BraintreeCheckoutActions.BraintreeSetDeliveryAddress
      | CheckoutActions.AddDeliveryAddressFail
      > = this.actions$.pipe(
      ofType(BraintreeCheckoutActions.BRAINTREE_ADD_DELIVERY_ADDRESS),
      map((action: CheckoutActions.AddDeliveryAddress) => action.payload),
      mergeMap((payload) =>
          this.checkoutDeliveryConnector
          .createAddress(payload.userId, payload.cartId, payload.address)
          .pipe(
              mergeMap((address) => {
                address.titleCode = payload.address.titleCode;
                if (payload.address.region && payload.address.region.isocodeShort) {
                  Object.assign(address.region, {
                    isocodeShort: payload.address.region.isocodeShort,
                  });
                }
                if (payload.userId === OCC_USER_ID_ANONYMOUS) {
                  return [
                    new BraintreeCheckoutActions.BraintreeSetDeliveryAddress({
                      userId: payload.userId,
                      cartId: payload.cartId,
                      address,
                    }),
                  ];
                } else {
                  return [
                    new UserActions.LoadUserAddresses(payload.userId),
                    new BraintreeCheckoutActions.BraintreeSetDeliveryAddress({
                      userId: payload.userId,
                      cartId: payload.cartId,
                      address,
                    }),
                  ];
                }
              }),
              catchError((error) =>
                  of(
                      new CheckoutActions.AddDeliveryAddressFail(
                          normalizeHttpError(error)
                      )
                  )
              )
          )
      ),
      withdrawOn(this.contextChange$)
  );

  @Effect()
  setDeliveryAddress$: Observable<
      | CheckoutActions.SetDeliveryAddressSuccess
      | CheckoutActions.ClearSupportedDeliveryModes
      | CheckoutActions.ClearCheckoutDeliveryMode
      | CheckoutActions.ResetLoadSupportedDeliveryModesProcess
      | BraintreeCheckoutActions.BraintreeLoadSupportedDeliveryModes
      | CheckoutActions.SetDeliveryAddressFail
      > = this.actions$.pipe(
      ofType(BraintreeCheckoutActions.BRAINTREE_SET_DELIVERY_ADDRESS),
      map((action: any) => action.payload),
      mergeMap((payload) => {
        return this.checkoutDeliveryConnector
        .setAddress(payload.userId, payload.cartId, payload.address.id)
        .pipe(
            mergeMap(() => [
              new CheckoutActions.SetDeliveryAddressSuccess(payload.address),
              new CheckoutActions.ClearSupportedDeliveryModes(),
              new CheckoutActions.ResetLoadSupportedDeliveryModesProcess(),
              new BraintreeCheckoutActions.BraintreeLoadSupportedDeliveryModes({
                userId: payload.userId,
                cartId: payload.cartId,
              }),
            ]),
            catchError((error) =>
                of(
                    new CheckoutActions.SetDeliveryAddressFail(
                        normalizeHttpError(error)
                    )
                )
            )
        );
      }),
      withdrawOn(this.contextChange$)
  );

  @Effect()
  loadSupportedDeliveryModes$: Observable<
      | CheckoutActions.LoadSupportedDeliveryModesSuccess
      | CheckoutActions.LoadSupportedDeliveryModesFail
      | CheckoutActions.ClearCheckoutDeliveryMode
      > = this.actions$.pipe(
      ofType(BraintreeCheckoutActions.BRAINTREE_LOAD_SUPPORTED_DELIVERY_MODES),
      map((action: any) => action.payload),
      mergeMap((payload) => {
        return this.checkoutDeliveryConnector
        .getSupportedModes(payload.userId, payload.cartId)
        .pipe(
            mergeMap((data) =>  [
                new CheckoutActions.LoadSupportedDeliveryModesSuccess(data),
                new CheckoutActions.ClearCheckoutDeliveryMode({
                  userId: payload.userId,
                  cartId: payload.cartId,
                }),
                ]
            ),
            catchError((error) =>
                of(
                    new CheckoutActions.LoadSupportedDeliveryModesFail(
                        normalizeHttpError(error)
                    )
                )
            )
        );
      }),
      withdrawOn(this.contextChange$)
  );

  public processPaymentCreationResult(savingResult: Observable<PaymentDetails>, payload: any): Observable<
    | UserActions.LoadUserPaymentMethods
    | CheckoutActions.CreatePaymentDetailsSuccess
    | BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess
    | CheckoutActions.CreatePaymentDetailsFail> {

    return savingResult.pipe(mergeMap((details) => {
        if (payload.userId === OCC_USER_ID_ANONYMOUS) {
          return [new BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess(details),
          new CheckoutActions.CreatePaymentDetailsSuccess(details)];
        } else {
          return [
            new UserActions.LoadUserPaymentMethods(payload.userId),
            new BraintreePaymentDetailsActions.BraintreeCreatePaymentDetailsSuccess(details),
              new CheckoutActions.CreatePaymentDetailsSuccess(details)
          ];
        }
      }),
      catchError((error) =>
        of(
          new CheckoutActions.CreatePaymentDetailsFail(
            normalizeHttpError(error)
          )
        )
      )
    );

  }

  constructor(
    private actions$: Actions,
    private checkoutPaymentConnector: BraintreePaymentDetailsConnector,
    private checkoutDeliveryConnector: CheckoutDeliveryConnector,
    private eventService: EventService
  ) {
  }

}
