package com.braintree.controllers;

import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.facade.backoffice.BraintreeBackofficeAuthorizeFacade;
import com.braintree.facade.backoffice.BraintreeBackofficeMultiCaptureFacade;
import com.braintree.facade.backoffice.BraintreeBackofficePartialRefundFacade;
import com.braintree.facade.backoffice.BraintreeBackofficeVoidFacade;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.hybris.data.ws.BrainTreePaymentTransactionEntryWsDTO;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.transaction.service.BrainTreePaymentTransactionService;
import de.hybris.platform.commerceservices.customer.CustomerAccountService;
import de.hybris.platform.commercewebservicescommons.errors.exceptions.PaymentAuthorizationException;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.exceptions.UnknownIdentifierException;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiFieldsParam;
import de.hybris.platform.ycommercewebservices.strategies.OrderCodeIdentificationStrategy;
import de.hybris.platform.ycommercewebservices.v2.controller.BaseCommerceController;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.Authorization;
import org.springframework.security.access.annotation.Secured;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static com.braintree.controllers.BraintreeoccaddonControllerConstants.AUTHORIZATION_TRANSACTION_TYPE;
import static com.braintree.controllers.BraintreeoccaddonControllerConstants.CAPTURE_TRANSACTION_TYPE;
import static com.braintree.controllers.BraintreeoccaddonControllerConstants.PARTIAL_CAPTURE_TRANSACTION_TYPE;
import static com.braintree.controllers.BraintreeoccaddonControllerConstants.REFUND_PARTIAL_TRANSACTION_TYPE;
import static com.braintree.controllers.BraintreeoccaddonControllerConstants.REFUND_TRANSACTION_TYPE;

@RestController
@RequestMapping(value = "/{baseSiteId}/orders/{code}/braintree")
@Api(tags = "Braintree Transactions")
public class BraintreeTransactionsController extends BaseCommerceController {

    private static final String ORDER_NOT_FOUND_FOR_USER_AND_BASE_STORE = "Order with code %s not found for current user in current BaseStore";

    @Resource(name = "orderCodeIdentificationStrategy")
    private OrderCodeIdentificationStrategy orderCodeIdentificationStrategy;
    @Resource(name = "customerAccountService")
    private CustomerAccountService customerAccountService;
    @Resource(name = "baseStoreService")
    private BaseStoreService baseStoreService;
    @Resource(name = "paymentTransactionEntryConverter")
    private Converter<PaymentTransactionEntryModel, BrainTreePaymentTransactionEntryWsDTO> paymentTransactionEntryConverter;
    @Resource(name = "brainTreeCheckoutFacade")
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;
    @Resource(name = "brainTreePaymentTransactionService")
    private BrainTreePaymentTransactionService brainTreePaymentTransactionService;
    @Resource(name = "braintreeBackofficeOrderFacade")
    private BraintreeBackofficeVoidFacade braintreeBackofficeOrderFacade;

    @Resource(name = "braintreeBackofficePartialRefundFacade")
    private BraintreeBackofficePartialRefundFacade braintreeBackofficePartialRefundFacade;
    @Resource(name = "braintreeBackofficeAuthorizeFacade")
    private BraintreeBackofficeAuthorizeFacade braintreeBackofficeAuthorizeFacade;
    @Resource
    private BraintreeBackofficeMultiCaptureFacade braintreeBackofficeMultiCaptureFacade;

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT"})
    @PostMapping(value = "/authorize")
    @ApiOperation(nickname = "authorization", value = "Authorize an order", notes = "Returns details of an order authorization details.", authorizations = {
        @Authorization(value = "oauth2_client_credentials")})
    @ApiBaseSiteIdParam
    public BrainTreePaymentTransactionEntryWsDTO authorizeOrder(
        @ApiParam(value = "Order GUID (Globally Unique Identifier) or order CODE", required = true) @PathVariable final String code,
        @ApiParam(value = "Authorization amount", required = true) @RequestParam final BigDecimal authorizationAmount,
        @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields)
        throws PaymentAuthorizationException {
        final OrderModel order = getOrderForeCode(code);
        validateAmount(authorizationAmount);
        BrainTreePaymentInfoModel paymentInfoModel = (BrainTreePaymentInfoModel) order.getPaymentInfo();
        PaymentTransactionEntryData paymentTransactionEntryData = braintreeBackofficeAuthorizeFacade
            .authorizePayment(order, paymentInfoModel.getCustomFields(), authorizationAmount);

        return getDataMapper().map(paymentTransactionEntryData, BrainTreePaymentTransactionEntryWsDTO.class, fields);
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT"})
    @GetMapping(value = "/transactions")
    @ApiOperation(nickname = "getTransactions", value = "Get transactions.", notes = "Returns details of a transactions of a specific order based on the order GUID (Globally Unique Identifier) or the order CODE and transactions type.", authorizations = {
        @Authorization(value = "oauth2_client_credentials")})
    @ApiBaseSiteIdParam
    public List<BrainTreePaymentTransactionEntryWsDTO> getOrderTransactions(
        @ApiParam(value = "Order GUID (Globally Unique Identifier) or order CODE", required = true) @PathVariable final String code,
        @ApiParam(value = "Type of the transactions") @RequestParam(required = false) final String transactionType) {

        final OrderModel orderForeCode = getOrderForeCode(code);
        return getTransactionEntriesForOrder(orderForeCode, transactionType);
    }

    private List<BrainTreePaymentTransactionEntryWsDTO> getTransactionEntriesForOrder(final OrderModel order,
        String transactionType) {
        if (AUTHORIZATION_TRANSACTION_TYPE.equalsIgnoreCase(transactionType)) {
            return getPaymentTransactionsForType(
                braintreeBackofficeMultiCaptureFacade.getAuthorizeTransactionTypePredicate(), order);
        } else if (CAPTURE_TRANSACTION_TYPE.equalsIgnoreCase(transactionType)
            || PARTIAL_CAPTURE_TRANSACTION_TYPE.equalsIgnoreCase(transactionType)) {
            return getPaymentTransactionsForType(
                braintreeBackofficeMultiCaptureFacade.getCaptureTransactionTypePredicate(), order);
        } else if (REFUND_TRANSACTION_TYPE.equalsIgnoreCase(transactionType)
            || REFUND_PARTIAL_TRANSACTION_TYPE.equalsIgnoreCase(transactionType)) {
            return getPaymentTransactionsForType(
                braintreeBackofficeMultiCaptureFacade.getRefundTransactionTypePredicate(), order);
        } else {
            return getAllPaymentTransactions(order);
        }
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT"})
    @PostMapping(value = "/capture")
    @ApiOperation(nickname = "transactionCapture", value = "Capture a transaction.", notes = "Returns details of a specific transaction capturing result", authorizations = {
        @Authorization(value = "oauth2_client_credentials")})
    @ApiBaseSiteIdParam
    public BrainTreePaymentTransactionEntryWsDTO captureOrder(
        @ApiParam(value = "Order GUID (Globally Unique Identifier) or order CODE", required = true) @PathVariable final String code,
        @ApiParam(value = "Capture amount", required = true) @RequestParam final BigDecimal captureAmount,
        @ApiParam(value = "Id of authorization transaction for capturing") @RequestParam final String transactionId,
        @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields)
        throws PaymentAuthorizationException, BraintreeErrorException {

        final OrderModel orderForeCode = getOrderForeCode(code);
        validateAmount(captureAmount);
        if (braintreeBackofficeMultiCaptureFacade.isPartialCapturePossible(orderForeCode)
            && brainTreeCheckoutFacade.isPayPalCheckout(orderForeCode)
            && !isCaptureAmountEqualsTotal(captureAmount, orderForeCode)) {
            return getDataMapper().map(partialCapture(orderForeCode, captureAmount, transactionId),
                BrainTreePaymentTransactionEntryWsDTO.class, fields);
        } else {
            return getDataMapper().map(fullCapture(orderForeCode, captureAmount, transactionId),
                BrainTreePaymentTransactionEntryWsDTO.class, fields);
        }
    }

    private PaymentTransactionEntryData partialCapture(final OrderModel order, final BigDecimal amount,
        final String transactionId) throws BraintreeErrorException {
        if (brainTreePaymentTransactionService.isValidTransactionId(order, transactionId)) {
            return braintreeBackofficeMultiCaptureFacade.partialCapture(order, amount, transactionId);
        }
        throw new BraintreeErrorException(
            String.format("Error during order capture process, transactionId - %s is not valid", transactionId));
    }

    private PaymentTransactionEntryData fullCapture(final OrderModel order, final BigDecimal amount,
        final String transactionId) throws BraintreeErrorException {
        if (order != null && brainTreePaymentTransactionService.isValidTransactionId(order, transactionId)) {
            if (braintreeBackofficeMultiCaptureFacade.isSubmitForSettlementAvailable(order)) {
                return braintreeBackofficeMultiCaptureFacade.submitForSettlement(order, amount, transactionId);
            }
            if (brainTreePaymentTransactionService.canPerformDelayedCapture(order, amount)) {
                return brainTreeCheckoutFacade.authorizePayment(order, amount);
            }
        }
        throw new BraintreeErrorException(
            String.format("Error during order capture process, transactionId - %s is not valid", transactionId));
    }

    private boolean isCaptureAmountEqualsTotal(final BigDecimal amount, final OrderModel order) {
        return amount.doubleValue() >= order.getTotalPrice();
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT"})
    @PostMapping(value = "/refund")
    @ApiOperation(nickname = "transactionPartialRefund", value = "Partial refund of the order transactions.", notes = "Returns details of a specific transaction refunding result", authorizations = {
        @Authorization(value = "oauth2_client_credentials")})
    @ApiBaseSiteIdParam
    public BrainTreePaymentTransactionEntryWsDTO partialRefund(
        @ApiParam(value = "Order GUID (Globally Unique Identifier) or order CODE", required = true) @PathVariable final String code,
        @ApiParam(value = "Refund amount", required = true) @RequestParam final BigDecimal refundAmount,
        @ApiParam(value = "Id of transaction to refund", required = true) @RequestParam final String transactionId,
        @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields)
        throws PaymentAuthorizationException, BraintreeErrorException {
        final OrderModel orderForeCode = getOrderForeCode(code);
        validateAmount(refundAmount);

        PaymentTransactionEntryModel capturedTransactionById = getTransactionById(orderForeCode, transactionId);
        PaymentTransactionEntryData brainTreeRefundTransactionResult = braintreeBackofficePartialRefundFacade
            .partialRefundTransaction(orderForeCode,
                capturedTransactionById, refundAmount);

        return getDataMapper()
            .map(brainTreeRefundTransactionResult, BrainTreePaymentTransactionEntryWsDTO.class, fields);
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT"})
    @PostMapping(value = "/void")
    @ApiOperation(nickname = "orderVoid", value = "Canceling of the order transactions.", notes = "Returns details of a specific order transaction void result", authorizations = {
        @Authorization(value = "oauth2_client_credentials")})
    @ApiBaseSiteIdParam
    public BrainTreePaymentTransactionEntryWsDTO orderVoid(
        @ApiParam(value = "Order GUID (Globally Unique Identifier) or order CODE", required = true) @PathVariable final String code,
        @ApiParam(value = "Voidable transaction id", required = true) @RequestParam final String transactionId)
        throws BraintreeErrorException {
        final OrderModel orderForeCode = getOrderForeCode(code);

        PaymentTransactionEntryModel transactionById = getTransactionForVoid(orderForeCode, transactionId);

        braintreeBackofficeOrderFacade.executeVoid(transactionById);
        return paymentTransactionEntryConverter.convert(transactionById);
    }

    private PaymentTransactionEntryModel getTransactionForVoid(final OrderModel orderModel,
        final String transactionId) {
        return orderModel.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .filter(entry -> entry.getRequestId().equals(transactionId))
            .filter(entry -> (PaymentTransactionType.ORDER.equals(entry.getType())
                || PaymentTransactionType.AUTHORIZATION.equals(entry.getType()))
                && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus()))
            .findFirst().orElseThrow();
    }

    private PaymentTransactionEntryModel getTransactionById(final OrderModel orderModel, final String transactionId) {
        return orderModel.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
            .filter(entry -> entry.getRequestId().equals(transactionId))
            .filter(braintreeBackofficeMultiCaptureFacade.getCaptureTransactionTypePredicate())
            .findFirst().orElseThrow();
    }

    private List<BrainTreePaymentTransactionEntryWsDTO> getPaymentTransactionsForType(
        final Predicate<PaymentTransactionEntryModel> transactionsPredicate,
        final OrderModel order) {
        return order.getPaymentTransactions().stream()
            .flatMap(paymentTransaction -> paymentTransaction.getEntries().stream())
            .filter(transactionsPredicate)
            .map(transactionEntry -> paymentTransactionEntryConverter.convert(transactionEntry))
            .collect(Collectors.toList());
    }

    private List<BrainTreePaymentTransactionEntryWsDTO> getAllPaymentTransactions(final OrderModel order) {
        return order.getPaymentTransactions().stream()
            .flatMap(paymentTransaction -> paymentTransaction.getEntries().stream())
            .map(transactionEntry -> paymentTransactionEntryConverter.convert(transactionEntry))
            .collect(Collectors.toList());
    }

    private OrderModel getOrderForeCode(final String code) {
        final OrderModel orderModel;
        final BaseStoreModel baseStoreModel = baseStoreService.getCurrentBaseStore();
        if (orderCodeIdentificationStrategy.isID(code)) {
            orderModel = customerAccountService.getGuestOrderForGUID(code, baseStoreModel);
        } else {
            orderModel = customerAccountService.getOrderForCode(code, baseStoreModel);
        }
        if (orderModel == null) {
            throw new UnknownIdentifierException(String.format(ORDER_NOT_FOUND_FOR_USER_AND_BASE_STORE, code));
        }
        return orderModel;
    }

    private void validateAmount(final BigDecimal amount) throws PaymentAuthorizationException {
        if (amount == null || (BigDecimal.ZERO.compareTo(amount) >= 0)) {
            throw new PaymentAuthorizationException();
        }
    }

}
