/**
 *
 */
package braintreehac.services;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import de.hybris.platform.braintreehac.data.ApplePayDropInButtonStyleData;
import de.hybris.platform.braintreehac.data.DropInButtonStyleData;
import de.hybris.platform.braintreehac.data.GooglePayDropInButtonStyleData;
import de.hybris.platform.braintreehac.data.PayPalDropInButtonStyleData;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import javax.annotation.PostConstruct;
import org.apache.log4j.Logger;
import org.springframework.stereotype.Component;


@Component
public class BrainTreeDropInConfigurationService {

    private static final Logger LOG = Logger.getLogger(BrainTreeDropInConfigurationService.class);

    private static final String DROP_IN_BUTTON_CONFIG_FILE =
        System.getProperty("HYBRIS_CONFIG_DIR") + "/dropInButtonConfig.properties";

    public static final String CART = "Cart";
    public static final String MINI_CART = "Mini cart";
    public static final String BILLING = "Billing";
    public static final String PAYPAL = "payPal";
    public static final String GOOGLE_PAY = "googlePay";
    public static final String APPLE_PAY = "applePay";

    private static final String DROP_IN_PAYPAL_CART = "dropIn.payPal.cart";
    private static final String DROP_IN_PAYPAL_MINI_CART = "dropIn.payPal.miniCart";
    private static final String DROP_IN_PAYPAL_BILLING = "dropIn.payPal.billing";
    private static final String DROP_IN_GOOGLE_PAY_CART = "dropIn.googlePay.cart";
    private static final String DROP_IN_GOOGLE_PAY_MINI_CART = "dropIn.googlePay.miniCart";
    private static final String DROP_IN_GOOGLE_PAY_BILLING = "dropIn.googlePay.billing";
    private static final String DROP_IN_APPLE_PAY_CART = "dropIn.applePay.cart";
    private static final String DROP_IN_APPLE_PAY_MINI_CART = "dropIn.applePay.miniCart";
    private static final String DROP_IN_APPLE_PAY_BILLING = "dropIn.applePay.billing";

    private Properties prop;
    private File propFile;
    private ObjectMapper mapper;

    private List<String> allPlacingDropIns = new ArrayList<>();
    private List<String> allPayPalDropInColor = new ArrayList<>();
    private List<String> allPayPalDropInShape = new ArrayList<>();
    private List<String> allPayPalDropInLabel = new ArrayList<>();
    private List<String> allPayPalDropInLayout = new ArrayList<>();
    private List<String> allGooglePayDropInColor = new ArrayList<>();
    private List<String> allGooglePayDropInType = new ArrayList<>();
    private List<String> allGooglePayDropInSizeMode = new ArrayList<>();
    private List<String> allApplePayDropInButtonStyle = new ArrayList<>();

    /**
     * Used to initialization default value
     */
    @PostConstruct
    public void init() {
        prop = new Properties();
        propFile = new File(DROP_IN_BUTTON_CONFIG_FILE);
        mapper = new ObjectMapper();
        mapper.configure(JsonGenerator.Feature.QUOTE_FIELD_NAMES, false);
        mapper.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);

        if (!propFile.isFile()) {
            LOG.info("Created new " + DROP_IN_BUTTON_CONFIG_FILE);
            setDefaultButtonConfigs(DROP_IN_PAYPAL_CART);
            setDefaultButtonConfigs(DROP_IN_PAYPAL_MINI_CART);
            setDefaultButtonConfigs(DROP_IN_PAYPAL_BILLING);
            setDefaultButtonConfigs(DROP_IN_GOOGLE_PAY_CART);
            setDefaultButtonConfigs(DROP_IN_GOOGLE_PAY_MINI_CART);
            setDefaultButtonConfigs(DROP_IN_GOOGLE_PAY_BILLING);
            setDefaultButtonConfigs(DROP_IN_APPLE_PAY_CART);
            setDefaultButtonConfigs(DROP_IN_APPLE_PAY_MINI_CART);
            setDefaultButtonConfigs(DROP_IN_APPLE_PAY_BILLING);
            storeInPropertiesFile();
        }
        LOG.info("Loaded " + DROP_IN_BUTTON_CONFIG_FILE);
        loadPropertiesFromFile();

        Collections.addAll(allPlacingDropIns, CART, MINI_CART, BILLING);
        Collections.addAll(allPayPalDropInColor, "gold", "blue", "silver", "white", "black");
        Collections.addAll(allPayPalDropInShape, "rect", "pill");
        Collections.addAll(allPayPalDropInLabel, "paypal", "checkout", "buynow", "pay");
        Collections.addAll(allPayPalDropInLayout, "vertical", "horizontal");
        Collections.addAll(allGooglePayDropInColor, "default", "black", "white");
        Collections.addAll(allGooglePayDropInType, "book", "buy", "checkout", "donate", "order", "pay",
            "plain", "subscribe");
        Collections.addAll(allGooglePayDropInSizeMode, "fill", "static");
        Collections.addAll(allApplePayDropInButtonStyle, "black", "white", "white-outline");
    }

    /**
     * Used for set property into file
     * @param key
     * @param value
     */
    public void setProperty(String key, String value) {
        prop.setProperty(key, value);
        storeInPropertiesFile();
    }

    public void setDefaultButtonConfigs(final String key) {
        PayPalDropInButtonStyleData miniCartPayPal =
            createPayPalDropInButtonStyleData("gold", "rect", "paypal", "vertical", false);
        PayPalDropInButtonStyleData cartPayPalDrop =
            createPayPalDropInButtonStyleData("gold", "rect", "paypal", "vertical", false);
        PayPalDropInButtonStyleData billingPayPal =
            createPayPalDropInButtonStyleData("gold", "rect", "paypal", "vertical", false);
        GooglePayDropInButtonStyleData miniCartGooglePay =
            createGooglePayDropInButtonStyleData("default", "book", "fill");
        GooglePayDropInButtonStyleData cartGooglePay =
            createGooglePayDropInButtonStyleData("default", "book", "static");
        GooglePayDropInButtonStyleData billingGooglePay =
            createGooglePayDropInButtonStyleData("default", "book", "static");
        ApplePayDropInButtonStyleData miniCartApplePay = createApplePayDropInButtonStyleData("black");
        ApplePayDropInButtonStyleData cartApplePay = createApplePayDropInButtonStyleData("black");
        ApplePayDropInButtonStyleData billingApplePay = createApplePayDropInButtonStyleData("black");

        String miniCartPayPalDropIn = parsePayPalDropInButtonStyleDataIntoJsonString(miniCartPayPal);
        String cartPayPalDropIn = parsePayPalDropInButtonStyleDataIntoJsonString(cartPayPalDrop);
        String bullingPayPalDropIn = parsePayPalDropInButtonStyleDataIntoJsonString(billingPayPal);
        String miniCartGooglePayDropIn = parseGooglePayDropInButtonStyleDataIntoJsonString(miniCartGooglePay);
        String cartGooglePayDropIn = parseGooglePayDropInButtonStyleDataIntoJsonString(cartGooglePay);
        String billingGooglePayDropIn = parseGooglePayDropInButtonStyleDataIntoJsonString(billingGooglePay);
        String miniCartApplePayDropIn = parseApplePayDropInButtonStyleDataIntoJsonString(miniCartApplePay);
        String cartApplePayDropIn = parseApplePayDropInButtonStyleDataIntoJsonString(cartApplePay);
        String billingApplePayDropIn = parseApplePayDropInButtonStyleDataIntoJsonString(billingApplePay);

        switch (key) {
            case DROP_IN_PAYPAL_MINI_CART:
                setProperty(key, miniCartPayPalDropIn);
                break;
            case DROP_IN_PAYPAL_CART:
                setProperty(key, cartPayPalDropIn);
                break;
            case DROP_IN_PAYPAL_BILLING:
                setProperty(key, bullingPayPalDropIn);
                break;
            case DROP_IN_GOOGLE_PAY_MINI_CART:
                setProperty(key, miniCartGooglePayDropIn);
                break;
            case DROP_IN_GOOGLE_PAY_CART:
                setProperty(key, cartGooglePayDropIn);
                break;
            case DROP_IN_GOOGLE_PAY_BILLING:
                setProperty(key, billingGooglePayDropIn);
                break;
            case DROP_IN_APPLE_PAY_MINI_CART:
                setProperty(key, miniCartApplePayDropIn);
                break;
            case DROP_IN_APPLE_PAY_CART:
                setProperty(key, cartApplePayDropIn);
                break;
            case DROP_IN_APPLE_PAY_BILLING:
                setProperty(key, billingApplePayDropIn);
                break;
            default: break;
        }
    }

    /**
     * Used to create style for ApplePay button into DropIn
     * @param buttonStyle
     * @return ApplePayDropInButtonStyleData
     */
    public ApplePayDropInButtonStyleData createApplePayDropInButtonStyleData(String buttonStyle) {
        ApplePayDropInButtonStyleData buttonStyleData = new ApplePayDropInButtonStyleData();
        buttonStyleData.setButtonStyle(buttonStyle);
        return buttonStyleData;
    }

    /**
     * Used to create style for PayPal button into DropIn
     * @param color
     * @param shape
     * @param label
     * @param layout
     * @param tagline
     * @return PayPalDropInButtonStyleData
     */
    public PayPalDropInButtonStyleData createPayPalDropInButtonStyleData(String color, String shape, String label,
        String layout, boolean tagline) {
        PayPalDropInButtonStyleData buttonStyleData = new PayPalDropInButtonStyleData();
        buttonStyleData.setColor(color);
        buttonStyleData.setShape(shape);
        buttonStyleData.setLabel(label);
        buttonStyleData.setLayout(layout);
        buttonStyleData.setTagline(tagline);
        return buttonStyleData;
    }

    /**
     * Used to create style for GooglePay button into DropIn
     * @param color
     * @param type
     * @param sizeMode
     * @return GooglePayDropInButtonStyleData
     */
    public GooglePayDropInButtonStyleData createGooglePayDropInButtonStyleData(String color, String type,
        String sizeMode) {
        GooglePayDropInButtonStyleData buttonStyleData = new GooglePayDropInButtonStyleData();
        buttonStyleData.setButtonColor(color);
        buttonStyleData.setButtonType(type);
        buttonStyleData.setButtonSizeMode(sizeMode);
        return buttonStyleData;
    }

    /**
     * Used for parsing json data into PayPalDropInButtonStyleData object
     * @param paymentSystem
     * @param placing
     * @return PayPalDropInButtonStyleData
     */
    public PayPalDropInButtonStyleData parsingJsonStringIntoPayPalDropInButtonStyleData(String paymentSystem,
        String placing) {
        String keyDropIn = parseStringIntoKey(paymentSystem, placing);
        loadPropertiesFromFile();
        PayPalDropInButtonStyleData payPalDropIn = null;
        try {
            payPalDropIn = mapper.readValue(prop.getProperty(keyDropIn), PayPalDropInButtonStyleData.class);

        } catch (JsonProcessingException e) {
            payPalDropIn = new PayPalDropInButtonStyleData();
            LOG.error("Error during parsing JSON date into PayPalDropInButtonStyleData\n" + e.getMessage(), e);
            e.printStackTrace();
        }
        return payPalDropIn;
    }

    /**
     * Used for parsing json data into GooglePayDropInButtonStyleData object
     * @param paymentSystem
     * @param placing
     * @return GooglePayDropInButtonStyleData
     */
    public GooglePayDropInButtonStyleData parsingJsonStringIntoGooglePayDropInButtonStyleData(String paymentSystem,
        String placing) {
        String keyDropIn = parseStringIntoKey(paymentSystem, placing);
        loadPropertiesFromFile();
        GooglePayDropInButtonStyleData googlePayDropIn = null;
        try {
            googlePayDropIn = mapper.readValue(prop.getProperty(keyDropIn), GooglePayDropInButtonStyleData.class);

        } catch (JsonProcessingException e) {
            googlePayDropIn = new GooglePayDropInButtonStyleData();
            LOG.error("Error during parsing JSON date into GooglePayDropInButtonStyleData\n" + e.getMessage(), e);
            e.printStackTrace();
        }
        return googlePayDropIn;
    }

    /**
     * Used for parsing json data into ApplePayDropInButtonStyleData object
     * @param paymentSystem
     * @param placing
     * @return ApplePayDropInButtonStyleData
     */
    public ApplePayDropInButtonStyleData parsingJsonStringIntoApplePayDropInButtonStyleData(String paymentSystem,
        String placing) {
        String keyDropIn = parseStringIntoKey(paymentSystem, placing);
        loadPropertiesFromFile();
        ApplePayDropInButtonStyleData applePayDropIn = null;
        try {
            applePayDropIn = mapper.readValue(prop.getProperty(keyDropIn), ApplePayDropInButtonStyleData.class);

        } catch (JsonProcessingException e) {
            applePayDropIn = new ApplePayDropInButtonStyleData();
            LOG.error("Error during parsing JSON date into ApplePayDropInButtonStyleData\n" + e.getMessage(), e);
            e.printStackTrace();
        }
        return applePayDropIn;
    }

    /**
     * Used to update style for PayPal button into DropIn
     * @param paymentSystem
     * @param placing
     * @param payPalDropIn
     */
    public void updateStylePayPalDropInButtonStyleData(String paymentSystem, String placing,
        PayPalDropInButtonStyleData payPalDropIn) {
        String firstPartKey = parseStringIntoKey(paymentSystem, placing);
        String payPalDropInButtonStyleDataJson = parsePayPalDropInButtonStyleDataIntoJsonString(payPalDropIn);
        setProperty(firstPartKey, payPalDropInButtonStyleDataJson);
    }

    /**
     * Used to update style for GooglePay button into DropIn
     * @param paymentSystem
     * @param placing
     * @param googlePayDropIn
     */
    public void updateStyleGooglePayDropInButtonStyleData(String paymentSystem, String placing,
        GooglePayDropInButtonStyleData googlePayDropIn) {
        String firstPartKey = parseStringIntoKey(paymentSystem, placing);
        String googlePayDropInButtonStyleDataJson = parseGooglePayDropInButtonStyleDataIntoJsonString(googlePayDropIn);
        setProperty(firstPartKey, googlePayDropInButtonStyleDataJson);
    }

    /**
     * Used to update style for ApplePay button into DropIn
     * @param paymentSystem
     * @param placing
     * @param applePayDropIn
     */
    public void updateStyleApplePayDropInButtonStyleData(String paymentSystem, String placing,
        ApplePayDropInButtonStyleData applePayDropIn) {
        String firstPartKey = parseStringIntoKey(paymentSystem, placing);
        String applePayDropInButtonStyleDataJson = parseApplePayDropInButtonStyleDataIntoJsonString(applePayDropIn);
        setProperty(firstPartKey, applePayDropInButtonStyleDataJson);
    }

    /**
     * Used to parse String into keys for file
     * @param paymentSystem
     * @param placing
     * @return String
     */
    public String parseStringIntoKey(String paymentSystem, String placing) {
        String firstPartKey = null;
        if (paymentSystem.equals(PAYPAL)) {
            firstPartKey = determiningTypePageForPayPal(placing);
        } else if (paymentSystem.equals(GOOGLE_PAY)) {
            firstPartKey = determiningTypePageForGooglePay(placing);
        } else if (paymentSystem.equals(APPLE_PAY)) {
            firstPartKey = determiningTypePageForApplePay(placing);
        }
        return firstPartKey;
    }

    private String determiningTypePageForApplePay(String placing) {
        String firstPartKey = null;
        if (placing.equals(CART)) {
            firstPartKey = DROP_IN_APPLE_PAY_CART;
        } else if (placing.equals(MINI_CART)) {
            firstPartKey = DROP_IN_APPLE_PAY_MINI_CART;
        } else if (placing.equals(BILLING)) {
            firstPartKey = DROP_IN_APPLE_PAY_BILLING;
        }
        return firstPartKey;
    }

    private String determiningTypePageForGooglePay(String placing) {
        String firstPartKey = null;
        if (placing.equals(CART)) {
            firstPartKey = DROP_IN_GOOGLE_PAY_CART;
        } else if (placing.equals(MINI_CART)) {
            firstPartKey = DROP_IN_GOOGLE_PAY_MINI_CART;
        } else if (placing.equals(BILLING)) {
            firstPartKey = DROP_IN_GOOGLE_PAY_BILLING;
        }
        return firstPartKey;
    }

    private String determiningTypePageForPayPal(String placing) {
        String firstPartKey = null;
        if (placing.equals(CART)) {
            firstPartKey = DROP_IN_PAYPAL_CART;
        } else if (placing.equals(MINI_CART)) {
            firstPartKey = DROP_IN_PAYPAL_MINI_CART;
        } else if (placing.equals(BILLING)) {
            firstPartKey = DROP_IN_PAYPAL_BILLING;
        }
        return firstPartKey;
    }

    /**
     * Used for parsing PayPalDropInButtonStyleData object into json data
     * @param buttonStyleData
     * @return String
     */
    public String parsePayPalDropInButtonStyleDataIntoJsonString(PayPalDropInButtonStyleData buttonStyleData) {
        String buttonStyleConfig = "";
        try {
            buttonStyleConfig = mapper.writeValueAsString(buttonStyleData);
        } catch (JsonProcessingException e) {
            LOG.error("Error during parsing PayPalButtonConfigData into JSON\n" + e.getMessage(), e);
            e.printStackTrace();
        }
        return buttonStyleConfig;
    }

    /**
     * Used for parsing GooglePayDropInButtonStyleData object into json data
     * @param buttonStyleData
     * @return String
     */
    public String parseGooglePayDropInButtonStyleDataIntoJsonString(GooglePayDropInButtonStyleData buttonStyleData) {
        String buttonStyleConfig = "";
        try {
            buttonStyleConfig = mapper.writeValueAsString(buttonStyleData);
        } catch (JsonProcessingException e) {
            LOG.error("Error during parsing GooglePayDropInButtonStyleData into JSON\n" + e.getMessage(), e);
            e.printStackTrace();
        }
        return buttonStyleConfig;
    }

    /**
     * Used for parsing ApplePayDropInButtonStyleData object into json data
     * @param buttonStyleData
     * @return
     */
    public String parseApplePayDropInButtonStyleDataIntoJsonString(ApplePayDropInButtonStyleData buttonStyleData) {
        String buttonStyleConfig = "";
        try {
            buttonStyleConfig = mapper.writeValueAsString(buttonStyleData);
        } catch (JsonProcessingException e) {
            LOG.error("Error during parsing ApplePayDropInButtonStyleData into JSON\n" + e.getMessage(), e);
            e.printStackTrace();
        }
        return buttonStyleConfig;
    }

    /**
     * Used to get all styles for DropIn button
     * @param placing
     * @return DropInButtonStyleData
     */
    public DropInButtonStyleData getDropInButtonStyleData(String placing) {
        DropInButtonStyleData dropInButtonStyle = new DropInButtonStyleData();
        dropInButtonStyle.setPayPalDropIn(parsingJsonStringIntoPayPalDropInButtonStyleData(PAYPAL, placing));
        dropInButtonStyle.setGooglePayDropIn(parsingJsonStringIntoGooglePayDropInButtonStyleData(GOOGLE_PAY, placing));
        dropInButtonStyle.setApplePayDropIn(parsingJsonStringIntoApplePayDropInButtonStyleData(APPLE_PAY, placing));
        return dropInButtonStyle;
    }

    /**
     * Used to store in properties file
     */
    public void storeInPropertiesFile() {
        try (OutputStream out = new FileOutputStream(propFile)) {
            prop.store(out, "Save properties in file");
        } catch (Exception e) {
            LOG.error(DROP_IN_BUTTON_CONFIG_FILE + " is not saved,  error: " + e.getMessage(), e);
        }
    }

    private void loadPropertiesFromFile() {
        try (InputStream input = new FileInputStream(propFile)) {
            prop.load(input);
        } catch (Exception e) {
            LOG.error("Error during reading " + DROP_IN_BUTTON_CONFIG_FILE + ": " + e.getMessage(), e);
        }
    }

    public List<String> getAllPlacingDropIn() {
        return allPlacingDropIns;
    }

    public List<String> getAllPayPalDropInColor() {
        return allPayPalDropInColor;
    }

    public List<String> getAllPayPalDropInShape() {
        return allPayPalDropInShape;
    }

    public List<String> getAllPayPalDropInLabel() {
        return allPayPalDropInLabel;
    }

    public List<String> getAllPayPalDropInLayout() {
        return allPayPalDropInLayout;
    }

    public List<String> getAllGooglePayDropInColor() {
        return allGooglePayDropInColor;
    }

    public List<String> getAllGooglePayDropInType() {
        return allGooglePayDropInType;
    }

    public List<String> getAllGooglePayDropInSizeMode() {
        return allGooglePayDropInSizeMode;
    }

    public List<String> getAllApplePayDropInButtonStyle() {
        return allApplePayDropInButtonStyle;
    }
}
