/**
 *
 */
package com.braintree.facade.impl;

import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.command.result.BrainTreeCreatePaymentMethodResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.hybris.data.BrainTreePaymentInfoData;
import com.braintree.hybris.data.BrainTreeSubscriptionInfoData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.payment.dto.BraintreeInfo;
import com.braintree.payment.local.methods.service.BraintreeLocalPaymentMethodsService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.braintreegateway.WebhookNotification;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commerceservices.customer.CustomerEmailResolutionService;
import de.hybris.platform.commerceservices.strategies.CheckoutCustomerStrategy;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import javax.servlet.http.HttpServletRequest;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.verification.VerificationMode;


@UnitTest
public class DefaultBrainTreePaymentFacadeTest {

    private static final String ORDER_CODE = "123456";

    private static final String PAYMENT_ID = "testId";

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private CheckoutCustomerStrategy checkoutCustomerStrategy;

    @Mock
    private CartService cartService;

    @Mock
    private BaseStoreService baseStoreService;

    @Mock
    private BrainTreeCustomerAccountService customerAccountService;

    @Mock
    private BraintreeLocalPaymentMethodsService braintreeLocalPaymentMethodsService;

    @Mock
    private Converter<BrainTreePaymentInfoModel, BrainTreePaymentInfoData> brainTreePaymentInfoDataConverter;

    @Mock
    private Converter<AddressData, AddressModel> addressReverseConverter;

    @Mock
    private Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> brainTreeSubscriptionInfoConverter;

    @Mock
    private Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter;

    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private ModelService modelService;

    @Mock
    private CustomerEmailResolutionService customerEmailResolutionService;

    @InjectMocks
    private DefaultBrainTreePaymentFacade unit;

    private BrainTreeSubscriptionInfoData subscriptionInfo;
    private CustomerModel customerModel;
    private CartModel cartModel;
    private AddressModel addressModel;
    private AddressData addressData;
    private BrainTreePaymentInfoModel paymentInfoModel;
    private BraintreeInfo braintreeInfo;
    private BrainTreeCreatePaymentMethodResult result;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldGetBrainTreePaymentInfoData() {
        final CartModel cartModel1 = mock(CartModel.class);
        final BrainTreePaymentInfoModel paymentInfoModel1 = mock(BrainTreePaymentInfoModel.class);
        final BrainTreePaymentInfoData brainTreePaymentInfoDataExpected = mock(BrainTreePaymentInfoData.class);

        when(cartService.getSessionCart()).thenReturn(cartModel1);
        when(cartModel1.getPaymentInfo()).thenReturn(paymentInfoModel1);
        when(brainTreePaymentInfoDataConverter.convert(paymentInfoModel1)).thenReturn(brainTreePaymentInfoDataExpected);
        unit.setBrainTreePaymentInfoDataConverter(brainTreePaymentInfoDataConverter);

        final BrainTreePaymentInfoData brainTreePaymentInfoDataActual = unit
            .getBrainTreePaymentInfoData();

        assertEquals(brainTreePaymentInfoDataExpected, brainTreePaymentInfoDataActual);

    }

    @Test
    public void shouldGetBrainTreePaymentInfoDataByOrderCode() {
        final BrainTreePaymentInfoData brainTreePaymentInfoDataExpected = mock(BrainTreePaymentInfoData.class);
        final BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);
        final OrderModel order = mock(OrderModel.class);
        final BaseStoreModel baseStoreModel = mock(BaseStoreModel.class);

        when(baseStoreService.getCurrentBaseStore()).thenReturn(baseStoreModel);
        when(customerAccountService.getOrderForCode(ORDER_CODE, baseStoreModel)).thenReturn(order);
        when(order.getPaymentInfo()).thenReturn(paymentInfo);
        when(brainTreePaymentInfoDataConverter.convert(paymentInfo)).thenReturn(brainTreePaymentInfoDataExpected);
        unit.setBrainTreePaymentInfoDataConverter(brainTreePaymentInfoDataConverter);

        final BrainTreePaymentInfoData brainTreePaymentInfoDataActual = unit
            .getBrainTreePaymentInfoData(ORDER_CODE);

        assertEquals(brainTreePaymentInfoDataExpected, brainTreePaymentInfoDataActual);
    }

    @Test
    public void shouldGetWebhookNotification() {
        WebhookNotification webhookNotification = mock(WebhookNotification.class);
        when(brainTreePaymentService.getWebhookNotification(any())).thenReturn(webhookNotification);

        unit.getWebhookNotification(mock(HttpServletRequest.class));

        verify(brainTreePaymentService).getWebhookNotification(any());
    }

    @Test
    public void shouldGetOrderByPaymentId() {
        OrderModel orderModel = mock(OrderModel.class);
        when(braintreeLocalPaymentMethodsService.getOrderByPaymentId(PAYMENT_ID)).thenReturn(orderModel);

        unit.getOrderByPaymentId(PAYMENT_ID);

        verify(braintreeLocalPaymentMethodsService).getOrderByPaymentId(PAYMENT_ID);
    }

    private void defaultPreparingBeforeCompleteCreateSubscription() {
        subscriptionInfo = mock(BrainTreeSubscriptionInfoData.class);
        customerModel = mock(CustomerModel.class);
        cartModel = mock(CartModel.class);
        addressModel = mock(AddressModel.class);
        when(modelService.create(AddressModel.class)).thenReturn(addressModel);
        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        when(cartService.getSessionCart()).thenReturn(cartModel);

        addressData = mock(AddressData.class);
        when(subscriptionInfo.getAddressData()).thenReturn(addressData);

        paymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        when(brainTreeTransactionService.createSubscription(any(), any(), any(), any())).thenReturn(paymentInfoModel);

        braintreeInfo = mock(BraintreeInfo.class);
        when(brainTreeSubscriptionInfoConverter.convert(subscriptionInfo)).thenReturn(braintreeInfo);

        result = mock(BrainTreeCreatePaymentMethodResult.class);
        when(result.isSuccess()).thenReturn(true);
        when(brainTreePaymentService.createPaymentMethodForCustomer(customerModel, addressModel, braintreeInfo))
            .thenReturn(result);


    }

    @Test
    public void completeCreateSubscriptionShouldCreatePaymentMethodIfOrderIntentAndCreditEnabledAndChangePaymentButtonEnabled() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.getCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.getChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeConfigService.getStoreInVault()).thenReturn(Boolean.TRUE.toString());
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BraintreeConstants.PAYPAL_PAYMENT);
        when(subscriptionInfo.getIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_ORDER);
        when(brainTreeConfigService.getVaultingForCurrentUser(BraintreeConstants.PAYPAL_PAYMENT)).thenReturn(false);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreePaymentService).createPaymentMethodForCustomer(customerModel, addressModel, braintreeInfo);
    }

    @Test
    public void completeCreateSubscriptionShouldNotCreatePaymentMethodIfOrderSaleAndCreditEnabledAndChangePaymentButtonEnabled() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.getCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.getChangePaymentMethodButtonEnabled()).thenReturn(false);
        when(brainTreeConfigService.getStoreInVault()).thenReturn(Boolean.TRUE.toString());
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BraintreeConstants.PAYPAL_PAYMENT);
        when(subscriptionInfo.getIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_SALE);
        when(brainTreeConfigService.getVaultingForCurrentUser(BraintreeConstants.PAYPAL_PAYMENT)).thenReturn(false);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreePaymentService, never()).createPaymentMethodForCustomer(customerModel, addressModel, braintreeInfo);
    }

    @Test
    public void completeCreateSubscriptionShouldCreatePaymentMethodIfOrderSaleAndCreditNotEnabledAndChangePaymentButtonNotEnabled() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.getCreditEnabled()).thenReturn(false);
        when(brainTreeConfigService.getChangePaymentMethodButtonEnabled()).thenReturn(false);
        when(brainTreeConfigService.getStoreInVault()).thenReturn(Boolean.TRUE.toString());
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BraintreeConstants.PAYPAL_PAYMENT);
        when(subscriptionInfo.getIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_SALE);
        when(brainTreeConfigService.getVaultingForCurrentUser(BraintreeConstants.PAYPAL_PAYMENT)).thenReturn(true);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreePaymentService).createPaymentMethodForCustomer(customerModel, addressModel, braintreeInfo);
    }

    @Test
    public void completeCreateSubscriptionShouldCreatePaymentMethodForGooglePayWhenVaultTrue() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.getCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.getChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeConfigService.getStoreInVault()).thenReturn(Boolean.TRUE.toString());
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BraintreeConstants.ANDROID_PAY_CARD);
        when(subscriptionInfo.getIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_ORDER);
        when(brainTreeConfigService.getVaultingForCurrentUser(BraintreeConstants.ANDROID_PAY_CARD)).thenReturn(true);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreePaymentService).createPaymentMethodForCustomer(customerModel, addressModel, braintreeInfo);
    }

    @Test
    public void completeCreateSubscriptionShouldCreatePaymentMethodForVenmoWhenVaultFalse() {
        defaultPreparingBeforeCompleteCreateSubscription();
        when(brainTreeConfigService.getCreditEnabled()).thenReturn(true);
        when(brainTreeConfigService.getChangePaymentMethodButtonEnabled()).thenReturn(false);
        when(brainTreeConfigService.getStoreInVault()).thenReturn(Boolean.FALSE.toString());
        when(subscriptionInfo.getPaymentProvider()).thenReturn(BraintreeConstants.VENMO_CHECKOUT);
        when(subscriptionInfo.getIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_ORDER);
        when(brainTreeConfigService.getVaultingForCurrentUser(BraintreeConstants.VENMO_CHECKOUT)).thenReturn(false);

        unit.completeCreateSubscription(subscriptionInfo);

        verify(brainTreePaymentService).createPaymentMethodForCustomer(customerModel, addressModel, braintreeInfo);
    }

}
