/**
 *
 */
package com.braintree.facade.impl;

import static com.braintree.constants.BraintreeConstants.PAYPAL_INTENT_ORDER;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.command.result.BrainTreeFindCustomerResult;
import com.braintree.command.result.BrainTreePaymentDetailsResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants;
import com.braintree.customfield.service.BraintreeCustomFieldsService;
import com.braintree.enums.BraintreePageType;
import com.braintree.facade.BrainTreeUserFacade;
import com.braintree.hybris.data.BraintreeLocalPaymentMethodData;
import com.braintree.hybris.data.PayPalCheckoutData;
import com.braintree.hybris.data.PayPalConfigurationData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.model.BraintreeLocalPaymentMethodsModel;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorservices.uiexperience.UiExperienceService;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.catalog.CatalogVersionService;
import de.hybris.platform.catalog.model.CatalogVersionModel;
import de.hybris.platform.commercefacades.order.CartFacade;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commerceservices.customer.CustomerAccountService;
import de.hybris.platform.commerceservices.order.CommerceCheckoutService;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.commerceservices.service.data.CommerceOrderResult;
import de.hybris.platform.commerceservices.strategies.CheckoutCustomerStrategy;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.media.MediaModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.media.MediaService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import java.util.ArrayList;
import java.util.Currency;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

@UnitTest
public class DefaultBrainTreeCheckoutFacadeTest {

    @Mock
    private MediaService mediaService;

    @Mock
    private CheckoutCustomerStrategy checkoutCustomerStrategy;

    @Mock
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;

    @Mock
    private CommerceCheckoutService commerceCheckoutService;

    @Mock
    private Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter;

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private CatalogVersionService catalogVersionService;

    @Mock
    private CustomerAccountService customerAccountService;

    @Mock
    private CartService cartService;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private ModelService modelService;

    @Mock
    private I18NService i18NService;

    @Mock
    private Converter<OrderModel, OrderData> orderConverter;

    @Mock
    private CartFacade cartFacade;

    @Mock
    private UserService userService;

    @Mock
    private BrainTreeUserFacade brainTreeUserFacade;

    @Mock
    private Converter<BraintreeLocalPaymentMethodsModel, BraintreeLocalPaymentMethodData> brainTreeLocalPaymentsConverter;

    @Mock
    private UiExperienceService uiExperienceService;

    @Mock
    private BraintreeCustomFieldsService customFieldsService;

    @Spy
    @InjectMocks
    private DefaultBrainTreeCheckoutFacade unit;

    private CartModel sessionCardModel;

    private BrainTreePaymentInfoModel sessionPaymentInfoModel;

    private CCPaymentInfoData checkoutPaymentInfoModel;

    private Map<String, String> customFields;

    private static final String INTENT = "intent";
    private static final String SHIPS_FROM_POSTAL_CODE = "ships";
    private static final String CUSTOMER_FIELD_KEY = "customerFieldKey";
    private static final String CUSTOMER_FIELD_VALUE = "customFieldValue";
    private static final String SECURITY_CODE = "securityCode";
    private static final String PAYMENT_INFO_ID = "paymentInfoId";
    private static final String PAYMENT_METHOD_NONCE = "paymentMethodNonce";
    private static final String USD = "USD";
    private static final String GOOGLE_IMG_CODE = "googlePayMark";
    private static final String CURRENCY_ISO_CODE = "currencyIsoCode";
    private static final String SITE_UID = "siteUID";
    private static final String MERCHANT_ACCOUNT_ID = "MerchantAccountId";
    private static final String BRAINREE_CUSTOMER_ID = "braintreeCustomerId";
    private static final Double DOUBLE_TEN = 10d;
    private static final Double DOUBLE_ONE = 1d;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        unit.setBrainTreePaymentInfoConverter(brainTreePaymentInfoConverter);
        customFields = new HashMap<>();
        customFields.put(CUSTOMER_FIELD_KEY, CUSTOMER_FIELD_VALUE);
        sessionCardModel = Mockito.mock(CartModel.class);
        sessionPaymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);

        when(sessionCardModel.getPaymentInfo()).thenReturn(sessionPaymentInfoModel);
        when(cartService.getSessionCart()).thenReturn(sessionCardModel);
        when(brainTreeConfigService.getIntent()).thenReturn(INTENT);
        when(i18NService.getCurrentJavaCurrency()).thenReturn(Currency.getInstance(USD));
        when(i18NService.getCurrentLocale()).thenReturn(Locale.US);

        CartData checkoutCartData = Mockito.mock(CartData.class);
        checkoutPaymentInfoModel = Mockito.mock(CCPaymentInfoData.class);

        when(checkoutCartData.getPaymentInfo()).thenReturn(checkoutPaymentInfoModel);
        when(cartFacade.getSessionCart()).thenReturn(checkoutCartData);
        when(cartFacade.hasSessionCart()).thenReturn(true);
        when(customFieldsService.getDefaultCustomFieldsMap()).thenReturn(customFields);
    }

    @Test
    public void storeIntentToCartShouldSetIntentAndSavePaymentInfoModel() {
        unit.storeIntentToCart();
        verify(sessionPaymentInfoModel).setPayPalIntent(INTENT);
        verify(modelService).save(sessionPaymentInfoModel);
    }

    @Test
    public void storeShipsFromPostalCodeToCartShouldSetShipsAndSavePaymentInfoModel() {
        unit.storeShipsFromPostalCodeToCart(SHIPS_FROM_POSTAL_CODE);
        verify(sessionPaymentInfoModel).setShipsFromPostalCode(SHIPS_FROM_POSTAL_CODE);
        verify(modelService).save(sessionPaymentInfoModel);
    }

    @Test
    public void storeCustomFieldsToCartShouldSetCustomFieldsAndSavePaymentInfoModel() {
        unit.storeCustomFieldsToCart(customFields);
        verify(sessionPaymentInfoModel).setCustomFields(customFields);
        verify(modelService).save(sessionPaymentInfoModel);
    }

    @Test
    public void authorizePaymentShouldReturnTrueIfPayPalPaymentMethodOrderIntent() {
        when(brainTreeConfigService.getIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_ORDER);
        when(checkoutPaymentInfoModel.getSubscriptionId()).thenReturn(BraintreeConstants.PAYPAL_PAYMENT);
        Assert.assertTrue(unit.authorizePayment(SECURITY_CODE));
    }

    @Test
    public void authorizePaymentShouldCallCreatePaymentMethodIfIntentOrderIfPaymentAuthorizationNotAvailable() {
        when(brainTreeConfigService.getIntent()).thenReturn(PAYPAL_INTENT_ORDER);

        unit.authorizePayment(SECURITY_CODE);
        verify(brainTreePaymentFacade).completeCreateSubscription(any());
        modelService.save(sessionPaymentInfoModel);
    }

    @Test
    public void setPaymentDetailsShouldReturnFalseIfCurrentUserIsNotTheCartUser() {
        UserModel userModel = Mockito.mock(UserModel.class);
        when(sessionCardModel.getUser()).thenReturn(userModel);

        Assert.assertFalse(unit.setPaymentDetails(PAYMENT_INFO_ID, PAYMENT_METHOD_NONCE));
    }

    @Test
    public void setPaymentDetailsShouldSetNonceForPaymentInfoAndSavePaymentInfo() {
        CustomerModel customerModel = Mockito.mock(CustomerModel.class);
        when(sessionCardModel.getUser()).thenReturn(customerModel);
        when(checkoutCustomerStrategy.getCurrentUserForCheckout()).thenReturn(customerModel);
        BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);
        when(brainTreePaymentService.completeCreateSubscription(customerModel, PAYMENT_INFO_ID))
            .thenReturn(paymentInfoModel);

        Assert.assertTrue(unit.setPaymentDetails(PAYMENT_INFO_ID, PAYMENT_METHOD_NONCE));
        verify(paymentInfoModel).setNonce(PAYMENT_METHOD_NONCE);
        verify(modelService).save(paymentInfoModel);
    }

    @Test
    public void getPayPalCheckoutDataShouldReturnNotNull() {
        BraintreePageType pageType = BraintreePageType.MARK;
        Currency currency = Currency.getInstance(USD);
        Locale locale = Locale.US;
        CustomerModel user = Mockito.mock(CustomerModel.class);
        List<AddressModel> addressModels = new ArrayList<>();
        when(user.getAddresses()).thenReturn(addressModels);
        when(userService.getCurrentUser()).thenReturn(user);

        when(i18NService.getCurrentJavaCurrency()).thenReturn(currency);
        when(i18NService.getCurrentLocale()).thenReturn(locale);
        Assert.assertNotNull(unit.getPayPalCheckoutData(pageType));
    }

    @Test
    public void checkNotGenerateConfigurationDataWithBrainTreeCustomerId() {

        CustomerModel userModel = Mockito.mock(CustomerModel.class);
        Currency currency = Currency.getInstance(USD);
        Locale locale = Locale.US;
        List<AddressModel> savedAddresses = new ArrayList<>();
        List<BrainTreePaymentDetailsResult> savedPaymentMethods = new ArrayList<>();
        BrainTreeFindCustomerResult customerResult = Mockito.mock(BrainTreeFindCustomerResult.class);

        when(i18NService.getCurrentJavaCurrency()).thenReturn(currency);
        when(i18NService.getCurrentLocale()).thenReturn(locale);

        when(userService.getCurrentUser()).thenReturn(userModel);
        when(userModel.getAddresses()).thenReturn(savedAddresses);

        when(userModel.getBraintreeCustomerId()).thenReturn(BRAINREE_CUSTOMER_ID);
        when(brainTreeUserFacade.findBrainTreeCustomer(BRAINREE_CUSTOMER_ID)).thenReturn(customerResult);
        when(customerResult.getPaymentMethods()).thenReturn(savedPaymentMethods);

        when(brainTreeConfigService.getDropInEnabled()).thenReturn(false);

        unit.generateConfigurationData(BraintreePageType.ACCOUNT);

        verify(unit).generateClientToken(false);

    }

    @Test
    public void checkGenerateConfigurationDataWithBrainTreeCustomerId() {

        CustomerModel userModel = Mockito.mock(CustomerModel.class);
        Currency currency = Currency.getInstance(USD);
        Locale locale = Locale.US;
        List<AddressModel> savedAddresses = new ArrayList<>();
        List<BrainTreePaymentDetailsResult> savedPaymentMethods = new ArrayList<>();
        BrainTreeFindCustomerResult customerResult = Mockito.mock(BrainTreeFindCustomerResult.class);

        when(i18NService.getCurrentJavaCurrency()).thenReturn(currency);
        when(i18NService.getCurrentLocale()).thenReturn(locale);

        when(userService.getCurrentUser()).thenReturn(userModel);
        when(userModel.getAddresses()).thenReturn(savedAddresses);

        when(userModel.getBraintreeCustomerId()).thenReturn(BRAINREE_CUSTOMER_ID);
        when(brainTreeUserFacade.findBrainTreeCustomer(BRAINREE_CUSTOMER_ID)).thenReturn(customerResult);
        when(customerResult.getPaymentMethods()).thenReturn(savedPaymentMethods);

        when(brainTreeConfigService.getDropInEnabled()).thenReturn(true);

        unit.generateConfigurationData(BraintreePageType.ACCOUNT);

        verify(unit).generateClientToken(true);

    }

    @Test
    public void getBraintreeAssetsShouldCallGetDownloadURLForEnabledPaymentMethods() {
        when(brainTreeConfigService.getGooglePayEnabled()).thenReturn(true);
        when(brainTreeConfigService.getVenmoEnabled()).thenReturn(true);
        when(brainTreeConfigService.getApplePayEnabled()).thenReturn(true);

        CatalogVersionModel catalogVersionModel = Mockito.mock(CatalogVersionModel.class);
        List<CatalogVersionModel> catalogVersionModels = new ArrayList<>();
        catalogVersionModels.add(catalogVersionModel);
        MediaModel mediaModel = Mockito.mock(MediaModel.class);

        when(catalogVersionService.getSessionCatalogVersions()).thenReturn(catalogVersionModels);
        when(mediaService.getMedia(catalogVersionModel, GOOGLE_IMG_CODE)).thenReturn(mediaModel);

        unit.getBraintreeAssets();
        verify(mediaModel).getDownloadURL();
    }

    @Test
    public void handleOrderIntentViaSubscriptionShouldSetRequiredFieldsAndSavePaymentInfoModel() {
        OrderModel orderModel = mock(OrderModel.class);
        BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);
        BaseSiteModel siteModel = Mockito.mock(BaseSiteModel.class);
        CurrencyModel currencyModel = Mockito.mock(CurrencyModel.class);
        Map<String, String> customFieldsMap = new HashMap<>();
        when(customFieldsService.getDefaultCustomFieldsMap()).thenReturn(customFieldsMap);
        when(brainTreeConfigService
            .getMerchantAccountIdByCurrentSiteNameAndCurrency(SITE_UID, CURRENCY_ISO_CODE.toLowerCase())).thenReturn(MERCHANT_ACCOUNT_ID);

        when(siteModel.getUid()).thenReturn(SITE_UID);
        when(currencyModel.getIsocode()).thenReturn(CURRENCY_ISO_CODE);
        when(orderModel.getCurrency()).thenReturn(currencyModel);
        when(orderModel.getSite()).thenReturn(siteModel);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        unit.handleOrderIntentViaSubscription(orderModel);
        verify(paymentInfoModel).setPayPalIntent(BraintreeConstants.PAYPAL_INTENT_ORDER);
        verify(paymentInfoModel).setSaved(false);
        verify(paymentInfoModel).setCustomFields(customFieldsMap);
        verify(paymentInfoModel).setMerchantAccountIdForCurrentSite(MERCHANT_ACCOUNT_ID);
        verify(modelService).save(paymentInfoModel);
    }

    @Test
    public void placeOrderByCartShouldUseOrderConverterForPlacedOrderModel() throws InvalidCartException {
        CartModel cartModel = Mockito.mock(CartModel.class);
        AddressModel deliveryAddress = Mockito.mock(AddressModel.class);
        CommerceOrderResult orderResult = Mockito.mock(CommerceOrderResult.class);
        OrderModel orderModel = Mockito.mock(OrderModel.class);

        when(orderResult.getOrder()).thenReturn(orderModel);
        when(commerceCheckoutService.placeOrder(any(CommerceCheckoutParameter.class))).thenReturn(orderResult);
        when(cartModel.getDeliveryAddress()).thenReturn(deliveryAddress);
        unit.placeOrderByCart(cartModel);
        verify(orderConverter).convert(orderModel);
    }

    @Test
    public void generateConfigurationDataForMyAccountIsCorrect() {
        when(cartService.hasSessionCart()).thenReturn(false);
        when(brainTreeConfigService.getIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_SALE);
        when(brainTreeConfigService.getChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeUserFacade.isAnonymousUser()).thenReturn(false);
        when(brainTreeUserFacade.isCurrentUserHasBrainTreeCustomerId()).thenReturn(true);

        CustomerModel user = Mockito.mock(CustomerModel.class);
        List<AddressModel> addressModels = new ArrayList<>();
        addressModels.add(Mockito.mock(AddressModel.class));
        when(user.getAddresses()).thenReturn(addressModels);
        when(userService.getCurrentUser()).thenReturn(user);

        PayPalCheckoutData payPalCheckoutData = unit.generateConfigurationData(BraintreePageType.ACCOUNT);
        PayPalConfigurationData configurationData = payPalCheckoutData.getConfigurationData();
        verify(brainTreePaymentService).generateClientToken(StringUtils.EMPTY);
        Assert.assertEquals(BraintreeConstants.PAYPAL_INTENT_SALE, configurationData.getIntent());
        Assert.assertEquals(Currency.getInstance(USD).getCurrencyCode(), configurationData.getCurrency());
        Assert.assertEquals(Locale.US.toString(), configurationData.getLocale());
        Assert.assertFalse(payPalCheckoutData.getEnableShippingAddress());
        Assert.assertFalse(payPalCheckoutData.getPayPalPaymentMethod().getShouldRenderPayPalChangePaymentButton());
    }

    @Test
    public void generateConfigurationDataForMarkPageShouldRenderChangePaymentButton() {
        when(cartService.hasSessionCart()).thenReturn(true);
        when(brainTreeConfigService.getChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeUserFacade.isAnonymousUser()).thenReturn(false);
        when(brainTreeUserFacade.isCurrentUserHasBrainTreeCustomerId()).thenReturn(true);

        CartModel cartModel = Mockito.mock(CartModel.class);
        when(cartModel.getTotalPrice()).thenReturn(DOUBLE_TEN);
        when(cartModel.getSubtotal()).thenReturn(DOUBLE_ONE);
        when(cartService.getSessionCart()).thenReturn(cartModel);

        CustomerModel user = Mockito.mock(CustomerModel.class);
        when(user.getBraintreeCustomerId()).thenReturn(BRAINREE_CUSTOMER_ID);
        List<AddressModel> addressModels = new ArrayList<>();
        when(user.getAddresses()).thenReturn(addressModels);
        when(userService.getCurrentUser()).thenReturn(user);

        PayPalCheckoutData payPalCheckoutData = unit.generateConfigurationData(BraintreePageType.MARK);
        verify(brainTreePaymentService).generateClientToken(BRAINREE_CUSTOMER_ID);
        Assert.assertTrue(payPalCheckoutData.getPayPalPaymentMethod().getShouldRenderPayPalChangePaymentButton());
        Assert.assertEquals(DOUBLE_TEN, payPalCheckoutData.getAmount());
        Assert.assertEquals(DOUBLE_ONE, payPalCheckoutData.getSubtotal());
    }

    @Test
    public void generateConfigurationDataForMarkPageShouldNotRenderChangePaymentButtonWhenAnonymousUser() {
        when(cartService.hasSessionCart()).thenReturn(false);
        when(brainTreeConfigService.getChangePaymentMethodButtonEnabled()).thenReturn(true);
        when(brainTreeUserFacade.isAnonymousUser()).thenReturn(true);
        when(brainTreeUserFacade.isCurrentUserHasBrainTreeCustomerId()).thenReturn(false);

        CustomerModel user = Mockito.mock(CustomerModel.class);
        when(user.getBraintreeCustomerId()).thenReturn(BRAINREE_CUSTOMER_ID);
        List<AddressModel> addressModels = new ArrayList<>();
        when(user.getAddresses()).thenReturn(addressModels);
        when(userService.getCurrentUser()).thenReturn(user);

        PayPalCheckoutData payPalCheckoutData = unit.generateConfigurationData(BraintreePageType.MARK);
        verify(brainTreePaymentService).generateClientToken(StringUtils.EMPTY);
        Assert.assertFalse(payPalCheckoutData.getPayPalPaymentMethod().getShouldRenderPayPalChangePaymentButton());
    }




}
