/**
 *
 */
package com.braintree.facade.backoffice.impl;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.command.result.BrainTreePaymentMethodResult;
import com.braintree.command.result.BrainTreeVoidResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants;
import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.service.BraintreeOrderBackofficeUtilService;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.braintreegateway.Transaction;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import java.util.ArrayList;
import java.util.List;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

public class DefaultBraintreeBackofficeVoidFacadeTest {

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private UserService userService;

    @Mock
    private BraintreePaymentInfoService paymentInfoService;

    @Mock
    private BraintreeOrderBackofficeUtilService backofficeUtilService;

    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;

    @Mock
    private Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> ordermanagementPaymentTransactionEntryConverter;

    @Mock
    private ModelService modelService;

    @InjectMocks
    private DefaultBraintreeBackofficeVoidFacade unit;
    private static final Double TOTAL_PRICE = 200d;
    private static final String SUCCESFULL = "SUCCESFULL";
    private static final String CUSTOMER_UID = "customerUid";
    private static final String PAYMENT_METHOD_TOKEN = "paymentMethodToken";

    private OrderModel orderModel;
    private BrainTreePaymentInfoModel paymentInfoModel;
    private PaymentTransactionEntryModel entryModel;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        orderModel = Mockito.mock(OrderModel.class);
        paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);
        when(paymentInfoModel.getPaymentMethodToken()).thenReturn(PAYMENT_METHOD_TOKEN);
        when(paymentInfoModel.getCustomerId()).thenReturn(CUSTOMER_UID);

        PaymentTransactionModel transactionModel = Mockito.mock(PaymentTransactionModel.class);
        List<PaymentTransactionModel> transactionModelList = new ArrayList<>();
        transactionModelList.add(transactionModel);

        entryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        List<PaymentTransactionEntryModel> entryModels = new ArrayList<>();
        entryModels.add(entryModel);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getTransactionStatusDetails()).thenReturn(SUCCESFULL);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        when(transactionModel.getEntries()).thenReturn(entryModels);
        when(orderModel.getPaymentTransactions()).thenReturn(transactionModelList);
        when(orderModel.getTotalPrice()).thenReturn(TOTAL_PRICE);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        UserModel userModel = Mockito.mock(UserModel.class);
        when(userModel.getUid()).thenReturn(CUSTOMER_UID);
        when(userService.getCurrentUser()).thenReturn(userModel);
        when(backofficeUtilService.isOrderVoided(orderModel)).thenReturn(false);
        when(backofficeUtilService.isOrderAmountCaptured(orderModel)).thenReturn(false);
    }

    @Test
    public void isVoidPossibleShouldReturnFalseIfOrderModelNull() {
        Assert.assertFalse(unit.isVoidPossible(null));
    }

    @Test
    public void isVoidPossibleForOrderWhickHaveIncorrectIntentShouldReturnFalse() {
        Assert.assertFalse(unit.isVoidPossible(orderModel));
    }

    @Test
    public void isVoidPossibleForCardPaymentShouldReturnTrue() {
        when(backofficeUtilService.paidByCard(orderModel)).thenReturn(true);
        Assert.assertTrue(unit.isVoidPossible(orderModel));
    }

    @Test
    public void isVoidPossibleShouldReturnTrueForVenmo() {
        when(brainTreeConfigService.getSettlementConfigParameter()).thenReturn(false);
        when(paymentInfoModel.getPaymentProvider()).thenReturn(BraintreeConstants.VENMO_CHECKOUT);

        Assert.assertTrue(unit.isVoidPossible(orderModel));
    }

    @Test
    public void isVoidPossibleShouldReturnFalseForPayPalAndIntentSale() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(BraintreeConstants.PAYPAL_PAYMENT);
        when(paymentInfoModel.getPayPalIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_SALE);

        Assert.assertFalse(unit.isVoidPossible(orderModel));
    }

    @Test
    public void isVoidPossibleShouldReturnTrueForPayPalAndIntentAuthorizeWithSuccessTransaction() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(BraintreeConstants.PAYPAL_PAYMENT);
        when(paymentInfoModel.getPayPalIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_AUTHORIZE);

        Assert.assertTrue(unit.isVoidPossible(orderModel));
    }

    @Test
    public void isVoidPossibleShouldReturnTrueForPayPalAndIntentOrderWithSuccessTransaction() {
        when(paymentInfoModel.getPaymentProvider()).thenReturn(BraintreeConstants.PAYPAL_PAYMENT);
        when(paymentInfoModel.getPayPalIntent()).thenReturn(BraintreeConstants.PAYPAL_INTENT_ORDER);

        Assert.assertTrue(unit.isVoidPossible(orderModel));
    }

    @Test
    public void getVoidableTransactions() {
        Assert.assertEquals(entryModel, unit.getVoidableTransactions(orderModel).get(0));
        unit.getVoidableTransactions(orderModel);
    }

    @Test(expected = BraintreeErrorException.class)
    public void executeVoidShouldThrowExceptionIfBraintreeResultNotSuccess() throws BraintreeErrorException {
        BrainTreeVoidResult voidResult = Mockito.mock(BrainTreeVoidResult.class);
        when(brainTreePaymentService.voidTransaction(any())).thenReturn(voidResult);
        unit.executeVoid(entryModel);
    }

    @Test
    public void executeVoidShouldThrowExceptionIfBraintreeResultSuccess() throws BraintreeErrorException {
        BrainTreeVoidResult voidResult = Mockito.mock(BrainTreeVoidResult.class);
        when(voidResult.isSuccess()).thenReturn(true);
        when(brainTreePaymentService.voidTransaction(any())).thenReturn(voidResult);
        unit.executeVoid(entryModel);
        verify(entryModel).setTransactionStatus(Transaction.Status.VOIDED.name());
        verify(modelService).save(entryModel);
        verify(ordermanagementPaymentTransactionEntryConverter).convert(entryModel);
    }

    @Test
    public void isVoidAuthorizationPossibleShouldReturnFalseIfOrderNull() {
        Assert.assertFalse(unit.isVoidAuthorizationPossible(null));
    }

    @Test
    public void isVoidAuthorizationPossibleShouldReturnTrueIfSuccessfulTransactionPresent() {
        Assert.assertTrue(unit.isVoidAuthorizationPossible(orderModel));
    }

    @Test
    public void deletePaymentMethodShouldNotDeletePaymentMethodsIfPaymentMethodResultNotSuccess() {
        BrainTreePaymentMethodResult paymentMethodResult = Mockito.mock(BrainTreePaymentMethodResult.class);
        when(brainTreePaymentService.deletePaymentMethod(any())).thenReturn(paymentMethodResult);

        unit.deletePaymentMethod(paymentInfoModel);
        verify(paymentInfoService, never()).remove(CUSTOMER_UID, PAYMENT_METHOD_TOKEN);
    }

    @Test
    public void deletePaymentMethodShouldDeletePaymentMethodsIfPaymentMethodResultSuccess() {
        BrainTreePaymentMethodResult paymentMethodResult = Mockito.mock(BrainTreePaymentMethodResult.class);
        when(paymentMethodResult.isSuccess()).thenReturn(true);
        when(brainTreePaymentService.deletePaymentMethod(any())).thenReturn(paymentMethodResult);

        unit.deletePaymentMethod(paymentInfoModel);
        verify(paymentInfoService).remove(CUSTOMER_UID, PAYMENT_METHOD_TOKEN);
    }

}
