/**
 *
 */
package com.braintree.facade.impl;

import static com.braintree.constants.BraintreeConstants.BRAINTREE_TRY_DELETE_PAYMENT_DETAILS_WITH_NOT_CAPTURED_ORDER;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT;
import static com.braintree.constants.BraintreeConstants.PAY_PAL_EXPRESS_CHECKOUT;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import com.braintree.command.request.BrainTreeAddressRequest;
import com.braintree.command.request.BrainTreeCreateCreditCardPaymentMethodRequest;
import com.braintree.command.request.BrainTreeCustomerRequest;
import com.braintree.command.request.BrainTreeDeletePaymentMethodRequest;
import com.braintree.command.request.BrainTreeUpdatePaymentMethodRequest;
import com.braintree.command.result.BrainTreeAddressResult;
import com.braintree.command.result.BrainTreeCreatePaymentMethodResult;
import com.braintree.command.result.BrainTreeFindCustomerResult;
import com.braintree.command.result.BrainTreePaymentDetailsResult;
import com.braintree.command.result.BrainTreePaymentMethodResult;
import com.braintree.command.result.BrainTreeUpdatePaymentMethodResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.exceptions.BraintreeDeletePaymentDetailsWithNotCapturedOrderException;
import com.braintree.facade.BrainTreeUserFacade;
import com.braintree.hybris.data.BrainTreeSubscriptionInfoData;
import com.braintree.hybris.data.BraintreeUserData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.service.BraintreeOrderBackofficeUtilService;
import com.braintree.payment.dto.BraintreeInfo;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.paypal.converters.impl.BraintreeBillingAddressConverter;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.braintreegateway.PaymentMethod;
import com.google.common.collect.Lists;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercefacades.user.impl.DefaultUserFacade;
import de.hybris.platform.commerceservices.enums.CustomerType;
import de.hybris.platform.commerceservices.strategies.CheckoutCustomerStrategy;
import de.hybris.platform.commerceservices.strategies.CustomerNameStrategy;
import de.hybris.platform.converters.Populator;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.request.CreateSubscriptionRequest;
import de.hybris.platform.payment.commands.result.SubscriptionResult;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.function.BiPredicate;
import java.util.regex.Pattern;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

/**
 * This class is a default implementation of BrainTreeUserFacade interface
 */
public class DefaultBrainTreeUserFacade extends DefaultUserFacade implements BrainTreeUserFacade {

    private Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter;
    private Populator<CCPaymentInfoData, BrainTreePaymentInfoModel> braintreePaymentInfoReversePopulator;
    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;
    private BrainTreePaymentService brainTreePaymentService;
    private CheckoutCustomerStrategy checkoutCustomerStrategy;
    private BrainTreeTransactionService brainTreeTransactionService;
    private CustomerNameStrategy customerNameStrategy;
    private Converter<AddressData, AddressModel> addressReverseConverter;
    private Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> brainTreeSubscriptionInfoConverter;
    private BraintreePaymentInfoService paymentInfoService;
    private Converter<PaymentMethod, BrainTreePaymentInfoModel> paymentMethodConverter;
    private BraintreeBillingAddressConverter billingAddressConverter;
    private BrainTreeConfigService brainTreeConfigService;
    private BraintreeOrderBackofficeUtilService backofficeUtilService;
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;
    private UserService userService;
    private ModelService modelService;

    @Override
    public void addAddress(final AddressData addressData) {
        validateParameterNotNullStandardMessage("addressData", addressData);

        final CustomerModel currentCustomer = getCurrentUserForCheckout();

        final boolean makeThisAddressTheDefault = addressData.isDefaultAddress()
            || (currentCustomer.getDefaultShipmentAddress() == null && addressData.isVisibleInAddressBook());

        if (isDuplicateAddress(addressData, currentCustomer)) {
            addressData.setVisibleInAddressBook(false);
        }

        // Create the new address model
        final AddressModel newAddress = getModelService().create(AddressModel.class);
        getAddressReversePopulator().populate(addressData, newAddress);

        // Store the address against the user
        getCustomerAccountService().saveAddressEntry(currentCustomer, newAddress);

        // Update the address ID in the newly created address
        addressData.setId(newAddress.getPk().toString());

        if (makeThisAddressTheDefault) {
            getCustomerAccountService().setDefaultAddressEntry(currentCustomer, newAddress);
        }
    }

    private boolean isDuplicateAddress(final AddressData addressData, final CustomerModel customerModel) {
        BiPredicate<String, String> compare = (s1, s2) -> {
            if (s1 == null && s2 == null) {
                return true;
            }
            return s1 != null && s1.equals(s2);
        };

        for (AddressModel address : customerModel.getAddresses()) {
            if (address.getVisibleInAddressBook()
                && compare.test(address.getFirstname(), addressData.getFirstName())
                && compare.test(address.getLastname(), addressData.getLastName())
                && compare.test(address.getTown(), addressData.getTown())
                && compare.test(address.getPostalcode(), addressData.getPostalCode())
                && compare.test(address.getLine1(), addressData.getLine1())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void editAddress(final AddressData addressData) {
        final String brainTreeAddressId = getBrainTreeAddressId(addressData);

        if (StringUtils.isNotEmpty(brainTreeAddressId)) {
            final BrainTreeAddressRequest addressRequest = convertBrainTreeAddress(addressData);
            addressRequest.setAddressId(brainTreeAddressId);
            brainTreePaymentService.updateAddress(addressRequest);
        }

        super.editAddress(addressData);
    }

    @Override
    public void removeAddress(final AddressData addressData) {
        final String brainTreeAddressId = getBrainTreeAddressId(addressData);

        if (StringUtils.isNotEmpty(brainTreeAddressId)) {
            final BrainTreeAddressRequest addressRequest = convertBrainTreeAddress(addressData);
            addressRequest.setAddressId(brainTreeAddressId);
            brainTreePaymentService.removeAddress(addressRequest);
        }

        super.removeAddress(addressData);
    }

    private String getBrainTreeAddressId(final AddressData addressData) {
        validateParameterNotNullStandardMessage("addressData", addressData);
        final CustomerModel currentCustomer = getCurrentUserForCheckout();
        final AddressModel addressModel = getCustomerAccountService()
            .getAddressForCode(currentCustomer, addressData.getId());
        return addressModel.getBrainTreeAddressId();
    }

    private BrainTreeAddressRequest convertBrainTreeAddress(final AddressData address) {
        final String customerID = checkoutCustomerStrategy.getCurrentUserForCheckout().getBraintreeCustomerId();
        final BrainTreeAddressRequest addressRequest = new BrainTreeAddressRequest(customerID);
        addressRequest.setCompany(address.getTitle());
        addressRequest.setStreetAddress(address.getLine1());
        addressRequest.setExtendedAddress(address.getLine2());
        addressRequest.setFirstName(address.getFirstName());
        addressRequest.setLastName(address.getLastName());
        addressRequest.setLocality(address.getTown());
        addressRequest.setPostalCode(address.getPostalCode());

        if (address.getCountry() != null) {
            addressRequest.setCountryCodeAlpha2(address.getCountry().getIsocode());
        }
        if (address.getRegion() != null) {
            //The state or province. For PayPal addresses, the region must be a 2-letter abbreviation; for all other payment methods, it must be less than or equal to 255 characters.
            //because of hybris specific use the isocodeShort - 2 character isocode - and its right for braintree
            //the isocode return  2 character isocode US-CA - wrong for braintree
            addressRequest.setRegion(address.getRegion().getIsocodeShort());
        }

        return addressRequest;
    }

    @Override
    public boolean editPaymentMethod(CCPaymentInfoData paymentInfo, final String cardholderName,
        final String expirationDate,
        final String cvv, final AddressData addressData) {
        validateParameterNotNullStandardMessage("paymentInfo", paymentInfo);
        final BrainTreeUpdatePaymentMethodRequest request = new BrainTreeUpdatePaymentMethodRequest(
            paymentInfo.getPaymentMethodToken());

        request.setToken(paymentInfo.getPaymentMethodToken());
        request.setCardholderName(cardholderName);
        request.setCardExpirationDate(expirationDate);
        request.setCvv(cvv);
        if (addressData != null) {
            request.setBillingAddressId(addressData.getBrainTreeAddressId());
        }

        BrainTreeUpdatePaymentMethodResult result = getBrainTreePaymentService().updatePaymentMethod(request);

        if (result.isSuccess()) {
            final BrainTreePaymentInfoModel braintreePaymentInfo = getPaymentMethodConverter()
                .convert(result.getPaymentMethod());
            if (braintreePaymentInfo != null) {
                getPaymentInfoService().update(braintreePaymentInfo, addressData);
            }
            return true;
        }
        throw new AdapterException(result.getErrorMessage());
    }

    @Override
    public boolean isCurrentUserHasBrainTreeCustomerId() {
        UserModel currentUser = userService.getCurrentUser();
        return !StringUtils.isEmpty(((CustomerModel) currentUser).getBraintreeCustomerId());
    }


    @Override
    public List<CCPaymentInfoData> getCCPaymentInfos(boolean saved) {

        final List<CCPaymentInfoData> ccPaymentInfos = super.getCCPaymentInfos(saved);
        final List<CCPaymentInfoData> brainTreeCCPaymentInfos = getBrainTreeCCPaymentInfos(saved);
        ccPaymentInfos.addAll(brainTreeCCPaymentInfos);
        return ccPaymentInfos;

    }

    @Override
    public List<CCPaymentInfoData> getBrainTreeCCPaymentInfos(final boolean saved) {

        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
        final Collection<BrainTreePaymentInfoModel> paymentInfos = brainTreeCustomerAccountService
            .getBrainTreePaymentInfos(
                currentCustomer, saved);

        final List<BrainTreePaymentInfoModel> creditCards = Lists.newArrayList();

        boolean isPayPalEnabled = getBrainTreeConfigService().getPayPalStandardEnabled();
        boolean isHostedFieldsEnabled = getBrainTreeConfigService().getHostedFieldEnabled();
        boolean isVenmoEnabled = getBrainTreeConfigService().getVenmoEnabled();
        boolean isGooglePayEnabled = getBrainTreeConfigService().getGooglePayEnabled();
        boolean isSrcEnabled = getBrainTreeConfigService().getSrcEnabled();
        boolean isAccountPaymentEnabled = isPayPalEnabled && isVenmoEnabled && isGooglePayEnabled;
        if (isHostedFieldsEnabled && isAccountPaymentEnabled && isSrcEnabled) {
            addAllPaymentInfos(paymentInfos, creditCards);
        } else {
            for (final PaymentInfoModel paymentInfoModel : paymentInfos) {
                if (!(paymentInfoModel instanceof BrainTreePaymentInfoModel)) {
                    continue;
                }

                String paymentProvider = ((BrainTreePaymentInfoModel) paymentInfoModel).getPaymentProvider();
                if (isHostedFieldsEnabled && BraintreeConstants.BRAINTREE_PAYMENT.equals(paymentProvider)) {
                    creditCards.add((BrainTreePaymentInfoModel) paymentInfoModel);
                    continue;
                }
                if (isPayPalEnabled
                    && (BraintreeConstants.PAYPAL_PAYMENT.equals(paymentProvider)
                    || BraintreeConstants.PAY_PAL_EXPRESS_CHECKOUT.equals(paymentProvider))) {
                    creditCards.add((BrainTreePaymentInfoModel) paymentInfoModel);
                    continue;
                }
                if (isVenmoEnabled && BraintreeConstants.VENMO_CHECKOUT.equals(paymentProvider)) {
                    creditCards.add((BrainTreePaymentInfoModel) paymentInfoModel);
                    continue;
                }
                if (isGooglePayEnabled && BraintreeConstants.ANDROID_PAY_CARD.equals(paymentProvider)) {
                    creditCards.add((BrainTreePaymentInfoModel) paymentInfoModel);
                    continue;
                }
                if (isSrcEnabled && BraintreeConstants.SRC_CARD.equalsIgnoreCase(paymentProvider)) {
                    creditCards.add((BrainTreePaymentInfoModel) paymentInfoModel);
                }
            }
        }

        final List<CCPaymentInfoData> ccPaymentInfos = new ArrayList<CCPaymentInfoData>();
        final PaymentInfoModel defaultPaymentInfoModel = currentCustomer.getDefaultPaymentInfo();

        for (final BrainTreePaymentInfoModel ccPaymentInfoModel : creditCards) {
            final CCPaymentInfoData defaultPaymentInfoData = getBrainTreePaymentInfoConverter()
                .convert(ccPaymentInfoModel);

            if (ccPaymentInfoModel.equals(defaultPaymentInfoModel) &&
                Optional.ofNullable(defaultPaymentInfoData).isPresent()) {
                defaultPaymentInfoData.setDefaultPaymentInfo(true);
            }

            if (isPaymentMethodDeletedOnBraintree(defaultPaymentInfoData)) {
                ccPaymentInfoModel.setIsDeletedOnBraintree(true);
                modelService.save(ccPaymentInfoModel);
                continue;
            }

            ccPaymentInfos.add(defaultPaymentInfoData);
        }
        return ccPaymentInfos;
    }

    @Override
    public CCPaymentInfoData getBrainTreeCCPaymentInfoById(String paymentId) {
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
        final BrainTreePaymentInfoModel paymentInfo = brainTreeCustomerAccountService.getBrainTreePaymentInfoForCode(currentCustomer, paymentId);

        return getBrainTreePaymentInfoConverter().convert(paymentInfo);
    }

    private boolean isPaymentMethodDeletedOnBraintree(CCPaymentInfoData paymentInfoData) {
        brainTreePaymentFacade.setPaymentMethodNonce(paymentInfoData);
        return !Optional.ofNullable(paymentInfoData.getPaymentMethodNonce()).isPresent();
    }

    private void addAllPaymentInfos(Collection<BrainTreePaymentInfoModel> paymentInfos,
        List<BrainTreePaymentInfoModel> creditCards) {
        for (final PaymentInfoModel paymentInfoModel : paymentInfos) {
            if (paymentInfoModel instanceof BrainTreePaymentInfoModel) {
                creditCards.add((BrainTreePaymentInfoModel) paymentInfoModel);
            }
        }
    }

    @Override
    public CCPaymentInfoData addPaymentMethodData(final BrainTreeSubscriptionInfoData subscriptionInfoData) {
        return getBrainTreePaymentInfoConverter().convert(addPaymentMethod(subscriptionInfoData));
    }

    @Override
    public BrainTreePaymentInfoModel addPaymentMethod(final BrainTreeSubscriptionInfoData subscriptionInfoData) {
        validateParameterNotNullStandardMessage("subscriptionInfoData", subscriptionInfoData);
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
        return savePaymentMethod(subscriptionInfoData, currentCustomer);
    }

    private BrainTreePaymentInfoModel savePaymentMethod(
            final BrainTreeSubscriptionInfoData brainTreeSubscriptionInfoData,
            final CustomerModel customer) {
        createBrainTreeCustomerIfNotExist(customer);
        hasExistingPayPalAccount(customer, brainTreeSubscriptionInfoData);
        createBrainTreeAddressIfNotExist(brainTreeSubscriptionInfoData, customer);
        final BrainTreeCreateCreditCardPaymentMethodRequest request = new BrainTreeCreateCreditCardPaymentMethodRequest(
            customer.getBraintreeCustomerId());
        final AddressData addressData = brainTreeSubscriptionInfoData.getAddressData();
        if (addressData != null) {
            AddressModel billingAddress = convertAddressData(addressData, brainTreeSubscriptionInfoData);

            request.setPaymentMethodNonce(brainTreeSubscriptionInfoData.getNonce());
            request.setCustomerId(customer.getBraintreeCustomerId());
            request.setCardHolderName(brainTreeSubscriptionInfoData.getCardholder());
            request.setBillingAddressId(billingAddress.getBrainTreeAddressId());

            final BillingInfo billingInfo = billingAddressConverter.convert(billingAddress);
            request.setBillingAddress(billingInfo);

            final BrainTreePaymentMethodResult creditCardPaymentMethod = getBrainTreePaymentService()
                .createCreditCardPaymentMethod(request);

            if (creditCardPaymentMethod.isSuccess()) {
                addAdditionalPaymentMethodFields(brainTreeSubscriptionInfoData, creditCardPaymentMethod);
                final BraintreeInfo braintreeInfo = getBrainTreeSubscriptionInfoConverter()
                    .convert(brainTreeSubscriptionInfoData);

                return getBrainTreeTransactionService().createSubscription(billingAddress, customer, braintreeInfo);
            } else {
                throw new AdapterException(creditCardPaymentMethod.getErrorMessage());
            }
        }
        return null;
    }

    private void hasExistingPayPalAccount(CustomerModel customerModel,
        BrainTreeSubscriptionInfoData brainTreeSubscriptionInfoData) {
        BrainTreeFindCustomerResult result = findBrainTreeCustomer(customerModel.getBraintreeCustomerId());
        if (result.isCustomerExist() && StringUtils.isNotBlank(brainTreeSubscriptionInfoData.getEmail())) {
            boolean isMethodExistInAccount = checkForExistingPaymentMethodForCustomer(customerModel,
                brainTreeSubscriptionInfoData.getEmail());
            for (BrainTreePaymentDetailsResult account : result.getPayPalMethods()) {
                if (brainTreeSubscriptionInfoData.getEmail().equalsIgnoreCase(account.getEmail())
                    && isMethodExistInAccount) {
                    throw new AdapterException("This payment method is already exist.");
                }
            }
        }
    }

    private boolean checkForExistingPaymentMethodForCustomer(CustomerModel customer, String acctountEmail) {
        for (PaymentInfoModel paymentInfo : customer.getPaymentInfos()) {
            if (paymentInfo.isSaved()
                && paymentInfo instanceof BrainTreePaymentInfoModel
                && BraintreeConstants.PAYPAL_PAYMENT
                .equals(((BrainTreePaymentInfoModel) paymentInfo).getPaymentProvider())
                && acctountEmail.equals((paymentInfo).getBillingAddress().getEmail())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public BrainTreeFindCustomerResult findBrainTreeCustomer(String braintreeCustomerId) {
        if (StringUtils.isEmpty(braintreeCustomerId)) {
            return new BrainTreeFindCustomerResult(false);
        }
        final BrainTreeCustomerRequest findCustomerRequest = new BrainTreeCustomerRequest(braintreeCustomerId);
        findCustomerRequest.setCustomerId(braintreeCustomerId);
        final BrainTreeFindCustomerResult response = getBrainTreePaymentService().findCustomer(findCustomerRequest);
        return response;
    }

    private AddressModel convertAddressData(AddressData addressData,
        BrainTreeSubscriptionInfoData brainTreeSubscriptionInfoData) {
        AddressModel billingAddress = getModelService().create(AddressModel.class);
        Converter<AddressData, AddressModel> converter = getAddressReverseConverter();
        converter.convert(addressData, billingAddress);
        billingAddress.setBrainTreeAddressId(addressData.getBrainTreeAddressId());
        if (StringUtils.isNotBlank(brainTreeSubscriptionInfoData.getEmail())) {
            billingAddress.setEmail(brainTreeSubscriptionInfoData.getEmail());
        } else {
            billingAddress.setEmail(addressData.getEmail());
        }
        return billingAddress;

    }

    private void createBrainTreeCustomerIfNotExist(final CustomerModel currentCustomer) {
        if (StringUtils.isEmpty(currentCustomer.getBraintreeCustomerId())) {
            final BillingInfo billingInfo = new BillingInfo();
            billingInfo.setEmail(currentCustomer.getContactEmail());

            final String[] names = getCustomerNameStrategy().splitName(currentCustomer.getName());
            if (names != null) {
                billingInfo.setFirstName(names[0]);
                billingInfo.setLastName(names[1]);
            }

            final SubscriptionResult customerSubscription = getBrainTreePaymentService().createCustomerSubscription(
                new CreateSubscriptionRequest(null, billingInfo, null, null, null, null, null));

            currentCustomer.setBraintreeCustomerId(customerSubscription.getMerchantTransactionCode());
            getModelService().save(currentCustomer);
        }
    }

    private void createBrainTreeAddressIfNotExist(final BrainTreeSubscriptionInfoData subscriptionInfoData,
        final CustomerModel currentCustomer) {
        final AddressData addressData = subscriptionInfoData.getAddressData();
        if (addressData != null && StringUtils.isEmpty(addressData.getBrainTreeAddressId())) {
            final BrainTreeAddressRequest addressRequest = convertBrainTreeAddress(addressData);
            final BrainTreeAddressResult brainTreeAddress = getBrainTreePaymentService().createAddress(addressRequest,
                currentCustomer);
            if (brainTreeAddress != null && brainTreeAddress.getAddress() != null) {
                addressData.setBrainTreeAddressId(brainTreeAddress.getAddress().getId());
                updateCurrentAddress(currentCustomer, addressData, brainTreeAddress);
            }
        }
    }

    private void addAdditionalPaymentMethodFields(final BrainTreeSubscriptionInfoData brainTreeSubscriptionInfoData,
        final BrainTreeCreatePaymentMethodResult createPaymentMethodResult) {
        if (createPaymentMethodResult != null) {
            brainTreeSubscriptionInfoData.setPaymentMethodToken(createPaymentMethodResult.getPaymentMethodToken());
            brainTreeSubscriptionInfoData.setExpirationMonth(createPaymentMethodResult.getExpirationMonth());
            brainTreeSubscriptionInfoData.setExpirationYear(createPaymentMethodResult.getExpirationYear());
            brainTreeSubscriptionInfoData.setImageSource(createPaymentMethodResult.getImageSource());
            brainTreeSubscriptionInfoData.setCardNumber(createPaymentMethodResult.getCardNumber());
            brainTreeSubscriptionInfoData.setCardType(createPaymentMethodResult.getCardType());
            if (StringUtils.isNotBlank(createPaymentMethodResult.getEmail())) {
                brainTreeSubscriptionInfoData.setEmail(createPaymentMethodResult.getEmail());
            }
        }
    }

    private void updateCurrentAddress(final CustomerModel currentCustomer, final AddressData addressData,
        final BrainTreeAddressResult brainTreeAddress) {
        for (final AddressModel addressModel : currentCustomer.getAddresses()) {
            if (addressModel.getPk().toString().equals(addressData.getId())) {
                addressModel.setBrainTreeAddressId(brainTreeAddress.getAddress().getId());
                getModelService().save(addressModel);
            }
        }
        getModelService().save(currentCustomer);
    }

    @Override
    public void unlinkCCPaymentInfo(final String id) {
        validateParameterNotNullStandardMessage("id", id);
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();

        final BrainTreePaymentInfoModel brainTreePaymentInfo = brainTreeCustomerAccountService
            .getBrainTreePaymentInfoForCode(
                currentCustomer, id);

        brainTreeCustomerAccountService.unlinkCCPaymentInfo(currentCustomer, brainTreePaymentInfo);
    }

    @Override
    public void updateCCPaymentInfo(final CCPaymentInfoData paymentInfo) {
        validateParameterNotNullStandardMessage("paymentInfo", paymentInfo);
        validateParameterNotNullStandardMessage("paymentInfoID", paymentInfo.getId());
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();

        BrainTreePaymentInfoModel brainTreePaymentInfoForCode = brainTreeCustomerAccountService
            .getBrainTreePaymentInfoForCode(currentCustomer, paymentInfo.getId());
        if (brainTreePaymentInfoForCode != null) {
            braintreePaymentInfoReversePopulator.populate(paymentInfo, brainTreePaymentInfoForCode);
            modelService.save(brainTreePaymentInfoForCode);
            if (brainTreePaymentInfoForCode.getBillingAddress() != null) {
                getModelService().save(brainTreePaymentInfoForCode.getBillingAddress());
                getModelService().refresh(brainTreePaymentInfoForCode);
            }

        } else {
            super.updateCCPaymentInfo(paymentInfo);
        }

    }

    protected void updateDefaultPaymentInfo(final CustomerModel currentCustomer) {
        if (currentCustomer.getDefaultPaymentInfo() == null) {
            List<BrainTreePaymentInfoModel> brainTreePaymentInfoModelList = brainTreeCustomerAccountService
                .getBrainTreePaymentInfos(currentCustomer, true);

            if (CollectionUtils.isNotEmpty(brainTreePaymentInfoModelList)) {
                getCustomerAccountService().setDefaultPaymentInfo(currentCustomer,
                    brainTreePaymentInfoModelList.get(brainTreePaymentInfoModelList.size() - 1));
            } else {
                super.updateDefaultPaymentInfo(currentCustomer);
            }
        }
    }

    @Override
    public void removeBTCCPaymentInfo(final String paymentMethodToken) {
        validateParameterNotNullStandardMessage("paymentMethodToken", paymentMethodToken);
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
        final BrainTreeDeletePaymentMethodRequest request = new BrainTreeDeletePaymentMethodRequest(
            currentCustomer.getBraintreeCustomerId(), paymentMethodToken);
        brainTreePaymentService.deletePaymentMethod(request);
    }

    @Override
    public CCPaymentInfoData getCCPaymentInfoForCode(final String code) {
        if (StringUtils.isNotBlank(code)) {
            final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();
            final BrainTreePaymentInfoModel ccPaymentInfoModel = brainTreeCustomerAccountService
                .getBrainTreePaymentInfoForCode(
                    currentCustomer, code);
            if (ccPaymentInfoModel != null) {
                final PaymentInfoModel defaultPaymentInfoModel = currentCustomer.getDefaultPaymentInfo();
                final CCPaymentInfoData paymentInfoData = getBrainTreePaymentInfoConverter()
                    .convert(ccPaymentInfoModel);
                if (ccPaymentInfoModel.equals(defaultPaymentInfoModel)) {
                    paymentInfoData.setDefaultPaymentInfo(true);
                }
                return paymentInfoData;
            } else {
                return super.getCCPaymentInfoForCode(code);
            }
        }

        return null;
    }

    @Override
    public void setDefaultPaymentInfo(final CCPaymentInfoData paymentInfoData) {
        validateParameterNotNullStandardMessage("paymentInfoData", paymentInfoData);
        final CustomerModel currentCustomer = getCurrentUserForCheckout();

        PaymentInfoModel paymentInfoModel = brainTreeCustomerAccountService
            .getPaymentInfoForCode(currentCustomer, paymentInfoData.getId());

        if (paymentInfoModel != null) {
            getCustomerAccountService().setDefaultPaymentInfo(currentCustomer, paymentInfoModel);
        } else {
            super.setDefaultPaymentInfo(paymentInfoData);
        }
    }

    @Override
    public void removeCCPaymentInfo(final String id) {
        validateParameterNotNullStandardMessage("id", id);
        final CustomerModel currentCustomer = (CustomerModel) getUserService().getCurrentUser();

        final BrainTreePaymentInfoModel brainTreePaymentInfo = brainTreeCustomerAccountService
            .getBrainTreePaymentInfoForCode(
                currentCustomer, id);

        if (brainTreePaymentInfo != null) {
            if (getCustomerAccountService() instanceof BrainTreeCustomerAccountService) {

                boolean isAvailableDeletePaymentDetails =
                    PAYPAL_PAYMENT.equalsIgnoreCase(brainTreePaymentInfo.getPaymentProvider()) ||
                        PAY_PAL_EXPRESS_CHECKOUT.equalsIgnoreCase(brainTreePaymentInfo.getPaymentProvider())
                        || !isCustomerHaveNotCapturedOrdersForPaymentDetails(currentCustomer, brainTreePaymentInfo);

                if (!isAvailableDeletePaymentDetails) {
                    throw new BraintreeDeletePaymentDetailsWithNotCapturedOrderException(
                        BRAINTREE_TRY_DELETE_PAYMENT_DETAILS_WITH_NOT_CAPTURED_ORDER);
                }

                ((BrainTreeCustomerAccountService) getCustomerAccountService())
                    .deleteBrainTreePaymentInfo(currentCustomer, brainTreePaymentInfo);

                if (brainTreePaymentInfo.getPaymentMethodToken() != null) {
                    removeBTCCPaymentInfo(brainTreePaymentInfo.getPaymentMethodToken());
                    brainTreePaymentInfo.setIsDeletedOnBraintree(true);
                    modelService.save(brainTreePaymentInfo);
                }
            }

        } else {
            super.removeCCPaymentInfo(id);
        }

        updateDefaultPaymentInfo(currentCustomer);
    }

    private boolean isCustomerHaveNotCapturedOrdersForPaymentDetails(CustomerModel customerModel,
        BrainTreePaymentInfoModel paymentInfoModel) {
        return customerModel.getOrders().stream()
            .filter(order -> {
                PaymentInfoModel paymentInfo = order.getPaymentInfo();
                if (paymentInfo instanceof BrainTreePaymentInfoModel) {
                    String methodToken = ((BrainTreePaymentInfoModel) (paymentInfo)).getPaymentMethodToken();
                    return Optional.ofNullable(methodToken)
                        .map(s -> s.equalsIgnoreCase(paymentInfoModel.getPaymentMethodToken()))
                        .orElse(Boolean.FALSE);
                }
                return false;
            })
            .anyMatch(orderModel -> !OrderStatus.COMPLETED.equals(orderModel.getStatus())
                && !backofficeUtilService.isOrderVoided(orderModel));
    }

    /**
     * Used to check is customer has not captures for saved payment method type
     * @param paymentProvider
     * @return boolean
     */
    public boolean isCustomerHasNotCapturedOrdersForSavedPaymentMethodType(String paymentProvider) {
        final CustomerModel customerModel = (CustomerModel) getUserService().getCurrentUser();
        return customerModel.getOrders().stream()
            .filter(order -> {
                PaymentInfoModel paymentInfo = order.getPaymentInfo();
                if (paymentInfo instanceof BrainTreePaymentInfoModel) {
                    String curPaymentProvider = ((BrainTreePaymentInfoModel) (paymentInfo)).getPaymentProvider();
                    String token = ((BrainTreePaymentInfoModel) paymentInfo).getPaymentMethodToken();
                    if (token == null || StringUtils.isBlank(token)) {
                        return false;
                    }
                    return Optional.ofNullable(curPaymentProvider)
                        .map(s -> s.equalsIgnoreCase(paymentProvider))
                        .orElse(Boolean.FALSE);
                }
                return false;
            })
            .anyMatch(orderModel -> !OrderStatus.COMPLETED.equals(orderModel.getStatus())
                && !backofficeUtilService.isOrderVoided(orderModel));
    }

    public BraintreeUserData getUserData() {
        UserModel currentUser = getCartService().getSessionCart().getUser();

        BraintreeUserData userData = new BraintreeUserData();

        if (currentUser.getName() != null) {
            final String[] names = currentUser.getName().split(" ", 2);
            userData.setFirstName(names[0]);
            if (names.length > 1) {
                userData.setLastName(names[1]);
            }
        }

        if (CustomerType.GUEST.getCode().equalsIgnoreCase(currentUser.getName())) {
            String[] splatUid = currentUser.getUid().split(Pattern.quote("|"));
            userData.setEmail(splatUid[splatUid.length - 1]);
            userData.setLastName(currentUser.getName());

        } else {
            userData.setEmail(currentUser.getUid());
        }

        return userData;
    }

    /**
     * @return the checkoutCustomerStrategy
     */
    @Override
    public CheckoutCustomerStrategy getCheckoutCustomerStrategy() {
        return checkoutCustomerStrategy;
    }

    /**
     * @param checkoutCustomerStrategy
     *           the checkoutCustomerStrategy to set
     */
    @Override
    public void setCheckoutCustomerStrategy(final CheckoutCustomerStrategy checkoutCustomerStrategy) {
        this.checkoutCustomerStrategy = checkoutCustomerStrategy;
    }

    /**
     * @return the brainTreePaymentService
     */
    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    /**
     * @param brainTreePaymentService
     *           the brainTreePaymentService to set
     */
    public void setBrainTreePaymentService(final BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }

    public Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> getBrainTreePaymentInfoConverter() {
        return brainTreePaymentInfoConverter;
    }

    public void setBrainTreePaymentInfoConverter(
        final Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter) {
        this.brainTreePaymentInfoConverter = brainTreePaymentInfoConverter;
    }

    /**
     * @return the brainTreeCustomerAccountService
     */
    public BrainTreeCustomerAccountService getBrainTreeCustomerAccountService() {
        return brainTreeCustomerAccountService;
    }

    /**
     * @param brainTreeCustomerAccountService
     *           the brainTreeCustomerAccountService to set
     */
    public void setBrainTreeCustomerAccountService(
        final BrainTreeCustomerAccountService brainTreeCustomerAccountService) {
        this.brainTreeCustomerAccountService = brainTreeCustomerAccountService;
    }

    /**
     * @return the customerNameStrategy
     */
    public CustomerNameStrategy getCustomerNameStrategy() {
        return customerNameStrategy;
    }

    /**
     * @param customerNameStrategy
     *           the customerNameStrategy to set
     */
    public void setCustomerNameStrategy(final CustomerNameStrategy customerNameStrategy) {
        this.customerNameStrategy = customerNameStrategy;
    }

    /**
     * @return the brainTreeTransactionService
     */
    public BrainTreeTransactionService getBrainTreeTransactionService() {
        return brainTreeTransactionService;
    }

    /**
     * @param brainTreeTransactionService
     *           the brainTreeTransactionService to set
     */
    public void setBrainTreeTransactionService(final BrainTreeTransactionService brainTreeTransactionService) {
        this.brainTreeTransactionService = brainTreeTransactionService;
    }

    /**
     * @return the addressReverseConverter
     */
    public Converter<AddressData, AddressModel> getAddressReverseConverter() {
        return addressReverseConverter;
    }

    /**
     * @param addressReverseConverter
     *           the addressReverseConverter to set
     */
    public void setAddressReverseConverter(final Converter<AddressData, AddressModel> addressReverseConverter) {
        this.addressReverseConverter = addressReverseConverter;
    }

    /**
     * @return the brainTreeSubscriptionInfoConverter
     */
    public Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> getBrainTreeSubscriptionInfoConverter() {
        return brainTreeSubscriptionInfoConverter;
    }

    /**
     * @param brainTreeSubscriptionInfoConverter
     *           the brainTreeSubscriptionInfoConverter to set
     */
    public void setBrainTreeSubscriptionInfoConverter(
        final Converter<BrainTreeSubscriptionInfoData, BraintreeInfo> brainTreeSubscriptionInfoConverter) {
        this.brainTreeSubscriptionInfoConverter = brainTreeSubscriptionInfoConverter;
    }

    public BraintreePaymentInfoService getPaymentInfoService() {
        return paymentInfoService;
    }

    public void setPaymentInfoService(BraintreePaymentInfoService paymentInfoService) {
        this.paymentInfoService = paymentInfoService;
    }

    public Converter<PaymentMethod, BrainTreePaymentInfoModel> getPaymentMethodConverter() {
        return paymentMethodConverter;
    }

    public void setPaymentMethodConverter(Converter<PaymentMethod, BrainTreePaymentInfoModel> paymentMethodConverter) {
        this.paymentMethodConverter = paymentMethodConverter;
    }

    public BrainTreeConfigService getBrainTreeConfigService() {
        return brainTreeConfigService;
    }

    public void setBrainTreeConfigService(BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }

    @Override
    public UserService getUserService() {
        return userService;
    }

    public DefaultBrainTreePaymentFacade getBrainTreePaymentFacade() {
        return brainTreePaymentFacade;
    }

    public void setBrainTreePaymentFacade(DefaultBrainTreePaymentFacade brainTreePaymentFacade) {
        this.brainTreePaymentFacade = brainTreePaymentFacade;
    }

    @Override
    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public BraintreeOrderBackofficeUtilService getBackofficeUtilService() {
        return backofficeUtilService;
    }

    public void setBackofficeUtilService(BraintreeOrderBackofficeUtilService backofficeUtilService) {
        this.backofficeUtilService = backofficeUtilService;
    }

    @Override
    public ModelService getModelService() {
        return modelService;
    }

    @Override
    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public Populator<CCPaymentInfoData, BrainTreePaymentInfoModel> getBraintreePaymentInfoReversePopulator() {
        return braintreePaymentInfoReversePopulator;
    }

    public void setBraintreePaymentInfoReversePopulator(
        Populator<CCPaymentInfoData, BrainTreePaymentInfoModel> braintreePaymentInfoReversePopulator) {
        this.braintreePaymentInfoReversePopulator = braintreePaymentInfoReversePopulator;
    }

    public BraintreeBillingAddressConverter getBillingAddressConverter() {
        return billingAddressConverter;
    }

    public void setBillingAddressConverter(
        BraintreeBillingAddressConverter billingAddressConverter) {
        this.billingAddressConverter = billingAddressConverter;
    }
}
