package com.braintree.facade.impl;

import static com.braintree.constants.BraintreeConstants.BRAINTREE_ACCEPTED_CREDIT_CARD_PAYMENT_METHODS;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_CREDITCARD_PAYMENT;
import static com.braintree.constants.BraintreeConstants.PAYPAL_INTENT_CAPTURE;
import static com.braintree.constants.BraintreeConstants.PAYPAL_INTENT_ORDER;
import static com.braintree.constants.BraintreeConstants.PAYPAL_INTENT_SALE;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import braintreehac.services.BrainTreePayPalButtonConfigurationService;
import com.braintree.command.result.BrainTreePaymentDetailsResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.configuration.service.BrainTreeSupportedLocaleConfig;
import com.braintree.constants.BraintreeConstants;
import com.braintree.customfield.service.BraintreeCustomFieldsService;
import com.braintree.delivery.service.BraintreeDeliveryService;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.enums.BraintreePageType;
import com.braintree.facade.BrainTreeUserFacade;
import com.braintree.hybris.data.ApplePayPaymentMethodData;
import com.braintree.hybris.data.BrainTreeSubscriptionInfoData;
import com.braintree.hybris.data.BraintreeAssetsData;
import com.braintree.hybris.data.BraintreeLocalPaymentMethodData;
import com.braintree.hybris.data.CreditCardPaymentMethodData;
import com.braintree.hybris.data.DropInData;
import com.braintree.hybris.data.GooglePayPaymentMethodData;
import com.braintree.hybris.data.LocalPaymentMethodData;
import com.braintree.hybris.data.PayPalAddressData;
import com.braintree.hybris.data.PayPalCheckoutData;
import com.braintree.hybris.data.PayPalConfigurationData;
import com.braintree.hybris.data.PayPalPaymentMethodData;
import com.braintree.hybris.data.SrcPaymentMethodData;
import com.braintree.hybris.data.VenmoPaymentMethodData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.model.BraintreeLocalPaymentMethodsModel;
import com.braintree.paypal.converters.impl.BrainTreePayPalAddressDataConverter;
import com.braintree.transaction.service.BrainTreeTransactionService;
import de.hybris.platform.acceleratorfacades.order.impl.DefaultAcceleratorCheckoutFacade;
import de.hybris.platform.catalog.CatalogVersionService;
import de.hybris.platform.catalog.model.CatalogVersionModel;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.core.model.media.MediaModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.exceptions.UnknownIdentifierException;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.media.MediaService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.util.PriceValue;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.assertj.core.util.Lists;

/**
 * This class extends DefaultAcceleratorCheckoutFacade
 */
public class DefaultBrainTreeCheckoutFacade extends DefaultAcceleratorCheckoutFacade {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeCheckoutFacade.class);

    private Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter;
    private Converter<BraintreeLocalPaymentMethodsModel, BraintreeLocalPaymentMethodData> brainTreeLocalPaymentsConverter;
    private BrainTreePaymentService brainTreePaymentService;
    private BrainTreeTransactionService brainTreeTransactionService;
    private I18NService i18NService;
    private CartService cartService;
    private UserService userService;
    private MediaService mediaService;
    private CatalogVersionService catalogVersionService;
    private BraintreeDeliveryService braintreeDeliveryService;
    private BrainTreePayPalAddressDataConverter payPalAddressDataConverter;
    private BrainTreeConfigService brainTreeConfigService;
    private BrainTreePayPalButtonConfigurationService payPalButtonConfigurationService;
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;
    private BrainTreeUserFacade brainTreeUserFacade;
    private BraintreeCustomFieldsService customFieldsService;
    private Converter<CartModel, CartData> cartConverter;
    private Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> ordermanagementPaymentTransactionEntryConverter;

    public void storeIntentToCart() {
        CartModel cart = cartService.getSessionCart();
        storeIntentToCart(cart);
    }

    public void storeIntentToCart(CartModel cart) {
        BrainTreePaymentInfoModel paymentInfo = (BrainTreePaymentInfoModel) cart.getPaymentInfo();
        paymentInfo.setPayPalIntent(brainTreeConfigService.getIntent());

        getModelService().save(paymentInfo);
    }

    public void storeShipsFromPostalCodeToCart(final String shipsFromPostalCode) {
        CartModel cart = cartService.getSessionCart();
        storeShipsFromPostalCodeToCart(shipsFromPostalCode, cart);
    }

    public void storeShipsFromPostalCodeToCart(final String shipsFromPostalCode, final CartModel cart) {
        BrainTreePaymentInfoModel paymentInfo = (BrainTreePaymentInfoModel) cart.getPaymentInfo();
        paymentInfo.setShipsFromPostalCode(shipsFromPostalCode);

        getModelService().save(paymentInfo);
    }

    public void storeCustomFieldsToCart(final Map<String, String> customFields) {
        CartModel cart = cartService.getSessionCart();
        storeCustomFieldsToCart(customFields, cart);
    }

    public void storeCustomFieldsToCart(final Map<String, String> customFields, final CartModel cart) {
        BrainTreePaymentInfoModel paymentInfo = (BrainTreePaymentInfoModel) cart.getPaymentInfo();
        paymentInfo.setCustomFields(customFields);

        getModelService().save(paymentInfo);
    }

    @Override
    public boolean authorizePayment(final String securityCode) {
        if (isPayPalPaymentMethodOrderIntent()) {
            return true;
        }
        try {
            return authorizePayment(securityCode, getCustomFields());
        } catch (final AdapterException ae) {
            LOG.error(ae.getMessage(), ae);
            return false;
        }
    }

    private boolean isPayPalPaymentMethodOrderIntent() {
        CCPaymentInfoData paymentInfo = getCheckoutCart().getPaymentInfo();
        boolean isPayPalCheckout = BraintreeConstants.PAY_PAL_EXPRESS_CHECKOUT.equals(paymentInfo.getSubscriptionId())
            || BraintreeConstants.PAYPAL_PAYMENT.equals(paymentInfo.getSubscriptionId());
        boolean isOrderIntent = BraintreeConstants.PAYPAL_INTENT_ORDER
            .equalsIgnoreCase(brainTreeConfigService.getIntent());

        return isOrderIntent && isPayPalCheckout;
    }

    public boolean authorizePayment(final String securityCode, Map<String, String> customFields) {
        if (getCart().getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            LOG.info("!isAvailablePaymentAuthorization: " + !isAvailablePaymentAuthorization());
            if (!isAvailablePaymentAuthorization()) {
                LOG.warn("calling authorizePaymentIfIntentOrder");
                return authorizePaymentIfIntentOrder(customFields);
            }
            if (!brainTreeConfigService.isOneOfPaymentMethodsEnabled()) {
                LOG.info("Use default accelerator checkout flow.");
                return super.authorizePayment(securityCode);
            }

            return brainTreeTransactionService.createAuthorizationTransaction(customFields);

        } else {

            LOG.info(
                "Payment info is not instance of BrainTreePaymentInfoModel. Use default accelerator checkout flow.");
            return super.authorizePayment(securityCode);
        }
    }

    public boolean authorizePayment(CartModel cart) {
        PaymentTransactionEntryModel paymentTransactionEntryModel = brainTreeTransactionService
            .createAuthorizationTransaction(cart);
        return paymentTransactionEntryModel != null
            && (TransactionStatus.ACCEPTED.name().equals(paymentTransactionEntryModel.getTransactionStatus())
            || TransactionStatus.REVIEW.name().equals(paymentTransactionEntryModel.getTransactionStatus()));
    }

    public PaymentTransactionEntryData authorizePayment(OrderModel order, BigDecimal amount) {
        return ordermanagementPaymentTransactionEntryConverter
            .convert(brainTreeTransactionService.createAuthorizationTransaction(order, amount));
    }

    @Override
    public CCPaymentInfoData getPaymentDetails() {
        final CartModel cart = getCart();
        if (cart != null) {
            final PaymentInfoModel paymentInfo = cart.getPaymentInfo();
            if (paymentInfo instanceof BrainTreePaymentInfoModel) {
                return brainTreePaymentInfoConverter.convert((BrainTreePaymentInfoModel) paymentInfo);
            } else {
                return super.getPaymentDetails();
            }
        }

        return null;
    }

    public boolean setPaymentDetails(final String paymentInfoId, final String paymentMethodNonce) {
        validateParameterNotNullStandardMessage("paymentInfoId", paymentInfoId);

        if (checkIfCurrentUserIsTheCartUser()) {
            final CustomerModel currentUserForCheckout = getCurrentUserForCheckout();
            if (StringUtils.isNotBlank(paymentInfoId)) {
                final BrainTreePaymentInfoModel paymentInfo = brainTreePaymentService
                    .completeCreateSubscription(currentUserForCheckout, paymentInfoId);
                if (paymentInfo != null) {
                    paymentInfo.setNonce(paymentMethodNonce);
                    getModelService().save(paymentInfo);
                    return true;
                } else {
                    super.setPaymentDetails(paymentInfoId);
                }
            }
        }
        return false;
    }

    @Override
    public boolean setPaymentDetails(final String paymentInfoId) {
        validateParameterNotNullStandardMessage("paymentInfoId", paymentInfoId);

        if (checkIfCurrentUserIsTheCartUser()) {
            final CustomerModel currentUserForCheckout = getCurrentUserForCheckout();
            if (StringUtils.isNotBlank(paymentInfoId)) {
                final BrainTreePaymentInfoModel paymentInfo = brainTreePaymentService
                    .completeCreateSubscription(currentUserForCheckout, paymentInfoId);
                if (paymentInfo != null) {
                    return true;
                } else {
                    return super.setPaymentDetails(paymentInfoId);
                }

            }
        }

        return false;
    }

    public String generateClientToken(final boolean generateWithBraintreeCustomerId) {
        String clientToken = StringUtils.EMPTY;
        try {
            UserModel userModel = userService.getCurrentUser();
            String braintreeCustomerId = ((CustomerModel) userModel).getBraintreeCustomerId();
            if (generateWithBraintreeCustomerId && braintreeCustomerId != null) {
                clientToken = brainTreePaymentService
                    .generateClientToken(braintreeCustomerId);
            } else {
                clientToken = brainTreePaymentService.generateClientToken(StringUtils.EMPTY);
            }
        } catch (final AdapterException exception) {
            LOG.error("[View Handler] Error during token generation!", exception);
        }
        return clientToken;
    }

    public PayPalCheckoutData getPayPalCheckoutData(final BraintreePageType pageType) {
        PayPalCheckoutData payPalCheckoutData = generateConfigurationData(pageType);

        if (payPalCheckoutData != null) {
            payPalCheckoutData.setApplePayPaymentMethod(generateApplePayMethodData());
            payPalCheckoutData.setGooglePayPaymentMethod(generateGooglePayMethodData());
            payPalCheckoutData.setSrcPaymentMethod(generateSrcMethodData());
            if (checkPageType(pageType)) {
                payPalCheckoutData.setVenmoPaymentMethod(generateVenmoMethodData());
                payPalCheckoutData.setCreditCardPaymentMethod(generateCreditCardMethodData());
                if (pageType.equals(BraintreePageType.MARK)) {
                    if (cartService.hasSessionCart() && cartService.getSessionCart().getDeliveryAddress() != null) {
                        final PayPalAddressData payPalAddress = payPalAddressDataConverter
                            .convert(cartService.getSessionCart().getDeliveryAddress());
                        payPalCheckoutData.setShippingAddressOverride(payPalAddress);
                    }
                    payPalCheckoutData.setLocalPaymentMethod(generateLocalPaymentMethodData());
                } else if (checkPageTypeAndConfigService(pageType, payPalCheckoutData)) {
                    payPalCheckoutData = setDateIntoPayPalCheckoutData(payPalCheckoutData);
                }
            }
        }
        return payPalCheckoutData;
    }

    private PayPalCheckoutData setDateIntoPayPalCheckoutData(PayPalCheckoutData payPalCheckoutData) {
        UserModel userModel = userService.getCurrentUser();
        List<AddressModel> addressModels = userModel.getAddresses()
                .stream()
                .filter(addressModel -> !addressModel.getDuplicate() && addressModel.getVisibleInAddressBook())
                .collect(Collectors.toList());
        if (addressModels.size() > 0) {
            AddressModel sourceAddress = addressModels.get(addressModels.size() - 1);
            final PayPalAddressData payPalAddress = payPalAddressDataConverter
                    .convert(sourceAddress);
            if (sourceAddress.getCountry() != null) {
                Optional.ofNullable(getCart()).ifPresent(cartModel -> payPalCheckoutData.setDeliveryAmount(
                        countDeliveryAddress(sourceAddress.getCountry().getIsocode()).getValue()));
            }
            payPalCheckoutData.setShippingAddressOverride(payPalAddress);
        } else {
            payPalCheckoutData.getPayPalPaymentMethod().setShouldShowShippingAddressMessage(true);
            payPalCheckoutData.getPayPalPaymentMethod().setShouldRenderPayPalChangePaymentButton(false);
        }
        if (payPalCheckoutData.getPayPalPaymentMethod().getShouldShowShippingAddressMessage() && brainTreeUserFacade.isAnonymousUser()) {
            payPalCheckoutData.getPayPalPaymentMethod().setPayPalExpressEnabled(false);
        }
        return payPalCheckoutData;
    }

    private boolean checkPageTypeAndConfigService(BraintreePageType pageType, PayPalCheckoutData payPalCheckoutData) {
        return isPageTypeCartOrMinicart(pageType)
                && (payPalCheckoutData.getPayPalPaymentMethod().getShouldRenderPayPalChangePaymentButton()
                || (getBrainTreeConfigService().isBrainTreeGraphQLEnabled()
                && PAYPAL_INTENT_ORDER.equals(getBrainTreeConfigService().getIntent())));
    }

    private boolean isPageTypeCartOrMinicart(BraintreePageType pageType) {
        return pageType.equals(BraintreePageType.CART) || pageType.equals(BraintreePageType.MINICART);
    }

    private boolean checkPageType(BraintreePageType pageType) {
        return pageType.equals(BraintreePageType.ACCOUNT) || pageType.equals(BraintreePageType.MARK)
                || pageType.equals(BraintreePageType.CART) || pageType.equals(BraintreePageType.MINICART);
    }

    /**
     * Used to generate configuration data
     * @param pageType
     * @return PayPalCheckoutData
     */
    public PayPalCheckoutData generateConfigurationData(final BraintreePageType pageType) {
        final PayPalCheckoutData payPalCheckoutData = new PayPalCheckoutData();
        PayPalConfigurationData payPalConfigurationData = new PayPalConfigurationData();
        payPalConfigurationData.
            setStoreInVault(String.valueOf(getBrainTreeConfigService().getStoreInVaultIgnoringIntent()));
        payPalConfigurationData.setIntent(getBrainTreeConfigService().getIntent());
        payPalConfigurationData.setCurrency(getI18NService().getCurrentJavaCurrency().getCurrencyCode());
        payPalConfigurationData.setEnvironment(getBrainTreeConfigService().getEnvironmentTypeName());
        payPalConfigurationData.setLocale(getI18NService().getCurrentLocale().toString());
        payPalConfigurationData.setAdvancedFraudTools(getBrainTreeConfigService().getAdvancedFraudTools());
        payPalConfigurationData.setDbaName(getBrainTreeConfigService().getCreditCardStatementName());
        payPalConfigurationData.setEndpointURL(getBrainTreeConfigService().getEndpointURL());
        payPalConfigurationData.setUserAction(getBrainTreeConfigService().getUserAction());
        payPalConfigurationData
            .setCurrencyMerchantAccountId(getBrainTreeConfigService().getCurrencyMerchantAccountId());
        payPalConfigurationData.setGraphQLEnabled(getBrainTreeConfigService().isBrainTreeGraphQLEnabled());
        if (cartService.hasSessionCart()) {
            CartModel cart = cartService.getSessionCart();
            payPalCheckoutData.setEnableShippingAddress(!isOnlyPickUp());
            payPalCheckoutData.setAmount(cart.getTotalPrice());
            payPalCheckoutData.setSubtotal(cart.getSubtotal());
        } else {
            payPalCheckoutData.setEnableShippingAddress(false);
        }
        payPalCheckoutData.setPayPalPaymentMethod(generatePayPalMethodData(pageType));
        payPalCheckoutData.setDropIn(generateDropInData());
        payPalCheckoutData.setConfigurationData(payPalConfigurationData);
        payPalCheckoutData.getConfigurationData().setClient_id(generateClientToken(
            payPalCheckoutData.getPayPalPaymentMethod().getShouldRenderPayPalChangePaymentButton()
                || payPalCheckoutData.getDropIn().getDropInEnabled()));
        payPalCheckoutData.getConfigurationData().setBraintreeLocale(getBraintreeLocale());

        return payPalCheckoutData;
    }

    private Boolean isOnlyPickUp() {
        return getCheckoutCart().getDeliveryOrderGroups().isEmpty();
    }

    private PayPalPaymentMethodData generatePayPalMethodData(final BraintreePageType pageType) {
        PayPalPaymentMethodData paymentMethodData = new PayPalPaymentMethodData();
        paymentMethodData.setBillingAgreementDescription(
            (String) getBrainTreeConfigService().getBillingAgreementDescription());
        paymentMethodData.setCreditEnabled(getBrainTreeConfigService().getCreditEnabled());
        paymentMethodData.setPayPalStandardEnabled(getBrainTreeConfigService().getPayPalStandardEnabled());
        paymentMethodData.setPayPalExpressEnabled(getBrainTreeConfigService().getPayPalExpressEnabled());
        paymentMethodData.setShouldRenderPayPalChangePaymentButton(getShouldRenderPayPalChangePaymentButton(pageType));
        paymentMethodData.setDisableFunding(getBrainTreeConfigService().getDisableFunding(pageType));
        paymentMethodData
            .setIsPayLaterForNonUsCountriesEnabled(getBrainTreeConfigService().isPayLaterEnabledForNonUsCountries());
        paymentMethodData.setShouldShowShippingAddressMessage(false);
        return paymentMethodData;
    }

    private boolean getShouldRenderPayPalChangePaymentButton(BraintreePageType pageType) {
        List<AddressModel> savedAddresses = userService.getCurrentUser().getAddresses()
            .stream()
            .filter(addressModel -> !addressModel.getDuplicate() && addressModel.getVisibleInAddressBook())
            .collect(Collectors.toList());

        String braintreeCustomerId = ((CustomerModel) userService.getCurrentUser()).getBraintreeCustomerId();

        List<BrainTreePaymentDetailsResult> savedPaymentMethods = new ArrayList<>();

        if (braintreeCustomerId != null) {
            List<BrainTreePaymentDetailsResult> paymentMethods = brainTreeUserFacade
                .findBrainTreeCustomer(braintreeCustomerId)
                .getPaymentMethods();
            if (paymentMethods != null) {
                savedPaymentMethods = paymentMethods
                    .stream()
                    .filter(brainTreePaymentDetailsResult -> PAYPAL_PAYMENT
                        .equals(brainTreePaymentDetailsResult.getPaymentProvider()))
                    .collect(Collectors.toList());
            }
        }
        return canRenderPayPalChangePaymentButton(pageType, savedAddresses, savedPaymentMethods);
    }

    private boolean canRenderPayPalChangePaymentButton(BraintreePageType pageType, List<AddressModel> savedAddresses,
                                                       List<BrainTreePaymentDetailsResult> savedPaymentMethods) {
        final boolean isConfigDataTrue = getBrainTreeConfigService().getChangePaymentMethodButtonEnabled()
                && !getBrainTreeConfigService().getDropInEnabled();
        final boolean isUserDataTrue = !brainTreeUserFacade.isAnonymousUser()
                && brainTreeUserFacade.isCurrentUserHasBrainTreeCustomerId();
        final boolean isPageTypeAndSavedAddressesAndSavedPaymentMethodsTrue
                = !BraintreePageType.ACCOUNT.equals(pageType) && (BraintreePageType.MARK.equals(pageType)
                || savedAddresses.size() > 0) && savedPaymentMethods.size() > 0;
        return isConfigDataTrue && isUserDataTrue && isPageTypeAndSavedAddressesAndSavedPaymentMethodsTrue;
    }

    private DropInData generateDropInData() {
        DropInData dropInData = new DropInData();
        dropInData.setDropInEnabled(getBrainTreeConfigService().getDropInEnabled());
        dropInData.setIsLimitedDeletePaymentMethods(
            brainTreeUserFacade.isCustomerHasNotCapturedOrdersForSavedPaymentMethodType(BRAINTREE_CREDITCARD_PAYMENT));
        return dropInData;
    }

    private GooglePayPaymentMethodData generateGooglePayMethodData() {
        GooglePayPaymentMethodData paymentMethodData = new GooglePayPaymentMethodData();
        paymentMethodData.setGoogleMerchantId(getBrainTreeConfigService().getGooglePayMerchantId());
        paymentMethodData.setGooglePayCountryCode(getBrainTreeConfigService().getGooglePayCountryCode());
        paymentMethodData.setGooglePayEnabled(getBrainTreeConfigService().getGooglePayEnabled());
        return paymentMethodData;
    }

    private CreditCardPaymentMethodData generateCreditCardMethodData() {
        CreditCardPaymentMethodData paymentMethodData = new CreditCardPaymentMethodData();
        paymentMethodData
            .setPaymentsImagesUrl(getAcceptedPaymentMethodImages(BRAINTREE_ACCEPTED_CREDIT_CARD_PAYMENT_METHODS));
        paymentMethodData.setSecure3d(getBrainTreeConfigService().get3dSecureConfiguration());
        paymentMethodData.
            setSkip3dSecureLiabilityResult(getBrainTreeConfigService().resolveSkip3DSecureLiabilityResult());
        paymentMethodData.setHostedFieldsEnable(getBrainTreeConfigService().getHostedFieldEnabled());
        return paymentMethodData;
    }

    private ApplePayPaymentMethodData generateApplePayMethodData() {
        ApplePayPaymentMethodData paymentMethodData = new ApplePayPaymentMethodData();
        paymentMethodData.setApplePayEnabled(String.valueOf(getBrainTreeConfigService().getApplePayEnabled()));
        return paymentMethodData;
    }

    private VenmoPaymentMethodData generateVenmoMethodData() {
        VenmoPaymentMethodData paymentMethodData = new VenmoPaymentMethodData();
        paymentMethodData.setVenmoProfileId(getBrainTreeConfigService().getVenmoProfileId());
        paymentMethodData.setVenmoEnabled(getBrainTreeConfigService().getVenmoEnabled());
        return paymentMethodData;
    }

    private SrcPaymentMethodData generateSrcMethodData() {
        SrcPaymentMethodData paymentMethodData = new SrcPaymentMethodData();
        paymentMethodData.setSrcEnabled(getBrainTreeConfigService().getSrcEnabled());
        paymentMethodData.setSrcSdkUrl(getBrainTreeConfigService().getSrcSdkUrl());
        paymentMethodData.setSrcImgUrl(getBrainTreeConfigService().getSrcImgUrl());
        return paymentMethodData;
    }

    private LocalPaymentMethodData generateLocalPaymentMethodData() {
        LocalPaymentMethodData paymentMethodData = new LocalPaymentMethodData();
        boolean localPaymentsEnabled = brainTreeConfigService.getLocalPaymentsEnabled() &&
            brainTreeConfigService.getSettlementConfigParameter() &&
            (PAYPAL_INTENT_SALE.equalsIgnoreCase(brainTreeConfigService.getIntent()) ||
                PAYPAL_INTENT_CAPTURE.equalsIgnoreCase(brainTreeConfigService.getIntent()));
        paymentMethodData.setLocalPaymentMethods(getBrainTreeLocalPaymentsConverter()
            .convertAll(brainTreePaymentFacade.getLocalPaymentMethods()));
        paymentMethodData.setLocalPaymentMethodsEnabled(localPaymentsEnabled);
        return paymentMethodData;
    }

    /**
     * Used to count delivery address
     * @param countyCode
     * @return PriceValue
     */
    public PriceValue countDeliveryAddress(String countyCode) {
        if (getCart() != null) {
            return braintreeDeliveryService.countDeliveryPriceForOrderAndCountyCode(getCart(), countyCode);
        }
        return null;
    }

    public BraintreeAssetsData getBraintreeAssets() {
        BraintreeAssetsData braintreeAssetsData = new BraintreeAssetsData();
        braintreeAssetsData.setGooglePayLogo(getBraintreeImageUrlByCode("googlePayMark"));
        braintreeAssetsData.setVenmoLogo(getBraintreeImageUrlByCode("venmoLogo"));
        braintreeAssetsData.setVenmoMarkImg(getBraintreeImageUrlByCode("venmoAcceptanceMark"));
        braintreeAssetsData.setApplePayLogo(getBraintreeImageUrlByCode("applePayLogo"));
        return braintreeAssetsData;
    }

    private String getBraintreeImageUrlByCode(String code) {
        if (StringUtils.isNotEmpty(code)) {
            for (final CatalogVersionModel catalogVersionModel : catalogVersionService.getSessionCatalogVersions()) {
                try {
                    final MediaModel media = mediaService.getMedia(catalogVersionModel, code);
                    if (media != null) {
                        return media.getDownloadURL();
                    }
                } catch (final UnknownIdentifierException ignore) {
                    // Ignore this exception
                }
            }
        }
        return null;
    }

    public String getBraintreeLocale() {
        final String brainTreeLocale = brainTreeConfigService.getBrainTreeLocale();
        try {
            if (StringUtils.isNotEmpty(brainTreeLocale) && BrainTreeSupportedLocaleConfig
                .supportLocale(brainTreeLocale)) {
                return brainTreeLocale;
            } else {
                return BrainTreeSupportedLocaleConfig
                    .getSupportedLocaleTagByLanguage(getI18NService().getCurrentLocale().getLanguage());
            }
        } catch (final Exception e) {
            LOG.error(e.getMessage(), e);
            return BrainTreeSupportedLocaleConfig
                .getDefaultLocaleTag(getI18NService().getCurrentLocale().getLanguage());
        }
    }

    public boolean isAvailablePaymentAuthorization() {
        boolean isAccountPaymentSystems = isApplePay() || isVenmo() || isGooglePay();
        if (isCreditCard() || isAccountPaymentSystems || isSrc()) {
            return true;
        }
        boolean paypalIntentIsOrder = PAYPAL_INTENT_ORDER.equals(getBrainTreeConfigService().getIntent());
        boolean config = Boolean.parseBoolean(getBrainTreeConfigService().getStoreInVaultForCurrentUser());
        return !(paypalIntentIsOrder && !config);
    }

    private boolean isCreditCard() {
        PaymentInfoModel paymentInfoModel = getCart().getPaymentInfo();
        if (paymentInfoModel instanceof BrainTreePaymentInfoModel) {
            return BrainTreePaymentMethod.CREDITCARD.getCode()
                .equalsIgnoreCase(((BrainTreePaymentInfoModel) paymentInfoModel).getPaymentProvider());
        }
        return false;
    }

    private boolean isApplePay() {
        PaymentInfoModel paymentInfoModel = getCart().getPaymentInfo();
        if (paymentInfoModel instanceof BrainTreePaymentInfoModel) {
            return BrainTreePaymentMethod.APPLEPAYCARD.getCode()
                .equalsIgnoreCase(((BrainTreePaymentInfoModel) paymentInfoModel).getPaymentProvider());
        }
        return false;
    }

    private boolean isGooglePay() {
        PaymentInfoModel paymentInfoModel = getCart().getPaymentInfo();
        if (paymentInfoModel instanceof BrainTreePaymentInfoModel) {
            return BrainTreePaymentMethod.ANDROIDPAYCARD.getCode()
                .equalsIgnoreCase(((BrainTreePaymentInfoModel) paymentInfoModel).getPaymentProvider());
        }
        return false;
    }

    private boolean isSrc() {
        PaymentInfoModel paymentInfoModel = getCart().getPaymentInfo();
        if (paymentInfoModel instanceof BrainTreePaymentInfoModel) {
            return BrainTreePaymentMethod.VISACHECKOUTCARD.getCode()
                .equalsIgnoreCase(((BrainTreePaymentInfoModel) paymentInfoModel).getPaymentProvider());
        }
        return false;
    }

    private boolean isVenmo() {
        PaymentInfoModel paymentInfoModel = getCart().getPaymentInfo();
        if (paymentInfoModel instanceof BrainTreePaymentInfoModel) {
            return BrainTreePaymentMethod.VENMOACCOUNT.getCode()
                .equalsIgnoreCase(((BrainTreePaymentInfoModel) paymentInfoModel).getPaymentProvider());
        }
        return false;
    }

    public Boolean isPayPalCheckout(final OrderModel order) {
        final String paymentProvider = order.getPaymentInfo().getProperty("paymentProvider");
        return paymentProvider != null && (BraintreeConstants.PAY_PAL_EXPRESS_CHECKOUT.equals(paymentProvider)
            || BraintreeConstants.PAYPAL_PAYMENT.equals(paymentProvider));
    }

    public AddressData getDeliveryAddressForCurrentCart() {
        return getDeliveryAddress();
    }

    public void setBrainTreePaymentInfoConverter(
        final Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> brainTreePaymentInfoConverter) {
        this.brainTreePaymentInfoConverter = brainTreePaymentInfoConverter;
    }

    /**
     * Used to get accepted payment method images
     * @param paymentMethodsString
     * @return Map<String, String>
     */
    public Map<String, String> getAcceptedPaymentMethodImages(final String paymentMethodsString) {
        return brainTreeConfigService.getAcceptedPaymentMethodImages(paymentMethodsString);
    }

    private boolean authorizePaymentIfIntentOrder(Map<String, String> customFields) {
        createPaymentMethodIfIntentOrder(customFields);
        return true;
    }

    private void createPaymentMethodIfIntentOrder(Map<String, String> customFields) {
        CartModel cart = getCartService().getSessionCart();
        final BrainTreeSubscriptionInfoData subscriptionInfo = buildSubscriptionInfo(
            (BrainTreePaymentInfoModel) cart.getPaymentInfo(), true);
        subscriptionInfo.setIntent(BraintreeConstants.PAYPAL_INTENT_ORDER);
        subscriptionInfo.setAmount(String.valueOf(cart.getTotalPrice()));
        brainTreePaymentFacade.completeCreateSubscription(subscriptionInfo);

        BrainTreePaymentInfoModel brainTreePaymentInfo = (BrainTreePaymentInfoModel) cart.getPaymentInfo();
        brainTreePaymentInfo.setPayPalIntent(BraintreeConstants.PAYPAL_INTENT_ORDER);
        brainTreePaymentInfo.setSaved(false);
        brainTreePaymentInfo.setCustomFields(customFields);
        getModelService().save(brainTreePaymentInfo);
    }

    private BrainTreeSubscriptionInfoData buildSubscriptionInfo(final BrainTreePaymentInfoModel paymentInfoModel,
        final boolean isPaymentInfoSaved) {
        final BrainTreeSubscriptionInfoData subscriptionInfo = new BrainTreeSubscriptionInfoData();
        subscriptionInfo.setPaymentProvider(paymentInfoModel.getPaymentProvider());
        subscriptionInfo.setCardNumber(paymentInfoModel.getCardNumber());
        subscriptionInfo.setDeviceData(paymentInfoModel.getDeviceData());
        subscriptionInfo.setNonce(paymentInfoModel.getNonce());
        subscriptionInfo.setSavePaymentInfo(isPaymentInfoSaved);
        subscriptionInfo.setCardholder(paymentInfoModel.getCardholderName());
        if (paymentInfoModel.getLiabilityShifted() != null) {
            subscriptionInfo.setLiabilityShifted(paymentInfoModel.getLiabilityShifted());
        }
        if (paymentInfoModel.getCardType() != null) {
            subscriptionInfo.setCardType(paymentInfoModel.getCardType().getCode());
        }
        if (paymentInfoModel.getBillingAddress() != null) {
            subscriptionInfo.setEmail(paymentInfoModel.getBillingAddress().getEmail());
        }
        return subscriptionInfo;
    }

    public void handleOrderIntentViaSubscription(final OrderModel order) {
        getModelService().refresh(order);
        UserModel user = order.getUser();

        final BrainTreeSubscriptionInfoData subscriptionInfo = buildSubscriptionInfo(
            (BrainTreePaymentInfoModel) order.getPaymentInfo(), false);
        subscriptionInfo.setIntent(BraintreeConstants.PAYPAL_INTENT_ORDER);
        subscriptionInfo.setAmount(String.valueOf(order.getTotalPrice()));
        final boolean isCreditEnabled = brainTreeConfigService.getCreditEnabled();
        brainTreePaymentFacade
            .completeCreateSubscription(subscriptionInfo, (CustomerModel) user, order, isCreditEnabled,
                BraintreePageType.OTHER);

        BrainTreePaymentInfoModel brainTreePaymentInfo = (BrainTreePaymentInfoModel) order.getPaymentInfo();
        brainTreePaymentInfo.setPayPalIntent(BraintreeConstants.PAYPAL_INTENT_ORDER);
        brainTreePaymentInfo.setSaved(false);
        brainTreePaymentInfo.setCustomFields(getCustomFields());

        String orderSite = order.getSite().getUid();
        String orderCurrency = order.getCurrency().getIsocode().toLowerCase();
        brainTreePaymentInfo.setMerchantAccountIdForCurrentSite(
            getBrainTreeConfigService().getMerchantAccountIdByCurrentSiteNameAndCurrency(orderSite, orderCurrency));

        getModelService().save(brainTreePaymentInfo);
    }

    public OrderData placeOrderByCart(CartModel cartModel) throws InvalidCartException {
        if (cartModel != null) {
            beforePlaceOrder(cartModel);
            final OrderModel orderModel = placeOrder(cartModel);
            afterPlaceOrder(cartModel, orderModel);
            if (orderModel != null) {
                return getOrderConverter().convert(orderModel);
            }
        }
        return null;
    }

    public OrderModel placeOrderForLocalPaymentReversed(CartModel cartModel) throws InvalidCartException {
        if (cartModel != null) {
            beforePlaceOrder(cartModel);
            final OrderModel orderModel = placeOrder(cartModel);
            afterPlaceOrder(cartModel, orderModel);
            return orderModel;
        }
        return null;
    }

    @Override
    public void beforePlaceOrder(@SuppressWarnings("unused") final CartModel cartModel) // NOSONAR
    {
        PaymentInfoModel paymentInfo = cartModel.getPaymentInfo();
        if (paymentInfo instanceof BrainTreePaymentInfoModel) {

            storeIntentToCart(cartModel);

            if (((BrainTreePaymentInfoModel) paymentInfo).getShipsFromPostalCode() == null) {
                storeShipsFromPostalCodeToCart(cartModel.getDeliveryAddress().getPostalcode(), cartModel);
            }

            if (((BrainTreePaymentInfoModel) paymentInfo).getCustomFields() == null) {
                storeCustomFieldsToCart(getCustomFieldsService().getDefaultCustomFieldsMap(), cartModel);
            }
        }
    }

    @Override
    protected void afterPlaceOrder(CartModel cartModel, OrderModel orderModel) {
        if (orderModel != null) {
            getModelService().remove(cartModel);
            getModelService().refresh(orderModel);
        }
    }

    /**
     * This method is used to process Local Payment Reversed Webhook
     *
     * @param cartModel cart
     */
    public void processLocalPaymentReversedWebhook(final CartModel cartModel) {
        List<PaymentTransactionEntryModel> paymentTransactionEntrys = Lists.newArrayList();
        paymentTransactionEntrys.add(brainTreeTransactionService
            .createTransactionEntryForLocalPaymentReversed(cartModel, PaymentTransactionType.AUTHORIZATION));
        paymentTransactionEntrys.add(brainTreeTransactionService
            .createTransactionEntryForLocalPaymentReversed(cartModel, PaymentTransactionType.CAPTURE));
        paymentTransactionEntrys.add(brainTreeTransactionService
            .createTransactionEntryForLocalPaymentReversed(cartModel, PaymentTransactionType.REFUND_FOLLOW_ON));

        brainTreeTransactionService
            .createPaymentTransactionForLocalPaymentReversed(cartModel, paymentTransactionEntrys);
    }


    private Map<String, String> getCustomFields() {
        return customFieldsService.getDefaultCustomFieldsMap();
    }

    /**
     * @return the brainTreePaymentService
     */
    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    /**
     * @param brainTreePaymentService the brainTreePaymentService to set
     */
    public void setBrainTreePaymentService(final BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }

    /**
     * @return the brainTreeTransactionService
     */
    public BrainTreeTransactionService getBrainTreeTransactionService() {
        return brainTreeTransactionService;
    }

    /**
     * @param brainTreeTransactionService the brainTreeTransactionService to set
     */
    public void setBrainTreeTransactionService(final BrainTreeTransactionService brainTreeTransactionService) {
        this.brainTreeTransactionService = brainTreeTransactionService;
    }

    /**
     * @return the cartService
     */
    @Override
    public CartService getCartService() {
        return cartService;
    }

    /**
     * @param cartService the cartService to set
     */
    @Override
    public void setCartService(final CartService cartService) {
        this.cartService = cartService;
    }

    /**
     * @return the userService
     */
    @Override
    public UserService getUserService() {
        return userService;
    }

    /**
     * @param userService the userService to set
     */
    @Override
    public void setUserService(final UserService userService) {
        this.userService = userService;
    }

    /**
     * @return the payPalAddressDataConverter
     */
    public BrainTreePayPalAddressDataConverter getPayPalAddressDataConverter() {
        return payPalAddressDataConverter;
    }

    /**
     * @param payPalAddressDataConverter the payPalAddressDataConverter to set
     */
    public void setPayPalAddressDataConverter(final BrainTreePayPalAddressDataConverter payPalAddressDataConverter) {
        this.payPalAddressDataConverter = payPalAddressDataConverter;
    }

    /**
     * @return the brainTreeConfigService
     */
    public BrainTreeConfigService getBrainTreeConfigService() {
        return brainTreeConfigService;
    }

    /**
     * @param brainTreeConfigService the brainTreeConfigService to set
     */
    public void setBrainTreeConfigService(final BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }

    /**
     * @return the brainTreePaymentInfoConverter
     */
    public Converter<BrainTreePaymentInfoModel, CCPaymentInfoData> getBrainTreePaymentInfoConverter() {
        return brainTreePaymentInfoConverter;
    }

    public Converter<BraintreeLocalPaymentMethodsModel, BraintreeLocalPaymentMethodData> getBrainTreeLocalPaymentsConverter() {
        return brainTreeLocalPaymentsConverter;
    }

    public void setBrainTreeLocalPaymentsConverter(
        Converter<BraintreeLocalPaymentMethodsModel, BraintreeLocalPaymentMethodData>
            brainTreeLocalPaymentsConverter) {
        this.brainTreeLocalPaymentsConverter = brainTreeLocalPaymentsConverter;
    }

    public DefaultBrainTreePaymentFacade getBrainTreePaymentFacade() {
        return brainTreePaymentFacade;
    }

    public void setBrainTreePaymentFacade(DefaultBrainTreePaymentFacade brainTreePaymentFacade) {
        this.brainTreePaymentFacade = brainTreePaymentFacade;
    }

    public BraintreeCustomFieldsService getCustomFieldsService() {
        return customFieldsService;
    }

    public void setCustomFieldsService(BraintreeCustomFieldsService customFieldsService) {
        this.customFieldsService = customFieldsService;
    }

    public BrainTreePayPalButtonConfigurationService getPayPalButtonConfigurationService() {
        return payPalButtonConfigurationService;
    }

    public void setPayPalButtonConfigurationService(
        BrainTreePayPalButtonConfigurationService payPalButtonConfigurationService) {
        this.payPalButtonConfigurationService = payPalButtonConfigurationService;
    }

    public I18NService getI18NService() {
        return i18NService;
    }

    public void setI18NService(I18NService i18NService) {
        this.i18NService = i18NService;

    }

    public Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> getOrdermanagementPaymentTransactionEntryConverter() {
        return ordermanagementPaymentTransactionEntryConverter;
    }

    public void setOrdermanagementPaymentTransactionEntryConverter(
        Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> ordermanagementPaymentTransactionEntryConverter) {
        this.ordermanagementPaymentTransactionEntryConverter = ordermanagementPaymentTransactionEntryConverter;
    }

    public Converter<CartModel, CartData> getCartConverter() {
        return cartConverter;
    }

    public void setCartConverter(Converter<CartModel, CartData> cartConverter) {
        this.cartConverter = cartConverter;
    }

    public MediaService getMediaService() {
        return mediaService;
    }

    public void setMediaService(MediaService mediaService) {
        this.mediaService = mediaService;
    }

    public CatalogVersionService getCatalogVersionService() {
        return catalogVersionService;
    }

    public void setCatalogVersionService(CatalogVersionService catalogVersionService) {
        this.catalogVersionService = catalogVersionService;
    }

    public BrainTreeUserFacade getBrainTreeUserFacade() {
        return brainTreeUserFacade;
    }

    public void setBrainTreeUserFacade(BrainTreeUserFacade brainTreeUserFacade) {
        this.brainTreeUserFacade = brainTreeUserFacade;
    }

    public BraintreeDeliveryService getBraintreeDeliveryService() {
        return braintreeDeliveryService;
    }

    public void setBraintreeDeliveryService(BraintreeDeliveryService braintreeDeliveryService) {
        this.braintreeDeliveryService = braintreeDeliveryService;
    }
}
