package com.braintree.facade.backoffice.impl;

import static com.braintree.constants.BraintreeConstants.BRAINTREE_INTERACTION_TYPE_VOID;

import com.braintree.command.request.BrainTreeDeletePaymentMethodRequest;
import com.braintree.command.result.BrainTreePaymentMethodResult;
import com.braintree.command.result.BrainTreeVoidResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants;
import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.facade.backoffice.BraintreeBackofficeVoidFacade;
import com.braintree.hybris.data.BrainTreeResponseResultData;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.service.BraintreeOrderBackofficeUtilService;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.braintreegateway.Transaction;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.commands.request.VoidRequest;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

/**
 * This class is a default implementation of the BraintreeBackofficeVoidFacade interface
 */
public class DefaultBraintreeBackofficeVoidFacade implements BraintreeBackofficeVoidFacade {

    private final static Logger LOG = Logger.getLogger(DefaultBraintreeBackofficeVoidFacade.class);

    @Autowired
    private BrainTreePaymentService brainTreePaymentService;
    @Autowired
    private ModelService modelService;
    @Autowired
    private UserService userService;
    @Autowired
    BraintreePaymentInfoService paymentInfoService;
    @Autowired
    private BrainTreeConfigService brainTreeConfigService;
    @Autowired
    private BraintreeOrderBackofficeUtilService backofficeUtilService;
    @Autowired
    @Qualifier("ordermanagementPaymentTransactionEntryConverter")
    private Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> ordermanagementPaymentTransactionEntryConverter;
    @Autowired
    @Qualifier("brainTreeTransactionService")
    private BrainTreeTransactionService brainTreeTransactionService;

    @Override
    public boolean isVoidPossible(final OrderModel order) {
        if (null == order) {
            LOG.warn("order: " + order);
            return false;
        }
        LOG.info("isVoidPossible, order.getTotalPrice: " + order.getTotalPrice());

        modelService.refresh(order);
        boolean isVoidPossible = false;
        Boolean settlementConfigParameter = brainTreeConfigService.getSettlementConfigParameter();

        if (order.getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            final String paymentProvider = ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPaymentProvider();
            final String intent = ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPayPalIntent();

            if (backofficeUtilService.paidByCard(order) || isVenmoAccount(paymentProvider)) {
                if (!settlementConfigParameter && !backofficeUtilService.isOrderVoided(order)
                    && !backofficeUtilService.isOrderAmountCaptured(order) && !isOrderCaptureOrPartialCapture(order)) {
                    isVoidPossible = true;
                }
            } else if (!backofficeUtilService.isOrderVoided(order) && !backofficeUtilService.isOrderAmountCaptured(order)) {
                if (BraintreeConstants.BRAINTREE_CREDITCARD_PAYMENT.equals(paymentProvider)
                    || BraintreeConstants.APPLE_PAY_PAYMENT.equals(paymentProvider)) {
                    isVoidPossible = false;
                } else if (BraintreeConstants.PAYPAL_INTENT_SALE.equalsIgnoreCase(intent)) {
                    isVoidPossible = false;
                } else if (BraintreeConstants.PAYPAL_INTENT_AUTHORIZE.equalsIgnoreCase(intent)) {
                    isVoidPossible = checkOrderAndSetVoidPossible(order, isVoidPossible);
                } else if (BraintreeConstants.PAYPAL_INTENT_ORDER.equalsIgnoreCase(intent)) {
                    isVoidPossible = isVoidAuthorizationPossible(order);
                } else {
                    LOG.warn("Order was placed with incorrect intent = '" + intent);
                    isVoidPossible = false;
                }
            }
        }
        return isVoidPossible;
    }

    private boolean checkOrderAndSetVoidPossible(OrderModel order, boolean isVoidPossible) {
        if (!isOrderCaptureOrPartialCapture(order)) {
            isVoidPossible = isVoidAuthorizationPossible(order);
        }
        return isVoidPossible;
    }

    private boolean isVenmoAccount(final String paymentProvider) {
        return BraintreeConstants.VENMO_CHECKOUT.equals(paymentProvider);
    }

    @Override
    public List<PaymentTransactionEntryModel> getVoidableTransactions(final OrderModel order) {
        List<PaymentTransactionEntryModel> result = new ArrayList<>();

        for (PaymentTransactionModel transaction : order.getPaymentTransactions()) {
            for (PaymentTransactionEntryModel paymentEntry : transaction.getEntries()) {
                if (TransactionStatus.ACCEPTED.name().equals(paymentEntry.getTransactionStatus()) &&
                    (PaymentTransactionType.AUTHORIZATION.equals(paymentEntry.getType()) || PaymentTransactionType.ORDER
                        .equals(paymentEntry.getType()))) {
                    result.add(paymentEntry);
                }
            }
        }

        return result;
    }

    @Override
    public PaymentTransactionEntryData executeVoid(final PaymentTransactionEntryModel transaction)
        throws BraintreeErrorException {
        if (transaction.getType().equals(PaymentTransactionType.ORDER)) {
            BrainTreeResponseResultData resultData = deletePaymentMethod(
                (BrainTreePaymentInfoModel) transaction.getPaymentTransaction().getInfo(), transaction);
            if (resultData.isSuccess()) {
                return handleSuccessfulVoid(transaction);
            } else {
                LOG.error("Error, message: " + resultData.getErrorMessage());
                throw new BraintreeErrorException(resultData.getErrorMessage(), resultData.getTransactionId());
            }
        } else {
            final VoidRequest voidRequest = new VoidRequest("-not-used-", transaction.getRequestId(), StringUtils.EMPTY,
                StringUtils.EMPTY);
            final BrainTreeVoidResult voidResult = brainTreePaymentService.voidTransaction(voidRequest);
            brainTreeTransactionService.addTransactionLog(transaction.getPaymentTransaction(), voidResult,
                BRAINTREE_INTERACTION_TYPE_VOID);
            if (voidResult.isSuccess()) {
                return handleSuccessfulVoid(transaction);
            } else {
                LOG.error("Error, message: " + voidResult.getErrorMessage());
                throw new BraintreeErrorException(voidResult.getErrorMessage(), voidResult.getTransactionId());
            }
        }
    }

    private PaymentTransactionEntryData handleSuccessfulVoid(final PaymentTransactionEntryModel transaction) {
        OrderModel order = (OrderModel) transaction.getPaymentTransaction().getOrder();
        transaction.setTransactionStatus(Transaction.Status.VOIDED.name());
        order.setStatus(OrderStatus.CANCELLED);
        modelService.save(transaction);
        modelService.refresh(transaction);
        modelService.save(order);
        modelService.refresh(order);
        return ordermanagementPaymentTransactionEntryConverter.convert(transaction);
    }

    /**
     * This method is used to get possible for void authorization
     *
     * @param order order
     * @return possible for void authorization
     */
    public boolean isVoidAuthorizationPossible(final OrderModel order) {
        if (order != null) {
            return isSuccessfulTransactionPresent(order, PaymentTransactionType.AUTHORIZATION)
                || isSuccessfulTransactionPresent(order, PaymentTransactionType.ORDER);
        }
        return false;
    }

    private boolean isSuccessfulTransactionPresent(final OrderModel order,
        final PaymentTransactionType transactionType) {
        for (PaymentTransactionModel paymentTransaction : order.getPaymentTransactions()) {
            for (PaymentTransactionEntryModel transactionEntry : paymentTransaction.getEntries()) {
                if (transactionType.equals(transactionEntry.getType()) && transactionEntry.getTransactionStatusDetails()
                    .startsWith("SUCCESFULL")
                    && TransactionStatus.ACCEPTED.name().equals(transactionEntry.getTransactionStatus())) {
                    return true;
                }
            }
        }
        return false;
    }

    private boolean isOrderCaptureOrPartialCapture(OrderModel order) {
        boolean isOrderCapture = false;
        for (PaymentTransactionModel paymentTransaction : order.getPaymentTransactions()) {
            for (PaymentTransactionEntryModel transactionEntry : paymentTransaction.getEntries()) {
                if (PaymentTransactionType.PARTIAL_CAPTURE.equals(transactionEntry.getType())
                        || PaymentTransactionType.CAPTURE.equals(transactionEntry.getType())) {
                    isOrderCapture = true;
                }
            }
        }
        return isOrderCapture;
    }

    @Override
    public BrainTreePaymentMethodResult deletePaymentMethod(final BrainTreePaymentInfoModel paymentInfo) {
        final UserModel user = userService.getCurrentUser();
        final String braintreeCustomerId = user.getUid();
        if (braintreeCustomerId != null) {
            final BrainTreeDeletePaymentMethodRequest request = new BrainTreeDeletePaymentMethodRequest(
                braintreeCustomerId,
                paymentInfo.getPaymentMethodToken());
            final BrainTreePaymentMethodResult result = brainTreePaymentService.deletePaymentMethod(request);
            if (result.isSuccess()) {
                paymentInfoService.remove(paymentInfo.getCustomerId(), paymentInfo.getPaymentMethodToken());
            }
            return result;
        }
        return null;
    }

    @Override
    public BrainTreeResponseResultData deletePaymentMethod(BrainTreePaymentInfoModel paymentInfoModel,
        PaymentTransactionEntryModel paymentTransactionEntryModel) {
        BrainTreePaymentMethodResult brainTreePaymentMethodResult = deletePaymentMethod(paymentInfoModel);
        brainTreeTransactionService
            .addTransactionLogForVoidIntentOrder(paymentTransactionEntryModel.getPaymentTransaction(),
                brainTreePaymentMethodResult);
        BrainTreeResponseResultData brainTreeResponseResultData = new BrainTreeResponseResultData();
        brainTreeResponseResultData.setErrorCode(brainTreePaymentMethodResult.getErrorCode());
        brainTreeResponseResultData.setErrorMessage(brainTreePaymentMethodResult.getErrorMessage());
        brainTreeResponseResultData.setSuccess(brainTreePaymentMethodResult.isSuccess());
        paymentTransactionEntryModel.setTransactionStatus(Transaction.Status.VOIDED.name());
        paymentTransactionEntryModel.setTransactionStatusDetails(TransactionStatusDetails.REVIEW_NEEDED.name());
        paymentTransactionEntryModel.getPaymentTransaction().getEntries().get(0)
            .setTransactionStatus(Transaction.Status.VOIDED.name());
        paymentTransactionEntryModel.getPaymentTransaction().getEntries().get(0)
            .setTransactionStatusDetails(TransactionStatusDetails.REVIEW_NEEDED.name());
        modelService.saveAll(paymentTransactionEntryModel);
        return brainTreeResponseResultData;
    }

}
