package com.braintree.facade.backoffice.impl;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants;
import com.braintree.facade.backoffice.BraintreeBackofficeAuthorizeFacade;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.service.BraintreeOrderBackofficeUtilService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordermanagementfacades.payment.data.PaymentTransactionEntryData;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import java.math.BigDecimal;
import java.util.Map;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

/**
 * This class is a default implementation of the BraintreeBackofficeAuthorizeFacade interface
 */
public class DefaultBraintreeBackofficeAuthorizeFacade implements BraintreeBackofficeAuthorizeFacade {

    private final static Logger LOG = Logger.getLogger(DefaultBraintreeBackofficeAuthorizeFacade.class);

    @Autowired
    private ModelService modelService;
    @Autowired
    private BrainTreeConfigService brainTreeConfigService;
    @Autowired
    private BrainTreeTransactionService brainTreeTransactionService;
    @Autowired
    private BraintreeOrderBackofficeUtilService backofficeUtilService;
    @Autowired
    @Qualifier("ordermanagementPaymentTransactionEntryConverter")
    private Converter<PaymentTransactionEntryModel, PaymentTransactionEntryData> ordermanagementPaymentTransactionEntryConverter;

    @Override
    public boolean isAuthorizePossible(final OrderModel order) {
        if (null == order) {
            LOG.warn("order: " + order);
            return false;
        }
        LOG.info("isAuthorizePossible, order.getTotalPrice: " + order.getTotalPrice());

        modelService.refresh(order);
        boolean isAuthorizePossible = false;

        String paymentProvider = ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPaymentProvider();

        if (!backofficeUtilService.isOrderVoided(order) && !backofficeUtilService.isOrderAmountCaptured(order)
            && !(backofficeUtilService.paidByCard(order))) {
            isAuthorizePossible =
                isIntentOrder(order) && !isHostedFields(paymentProvider) && !isApplePay(paymentProvider)
                    && !isVenmoAccount(paymentProvider);
        }

        return isAuthorizePossible;
    }

    @Override
    public PaymentTransactionEntryData authorizePayment(final AbstractOrderModel cart, Map<String, String> customFields,
        BigDecimal totalAmount) {
        return ordermanagementPaymentTransactionEntryConverter
            .convert(brainTreeTransactionService.createAuthorizationTransaction(cart, customFields, totalAmount));
    }

    private boolean isIntentOrder(final OrderModel order) {
        if (order.getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            String intent = ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPayPalIntent();
            return BraintreeConstants.PAYPAL_INTENT_ORDER.equalsIgnoreCase(intent);
        } else {
            return false;
        }
    }

    private boolean isApplePay(final String paymentProvider) {
        return BraintreeConstants.APPLE_PAY_PAYMENT.equals(paymentProvider);
    }

    private boolean isHostedFields(final String paymentProvider) {
        return BraintreeConstants.BRAINTREE_CREDITCARD_PAYMENT.equals(paymentProvider);
    }

    private boolean isVenmoAccount(final String paymentProvider) {
        return BraintreeConstants.VENMO_CHECKOUT.equals(paymentProvider);
    }

    @Override
    public BigDecimal calculateTotalAmount(final AbstractOrderModel cart) {

        return brainTreeTransactionService.calculateTotalAmount(cart);
    }
}
