/**
 *
 */
package com.braintree.controllers;

import com.braintree.email.emailServices.BraintreeLocalPaymentReversedEmailEventService;
import com.braintree.facade.BrainTreeB2BCheckoutFacade;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.facade.impl.DefaultBrainTreePaymentFacade;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintree.replenishment.service.BrainTreeReplenishmentService;
import com.braintreegateway.RevokedPaymentMethodMetadata;
import com.braintreegateway.WebhookNotification;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.site.BaseSiteService;
import java.util.Optional;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.log4j.Logger;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;

@Controller
@RequestMapping(value = "/paypal/webhook")
public class BraintreeB2BWebhookListener {

    private static final Logger LOG = Logger.getLogger(BraintreeB2BWebhookListener.class);


    @Resource(name = "brainTreeCheckoutFacade")
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;

    @Resource(name = "brainTreePaymentFacadeImpl")
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;

    @Resource(name = "brainTreeB2BCheckoutFacade")
    private BrainTreeB2BCheckoutFacade brainTreeB2BCheckoutFacade;

    @Resource(name = "baseSiteService")
    private BaseSiteService baseSiteService;

    @Resource(name = "paymentInfoService")
    private BraintreePaymentInfoService paymentInfoService;

    @Resource(name = "brainTreeReplenishmentService")
    private BrainTreeReplenishmentService replenishmentService;

    @Resource(name = "localPaymentReversedEmailEventService")
    private BraintreeLocalPaymentReversedEmailEventService localPaymentReversedEmailEventService;

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
        "ROLE_ANONYMOUS"})
    @PostMapping(value = "/response")
    @ResponseStatus(HttpStatus.CREATED)
    public ResponseEntity<String> doHandleBillingAgreemenHopResponse(final HttpServletRequest request,
        final HttpServletResponse response) throws InvalidCartException {
        final WebhookNotification webhookNotification = brainTreePaymentFacade.getWebhookNotification(request);
        final String paymentId = webhookNotification.getLocalPaymentCompleted().getPaymentId();
        final String paymentMethodNonce = webhookNotification.getLocalPaymentCompleted().getPaymentMethodNonce();

        CartModel cart = brainTreePaymentFacade.getCartByPaymentId(paymentId);

        if (cart != null) {
            baseSiteService.setCurrentBaseSite(cart.getSite(), true);
            brainTreePaymentFacade.updateLocalPaymentMethodSubscription(paymentMethodNonce, cart);

            boolean authorized = brainTreeCheckoutFacade.authorizePayment(cart);
            if (authorized) {
                final OrderData orderData = brainTreeB2BCheckoutFacade.placeOrderByCart(cart);
                LOG.info(orderData.getCode() + " was placed due to the webhook  : " + webhookNotification.getKind());
            }
        }

        return ResponseEntity.ok("200");
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
            "ROLE_ANONYMOUS"})
    @PostMapping(value = "/localPayment/reversed")
    @ResponseStatus(HttpStatus.CREATED)
    public ResponseEntity<String> handleLocalPaymentRefund(final HttpServletRequest request,
                                                           final HttpServletResponse response) throws InvalidCartException {
        final WebhookNotification webhookNotification = brainTreePaymentFacade.getWebhookNotification(request);

        if (WebhookNotification.Kind.LOCAL_PAYMENT_REVERSED.equals(webhookNotification.getKind())) {
            LOG.info(webhookNotification);
            String paymentId = webhookNotification.getLocalPaymentReversed().getPaymentId();

            CartModel cartByPaymentId = brainTreePaymentFacade.getCartByPaymentId(paymentId);

            if(cartByPaymentId != null){
                baseSiteService.setCurrentBaseSite(cartByPaymentId.getSite(), true);
                brainTreeCheckoutFacade.processLocalPaymentReversedWebhook(cartByPaymentId);
                OrderModel orderModel = brainTreeB2BCheckoutFacade.placeOrderForLocalPaymentReversed(cartByPaymentId);

                localPaymentReversedEmailEventService
                        .sendBraintreeLocalPaymentReversedEmailEvent((CustomerModel) cartByPaymentId.getUser(), orderModel);

            }
        }
        return ResponseEntity.ok("200");
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT",
        "ROLE_ANONYMOUS"})
    @PostMapping(value = "/payment-method")
    @ResponseStatus(HttpStatus.CREATED)
    public ResponseEntity<String> doHandlePaymentMethodWebhook(final HttpServletRequest request,
        final HttpServletResponse response) {
        final WebhookNotification webhookNotification = brainTreePaymentFacade.getWebhookNotification(request);
        final RevokedPaymentMethodMetadata revokedPaymentMethodMetadata = webhookNotification
            .getRevokedPaymentMethodMetadata();
        if (revokedPaymentMethodMetadata != null) {
            String paymentMethodToken = Optional.of(revokedPaymentMethodMetadata.getToken()).get();
            String customerId = Optional.of(revokedPaymentMethodMetadata.getCustomerId()).get();
            replenishmentService.disableReplenishmentCronJobByPaymentMethodToken(paymentMethodToken);
            paymentInfoService.disable(customerId, paymentMethodToken);
            LOG.info("Payment method with payment method token: " + paymentMethodToken +
                " and braintree customer id: " + customerId + " has been disabled by payment method webhook");
        }
        return ResponseEntity.ok("200");
    }

}
