/**
 *
 */
package com.braintree.controllers;

import static com.braintree.constants.BraintreeConstants.BRAINTREE_PAGE_TYPE_OTHER;
import static com.braintree.constants.BraintreeConstants.PAYPAL_FUNDING_SOURCE;
import static com.braintree.constants.BraintreeConstants.PAYPAL_INTENT_ORDER;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.converters.BrainTreePayPalAddressConverterForOCC;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintree.enums.BraintreePageType;
import com.braintree.exceptions.BrainTreeDeliveryAddressNotFoundException;
import com.braintree.facade.BrainTreeUserFacade;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.facade.impl.DefaultBrainTreePaymentFacade;
import com.braintree.facade.impl.DefaultBrainTreeUserFacade;
import com.braintree.hybris.data.BrainTreeSubscriptionInfoData;
import com.braintree.hybris.data.BraintreePaymentMethodNonceData;
import com.braintree.hybris.data.ws.GooglePayAddressWsDTO;
import com.braintree.hybris.data.ws.ApplePayAddressWsDTO;
import com.braintree.hybris.data.ws.GooglePayPaymentRequestWsDTO;
import com.braintree.hybris.data.ws.PayPalAddressWsDTO;
import com.braintree.hybris.data.ws.PayPalDetailsWsDTO;
import com.braintree.hybris.data.ws.PayPalPaymentRequestWsDTO;
import com.braintree.hybris.data.ws.SrcAddressWsDTO;
import com.braintree.hybris.data.ws.SrcPaymentRequestWsDTO;
import com.braintree.hybris.data.ws.VenmoPaymentRequestWsDTO;
import com.braintree.hybris.data.ws.BraintreeCreditCardPaymentDetailsWsDTO;
import com.braintree.hybris.data.ws.ApplePayPaymentRequestWsDTO;
import com.braintree.hybris.data.ws.LocalPaymentRequestWsDTO;
import com.braintree.hybris.data.BraintreeTokenizedCreditCardData;
import com.braintree.transaction.service.impl.DefaultBrainTreeTransactionService;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoDatas;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.commercewebservicescommons.dto.order.OrderWsDTO;
import de.hybris.platform.commercewebservicescommons.dto.order.PaymentDetailsListWsDTO;
import de.hybris.platform.commercewebservicescommons.dto.order.PaymentDetailsWsDTO;
import de.hybris.platform.commercewebservicescommons.errors.exceptions.PaymentAuthorizationException;
import de.hybris.platform.commercewebservicescommons.errors.exceptions.RequestParameterException;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.webservicescommons.cache.CacheControl;
import de.hybris.platform.webservicescommons.cache.CacheControlDirective;
import de.hybris.platform.webservicescommons.errors.exceptions.NotFoundException;
import de.hybris.platform.webservicescommons.errors.exceptions.WebserviceValidationException;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdAndUserIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiBaseSiteIdUserIdAndCartIdParam;
import de.hybris.platform.webservicescommons.swagger.ApiFieldsParam;
import de.hybris.platform.ycommercewebservices.exceptions.InvalidPaymentInfoException;
import de.hybris.platform.ycommercewebservices.v2.controller.BaseCommerceController;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.annotation.Secured;
import org.springframework.validation.BeanPropertyBindingResult;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

@RestController
@RequestMapping(value = "/{baseSiteId}/users/{userId}")
@CacheControl(directive = CacheControlDirective.NO_CACHE)
@Api(tags = "Braintree Payment Details")
public class BraintreeCreatePaymentDetailsController extends BaseCommerceController
{

    @Resource(name = "brainTreePaymentFacadeImpl")
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;

    @Resource(name = "brainTreeUserFacade")
    private DefaultBrainTreeUserFacade brainTreeUserFacade;

    @Resource(name = "brainTreeCheckoutFacade")
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;

    @Resource(name = "brainTreeConfigService")
    private BrainTreeConfigService brainTreeConfigService;

    @Resource(name = "payPalPaymentRequestValidator")
    private Validator payPalPaymentRequestValidator;

    @Resource(name = "payPalDetailsValidator")
    private Validator payPalDetailsValidator;

    @Resource(name = "googlePayPaymentRequestValidator")
    private Validator googlePayPaymentRequestValidator;

    @Resource(name = "srcPaymentRequestValidator")
    private Validator srcPaymentRequestValidator;

    @Resource(name = "applePayPaymentRequestValidator")
    private Validator applePayPaymentRequestValidator;

    @Resource(name = "creditCardDataValidator")
    private Validator creditCardDataValidator;

    @Resource(name = "venmoRequestValidator")
    private Validator venmoRequestValidator;

    @Resource(name = "localPaymentRequestValidator")
    private Validator localPaymentRequestValidator;

    @Resource(name = "payPalOCCAddressConverter")
    private BrainTreePayPalAddressConverterForOCC payPalOCCAddressConverter;

    @Resource(name = "googlePayAddressConverter")
    private Converter<GooglePayAddressWsDTO, AddressData> googlePayAddressConverter;

    @Resource(name = "srcAddressConverter")
    private Converter<SrcAddressWsDTO, AddressData> srcAddressConverter;

    @Resource(name = "applePayAddressConverter")
    private Converter<ApplePayAddressWsDTO, AddressData> applePayAddressConverter;

    @Resource(name = "userFacade")
    protected BrainTreeUserFacade userFacade;

    @Resource(name = "brainTreeTransactionService")
    private DefaultBrainTreeTransactionService brainTreeTransactionService;


    private static final String ADDRESS_MAPPING = "firstName,lastName,titleCode,phone,cellphone,line1,line2,town,postalCode,region(isocode),district,country(isocode),defaultAddress";
    private static final String ADDRESS_DOES_NOT_EXIST = "Address with given id: '%s' doesn't exist or belong to another user";
    private static final String DELIVERY_ADDRESS_NOT_SET = "Delivery address is not set ";
    private static final String OBJECT_NAME_SELECTED_ADDRESS_ID = "selectedAddressCode";
    private static final Logger LOG = Logger.getLogger(BraintreeCreatePaymentDetailsController.class);

    @Secured({ "ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @PostMapping(value = "/braintree/paymentInfo/payPal")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "addPayPalPayment", value = "Save new PayPal payment info", notes = "Adding PayPal payment info into customer wallet")
    @ApiBaseSiteIdAndUserIdParam
    public PaymentDetailsWsDTO addPayPalPayment(@ApiParam(value = "Request body parameter that contains" +
            "payPalPaymentResponse(payment method nonce (nonce), payment type (type), email, " +
            "countryCode, payerId, phone, firstName, lastName " +
            "and the billing address (First line of the address, " +
            "billingAddress.lastName, billingAddress.titleCode, billingAddress.country.isocode, " +
            "\n\nThe DTO is in XML or .json format.", required = true) @RequestBody final PayPalPaymentRequestWsDTO payPalRequest,
                                                //NOSONAR
            @ApiParam(value = "Defines if payment info goes through credit flow") @RequestParam(defaultValue = "false") boolean credit,
            @ApiParam(value = "Defines if payment info should be saved") @RequestParam(defaultValue = "false") boolean shouldBeSaved,
            @ApiParam(value = "Data collected from client.") @RequestParam(required = false) final String deviceData,
            @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.",
                    allowableValues = "BASIC, DEFAULT, FULL") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields,
            @ApiParam(value = "Funding source. Detect type of paypal payment")
            @RequestParam(defaultValue = PAYPAL_FUNDING_SOURCE) final String fundingSource,
            @RequestParam(defaultValue = BRAINTREE_PAGE_TYPE_OTHER) final String pageType){
        validate(payPalRequest, "payPalRequest", payPalPaymentRequestValidator);
        validateBraintreePaymentDetailsParam(shouldBeSaved, credit);
        final String payPalEmail = payPalRequest.getDetails().getEmail();
        if(PAYPAL_INTENT_ORDER.equalsIgnoreCase(brainTreeConfigService.getIntent())) {
            shouldBeSaved = false;
        }
        final BrainTreeSubscriptionInfoData paymentInfoData = brainTreePaymentFacade
                .buildSubscriptionInfo(payPalRequest.getNonce(), payPalRequest.getType(), Boolean.FALSE,
                        shouldBeSaved, payPalEmail);

        paymentInfoData.setPaypalFundingSource(fundingSource);

        if (deviceData != null)
        {
            paymentInfoData.setDeviceData(deviceData);
        }

        final PayPalDetailsWsDTO brainTreeDetails = payPalRequest.getDetails();
        PayPalAddressWsDTO billingAddress = brainTreeDetails.getBillingAddress();
        AddressData addressData;
        if (billingAddress == null && brainTreeCheckoutFacade.getCheckoutCart().getDeliveryAddress() != null) {
            addressData = brainTreeCheckoutFacade.getCheckoutCart().getDeliveryAddress();
        } else if(billingAddress == null) {
            addressData = payPalOCCAddressConverter.convert(brainTreeDetails.getShippingAddress());
        } else {
            addressData = payPalOCCAddressConverter.convert(brainTreeDetails.getBillingAddress());
        }

        if (StringUtils.isBlank(brainTreeDetails.getFirstName()) && StringUtils.isNotBlank(addressData.getFirstName())) {
            brainTreeDetails.setFirstName(addressData.getFirstName());
            brainTreeDetails.setLastName(addressData.getLastName());
            brainTreeDetails.setCountryCode(addressData.getPostalCode());
        }

        validate(brainTreeDetails, "brainTreeDetails", payPalDetailsValidator);

        if (StringUtils.isBlank(addressData.getFirstName()))
        {
            addressData.setFirstName(brainTreeDetails.getFirstName());
            addressData.setLastName(brainTreeDetails.getLastName());
            addressData.setPhone(brainTreeDetails.getPhone());
        }
        addressData.setEmail(payPalEmail);
        paymentInfoData.setAddressData(addressData);

        return getDataMapper().map(brainTreePaymentFacade
                            .completeCreateSubscription(paymentInfoData, credit, convertPageTypeValue(pageType)), PaymentDetailsWsDTO.class, fields);
    }

    @Secured({ "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS" })
    @PostMapping(value = "/braintree/paymentInfo/googlePay")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "addGooglePayPaymentInfo", value = "Save new GooglePay payment info",
            notes = "Adding GooglePay payment info into customer wallet")
    @ApiBaseSiteIdAndUserIdParam
    public PaymentDetailsWsDTO addGooglePayPaymentInfo(@ApiParam(value = "Request body parameter that contains details such as  " +
            "payment method nonce (nonce), payment type (type), email, and the billing address " +
            "\n\nThe DTO is in XML or .json format.", required = true) @RequestBody final GooglePayPaymentRequestWsDTO googlePayRequest,//NOSONAR
            @ApiParam(value = "Defines if payment info should be saved") @RequestParam(defaultValue = "false") boolean shouldBeSaved,
            @ApiParam(value = "Data collected from client.") @RequestParam(required = false) final String deviceData,
            @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.",
                    allowableValues = "BASIC, DEFAULT, FULL") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields)
    {
        validate(googlePayRequest, "googlePayRequest", googlePayPaymentRequestValidator);
        validateBraintreePaymentDetailsParam(shouldBeSaved, Boolean.FALSE);
        final String googlePayEmail = googlePayRequest.getEmail();
        final BrainTreeSubscriptionInfoData paymentInfoData = brainTreePaymentFacade
                .buildSubscriptionInfo(googlePayRequest.getNonce(), googlePayRequest.getType(), Boolean.FALSE,
                        shouldBeSaved, googlePayEmail);
        paymentInfoData.setCardType(googlePayRequest.getCardType());
        paymentInfoData.setCardNumber(googlePayRequest.getLastFour());

        if (deviceData != null)
        {
            paymentInfoData.setDeviceData(deviceData);
        }
        final AddressData addressData = googlePayAddressConverter.convert(googlePayRequest.getBillingAddress());
        addressData.setEmail(googlePayEmail);
        paymentInfoData.setAddressData(addressData);

        return getDataMapper()
                    .map(brainTreePaymentFacade
                            .completeCreateSubscription(paymentInfoData, false, BraintreePageType.OTHER), PaymentDetailsWsDTO.class, fields);
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @PostMapping(value = "/braintree/paymentInfo/srcPayment")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "addSrcPaymentInfo", value = "Save new secure remote commerce payment info",
        notes = "Adding secure remote commerce payment info into customer wallet")
    @ApiBaseSiteIdAndUserIdParam
    public PaymentDetailsWsDTO addSrcPaymentInfo(
        @ApiParam(value = "Request body parameter that contains details such as  " +
            "payment method nonce (nonce), payment type (type), email, and the billing address " +
            "\n\nThe DTO is in XML or .json format.", required = true) @RequestBody final SrcPaymentRequestWsDTO srcRequest,
        @ApiParam(value = "Defines if payment info should be saved") @RequestParam(defaultValue = "false") boolean shouldBeSaved,
        @ApiParam(value = "Data collected from client.") @RequestParam(required = false) final String deviceData,
        @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.",
            allowableValues = "BASIC, DEFAULT, FULL") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields) {
        validate(srcRequest, "srcRequest", srcPaymentRequestValidator);
        validateBraintreePaymentDetailsParam(shouldBeSaved, Boolean.FALSE);
        final String payerEmail = srcRequest.getUserEmail();

        BrainTreeSubscriptionInfoData subscriptionInfo = brainTreePaymentFacade
            .buildSubscriptionInfoForSrc(srcRequest.getNonce(), srcRequest.getDetails().getExpirationMonth(),
                srcRequest.getDetails().getExpirationYear(), srcRequest.getType(), srcRequest.getDetails().getLastFour(),
                srcRequest.getDetails().getCardType(), payerEmail, shouldBeSaved);

        if (deviceData != null) {
            subscriptionInfo.setDeviceData(deviceData);
        }
        final AddressData addressData = srcAddressConverter.convert(srcRequest.getBillingAddress());
        if (addressData != null) {
            addressData.setEmail(payerEmail);
            subscriptionInfo.setAddressData(addressData);
        }
        return getDataMapper()
            .map(brainTreePaymentFacade
                .completeCreateSubscription(subscriptionInfo, false, BraintreePageType.OTHER), PaymentDetailsWsDTO.class, fields);
    }

    @Secured({ "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS" })
    @PostMapping(value = "/braintree/paymentInfo/creditCard")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "createCardPaymentInfo", value = "Save new credit card payment .",
            notes = "Defines the details of a new credit card payment option and add into customer wallet.")
    @ApiBaseSiteIdAndUserIdParam
    public PaymentDetailsWsDTO addCreditCardPaymentInfo(
            @ApiParam(value = "Defines if payment info should be saved") @RequestParam(defaultValue = "false") boolean shouldBeSaved,
            @ApiParam (value = "Billing address code. The address code may be empty if you provide a billing address in request") @RequestParam(value = "selectedAddressCode") final String selectedAddressCode,
            @RequestParam(required = false) String deviceData,
            @ApiParam(value =
                    "Request body parameter that contains details such as the name on the card (accountHolderName), the card number (cardNumber), the card type (cardType.code), "
                            + "the month of the expiry date (expiryMonth), the year of the expiry date (expiryYear), whether the payment details should be saved (saved), whether the payment details "
                            + "should be set as default (defaultPaymentInfo), and the billing address (billingAddress.firstName, billingAddress.lastName, billingAddress.titleCode, billingAddress.country.isocode, "
                            + "billingAddress.line1, billingAddress.line2, billingAddress.town, billingAddress.postalCode, billingAddress.region.isocode)\n\nThe DTO is in XML or .json format.", required = true) @RequestBody final BraintreeCreditCardPaymentDetailsWsDTO paymentDetails,
            @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.", allowableValues = "BASIC, DEFAULT, FULL")
            @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields)
            throws InvalidPaymentInfoException
    {
        String cardHolderName = paymentDetails.getTokenizedCardData().getDetails().getCardholderName();
        validateBraintreePaymentDetailsParam(shouldBeSaved, Boolean.FALSE);
        if (StringUtils.isBlank(cardHolderName)){
            throw new InvalidPaymentInfoException(paymentDetails.getId());
        }

        validate(paymentDetails.getTokenizedCardData(), "tokenizedCardData", creditCardDataValidator);

        final String creditCardCopiedfields = "nonce,details(cardholderName,expirationMonth,expirationYear,bin,cardType," +
                "lastFour,lastTwo),description,type,liabilityShifted";
        BraintreeTokenizedCreditCardData tokenizedCreditCardData = getDataMapper()
                .map(paymentDetails.getTokenizedCardData(), BraintreeTokenizedCreditCardData.class, creditCardCopiedfields);

        BrainTreeSubscriptionInfoData subscriptionInfo = brainTreePaymentFacade.buildSubscriptionInfo(tokenizedCreditCardData.getNonce(), tokenizedCreditCardData.getType(),
                tokenizedCreditCardData.getDetails().getLastFour(), tokenizedCreditCardData.getDetails().getCardType(),
                tokenizedCreditCardData.getDetails().getExpirationMonth(), tokenizedCreditCardData.getDetails().getExpirationYear(),
                tokenizedCreditCardData.getLiabilityShifted(), Boolean.FALSE, shouldBeSaved, cardHolderName);

        if (deviceData != null)
        {
            subscriptionInfo.setDeviceData(deviceData);
        }
        final AddressData addressData;
        if (StringUtils.isNotEmpty(selectedAddressCode))
        {
            addressData = userFacade.getAddressForCode(selectedAddressCode);
        }
        else
        {
            addressData = getDataMapper().map(paymentDetails.getBillingAddress(), AddressData.class, ADDRESS_MAPPING);
        }
        if (addressData == null)
        {
            throw new RequestParameterException(String.format(ADDRESS_DOES_NOT_EXIST, sanitize(selectedAddressCode)),
                    RequestParameterException.INVALID, OBJECT_NAME_SELECTED_ADDRESS_ID);
        }
        addressData.setBillingAddress(true);
        subscriptionInfo.setAddressData(addressData);

        if (!brainTreeConfigService.get3dSecureConfiguration() && shouldBeSaved)
        {
            return getDataMapper()
                    .map(userFacade.addPaymentMethodData(subscriptionInfo), PaymentDetailsWsDTO.class, fields);
        }
        else
        {
            return getDataMapper()
                    .map(brainTreePaymentFacade
                            .completeCreateSubscription(subscriptionInfo), PaymentDetailsWsDTO.class, fields);
        }
    }

    @Secured({ "ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_ANONYMOUS" })
    @PostMapping(value = "/braintree/paymentInfo/venmo")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "createVenmoPaymentDetails", value = "Save new Venmo payment info.",   notes = "Adding Venmo payment info into customer wallet")
    @ApiBaseSiteIdAndUserIdParam
    public PaymentDetailsWsDTO createVenmoPaymentInfo(
            @ApiParam(value = "Data collected from client.") @RequestParam(required = false) String deviceData,
            @ApiParam(value = "Defines if payment info should be saved") @RequestParam(defaultValue = "false") boolean shouldBeSaved,
            @ApiParam(value = "Code of selected address") @RequestParam(value = "selectedAddressCode") final String selectedAddressCode,
            @ApiParam(value =
                    "Request body parameter that contains details such as payment method nonce (nonce), payment type (type)\n\nThe DTO is in XML or .json format.", required = true) @RequestBody final VenmoPaymentRequestWsDTO venmoRequest,
            @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.", allowableValues = "BASIC, DEFAULT, FULL")
            @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields)
    {
        validate(venmoRequest, "venmoRequest", venmoRequestValidator);
        validateBraintreePaymentDetailsParam(shouldBeSaved, Boolean.FALSE);
        final AddressData addressData = userFacade.getAddressForCode(selectedAddressCode);
        if (addressData == null)
        {
            throw new RequestParameterException(String.format(ADDRESS_DOES_NOT_EXIST, sanitize(selectedAddressCode)),
                    RequestParameterException.INVALID, OBJECT_NAME_SELECTED_ADDRESS_ID);
        }
        BrainTreeSubscriptionInfoData subscriptionInfo = brainTreePaymentFacade.buildVenmoSubscriptionInfo(venmoRequest.getNonce(),
                BrainTreePaymentMethod.VENMOACCOUNT.toString(), Boolean.FALSE, shouldBeSaved, venmoRequest.getUsername(),
                addressData);

        if (deviceData != null){
            subscriptionInfo.setDeviceData(deviceData);
        }
        return getDataMapper()
                    .map(brainTreePaymentFacade.completeCreateSubscription(subscriptionInfo), PaymentDetailsWsDTO.class, fields);
    }

    @Secured({ "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS" })
    @PostMapping(value = "/braintree/paymentInfo/applePay")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "addApplePayPaymentInfo", value = "Save new ApplePay payment info",
            notes = "Adding ApplePay payment info into customer wallet")
    @ApiBaseSiteIdAndUserIdParam
    public PaymentDetailsWsDTO addApplePayPaymentInfo(@ApiParam(value = "Request body parameter that contains details such as  " +
            "payment method nonce (nonce), payment type (type), email, and the billing address " +
            "\n\nThe DTO is in XML or .json format.", required = true) @RequestBody final ApplePayPaymentRequestWsDTO applePayRequest,//NOSONAR
            @ApiParam(value = "Data collected from client.") @RequestParam(required = false) final String deviceData,
            @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.",
                    allowableValues = "BASIC, DEFAULT, FULL") @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields)
    {
        validate(applePayRequest, "applePayRequest", applePayPaymentRequestValidator);

        String applePayEmail = null;
        final AddressData addressData = applePayAddressConverter.convert(applePayRequest.getBillingContact());
        if (applePayRequest.getEmail() != null && addressData != null)
        {
            applePayEmail = applePayRequest.getEmail();
            addressData.setEmail(applePayEmail);
        }

        final BrainTreeSubscriptionInfoData paymentInfoData = brainTreePaymentFacade
                .buildSubscriptionInfo(applePayRequest.getNonce(), applePayRequest.getType(), Boolean.FALSE,
                        Boolean.FALSE, applePayEmail);
        paymentInfoData.setCardType(applePayRequest.getCardType());

        if (deviceData != null)
        {
            paymentInfoData.setDeviceData(deviceData);
        }
        paymentInfoData.setAddressData(addressData);

        return getDataMapper()
                .map(brainTreePaymentFacade.completeCreateSubscription(paymentInfoData), PaymentDetailsWsDTO.class, fields);
    }

    @Secured({ "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT" })
    @PostMapping(value = "/carts/{cartId}/braintree/paymentInfo/localPayment")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(nickname = "createLocalPaymentPaymentDetails", value = "Save new LocalPayment info.", notes = "Adding LocalPayment payment info into customer wallet")
    @ApiBaseSiteIdUserIdAndCartIdParam
    public OrderWsDTO processLocalPayment(
            @ApiParam(value =
                    "Request body parameter that contains details such as payment method nonce (nonce), correlationId, paymentId and local payment details\n\nThe DTO is in XML or .json format.", required = true) @RequestBody final LocalPaymentRequestWsDTO localPaymentRequest,
            @ApiParam(value = "Response configuration. This is the list of fields that should be returned in the response body.", allowableValues = "BASIC, DEFAULT, FULL")
            @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields) throws PaymentAuthorizationException, InvalidCartException {
        validate(localPaymentRequest, "localPaymentRequest", localPaymentRequestValidator);

        final String paymentMethodNonce = localPaymentRequest.getNonce();
        if(brainTreePaymentFacade.getOrderByPaymentId(paymentMethodNonce) != null){
            return getDataMapper().map(brainTreePaymentFacade.getOrderByPaymentId(paymentMethodNonce), OrderWsDTO.class, fields);
        }

        try
        {
            brainTreePaymentFacade.updateLocalPaymentMethodSubscription(localPaymentRequest.getNonce(),
                        localPaymentRequest.getCorrelationId(), localPaymentRequest.getDetails().getEmail());
        }
        catch (BrainTreeDeliveryAddressNotFoundException e)
        {
            LOG.error(DELIVERY_ADDRESS_NOT_SET, e);
            throw new NotFoundException(DELIVERY_ADDRESS_NOT_SET);
        }

        boolean isPaymentAuthorized = brainTreeTransactionService.createAuthorizationTransaction();
        if (!isPaymentAuthorized)
        {
            throw new PaymentAuthorizationException();
        }

        OrderData orderData;

            orderData = getCheckoutFacade().placeOrder();

            LOG.info("Order has been placed, number/code: " + orderData.getCode());
            return getDataMapper().map(orderData, OrderWsDTO.class, fields);
    }

    @Secured({ "ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP" })
    @GetMapping("/braintree")
    @ApiOperation(nickname = "getPaymentDetailsList", value = "Get customer's Braintree credit card payment details list.", notes = "Return customer's Braintree credit card payment details list.")
    @ApiBaseSiteIdAndUserIdParam
    public PaymentDetailsListWsDTO getBraintreePaymentDetailsList(
            @ApiParam(value = "Type of payment details.") @RequestParam(defaultValue = "false") final boolean saved,
            @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields)
    {
        LOG.debug("getBraintreePaymentDetailsList");
        final CCPaymentInfoDatas paymentInfoDataList = new CCPaymentInfoDatas();

        List<CCPaymentInfoData> brainTreeCCPaymentInfos = brainTreeUserFacade.getBrainTreeCCPaymentInfos(saved);

        paymentInfoDataList.setPaymentInfos(brainTreeCCPaymentInfos);

        return getDataMapper().map(paymentInfoDataList, PaymentDetailsListWsDTO.class, fields);
    }

    @Secured({"ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP"})
    @GetMapping("/braintree/{paymentId}")
    @ApiOperation(nickname = "getBraintreePaymentDetailsById", value = "Get customer's Braintree credit card payment details bt ID.", notes = "Return customer's Braintree credit card payment details by ID.")
    @ApiBaseSiteIdAndUserIdParam
    public PaymentDetailsWsDTO getBraintreePaymentDetailsById(
            @ApiParam(value = "Payment method ID") @PathVariable final String paymentId,
            @ApiFieldsParam @RequestParam(defaultValue = DEFAULT_FIELD_SET) final String fields) {
        LOG.debug("getBraintreePaymentDetailsList");

        CCPaymentInfoData paymentInfoData = brainTreeUserFacade.getCCPaymentInfoForCode(paymentId);
        String token = paymentInfoData.getPaymentMethodToken();

        BraintreePaymentMethodNonceData paymentMethodNonce = brainTreePaymentFacade.createPaymentMethodNonce(token);
        PaymentDetailsWsDTO paymentDetails = getDataMapper().map(paymentInfoData, PaymentDetailsWsDTO.class, fields);

        paymentDetails.setPaymentMethodNonce(paymentMethodNonce.getNonce());
        paymentDetails.setBin(paymentMethodNonce.getDetails().getBin());

        return paymentDetails;
    }

    @Secured({ "ROLE_CUSTOMERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_CUSTOMERMANAGERGROUP" })
    @PutMapping("/carts/{cartId}/braintree/choose-cc")
    @ApiOperation(nickname = "doSelectPaymentMethod", value = "Get customer's Braintree credit card payment details list.", notes = "Return customer's Braintree credit card payment details list.")
    @ApiBaseSiteIdAndUserIdParam
    @ResponseStatus(HttpStatus.OK)
    public void doSelectPaymentMethod(
            @ApiParam(value = "Selected payment method id.") @RequestParam(value = "selectedPaymentMethodId", required = true) final String selectedPaymentMethodId,
            @ApiParam(value = "Selected payment method nonce.") @RequestParam(value = "selectedPaymentMethodNonce", required = true) final String selectedPaymentMethodNonce)
            throws InvalidPaymentInfoException
    {
        boolean isSuccess = brainTreeCheckoutFacade.setPaymentDetails(selectedPaymentMethodId, selectedPaymentMethodNonce);
        if (!isSuccess)
        {
            throw new InvalidPaymentInfoException(selectedPaymentMethodId);
        }
    }

    private void validateBraintreePaymentDetailsParam(final boolean shouldBeSaved, final boolean credit)
    {
        Errors shouldBeSavedErrors = new BeanPropertyBindingResult(shouldBeSaved, "shouldBeSaved");
        Errors creditErrors = new BeanPropertyBindingResult(credit, "credit");
        if (!Boolean.parseBoolean(brainTreeConfigService.getStoreInVault()) && shouldBeSaved)
        {
            shouldBeSavedErrors.reject("store.in.vault.exception");
            throw new WebserviceValidationException(shouldBeSavedErrors);
        }
        else if (credit && !brainTreeConfigService.getCreditEnabled())
        {
            creditErrors.reject("credit.exception");
        }
        if (creditErrors.hasErrors())
        {
            throw new WebserviceValidationException(creditErrors);
        }
    }

    private BraintreePageType convertPageTypeValue(String pageType) {
        try {
            return BraintreePageType.valueOf(pageType);
        } catch (Exception e) {
            LOG.error(e);
            return BraintreePageType.OTHER;
        }
    }
}
