/**
 *
 */
package com.braintree.facade;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.constants.BraintreeConstants;
import com.braintree.facade.impl.DefaultBrainTreeCheckoutFacade;
import com.braintree.facade.impl.DefaultBrainTreePaymentFacade;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.google.common.collect.Lists;
import de.hybris.platform.b2bacceleratorfacades.checkout.data.PlaceOrderData;
import de.hybris.platform.b2bacceleratorfacades.exception.EntityValidationException;
import de.hybris.platform.b2bacceleratorfacades.order.data.B2BCommentData;
import de.hybris.platform.b2bacceleratorfacades.order.data.B2BReplenishmentRecurrenceEnum;
import de.hybris.platform.b2bacceleratorfacades.order.data.TriggerData;
import de.hybris.platform.b2bacceleratorfacades.order.impl.DefaultB2BAcceleratorCheckoutFacade;
import de.hybris.platform.b2bacceleratorservices.enums.CheckoutPaymentType;
import de.hybris.platform.commercefacades.order.data.AbstractOrderData;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.InvoicePaymentInfoModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import java.util.Optional;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import static com.braintree.constants.BraintreeConstants.BRAINTREE_PROVIDER_NAME;
import static com.braintree.constants.BraintreeConstants.FAKE_REQUEST_ID;
import static de.hybris.platform.util.localization.Localization.getLocalizedString;


public class BrainTreeB2BCheckoutFacade extends DefaultB2BAcceleratorCheckoutFacade {

    private static final String CART_CHECKOUT_DELIVERYADDRESS_INVALID = "cart.deliveryAddress.invalid";
    private static final String CART_CHECKOUT_DELIVERYMODE_INVALID = "cart.deliveryMode.invalid";
    private static final String CART_CHECKOUT_PAYMENTINFO_EMPTY = "cart.paymentInfo.empty";
    private static final String CART_CHECKOUT_NOT_CALCULATED = "cart.not.calculated";
    private static final String CART_CHECKOUT_TERM_UNCHECKED = "cart.term.unchecked";
    private static final String CART_CHECKOUT_NO_QUOTE_DESCRIPTION = "cart.no.quote.description";
    private static final String CART_CHECKOUT_REPLENISHMENT_NO_STARTDATE = "cart.replenishment.no.startdate";
    private static final String CART_CHECKOUT_REPLENISHMENT_NO_FREQUENCY = "cart.replenishment.no.frequency";
    private static final String CART_CHECKOUT_QUOTE_REQUIREMENTS_NOT_SATISFIED = "cart.quote.requirements.not.satisfied";

    private static final Integer MIN_HYBRIS_API_VERSION_FOR_QUOTE_VALIDATION = Integer.valueOf(6);

    @Resource(name = "brainTreeCheckoutFacade")
    private DefaultBrainTreeCheckoutFacade brainTreeCheckoutFacade;
    @Resource(name = "brainTreeTransactionService")
    private BrainTreeTransactionService brainTreeTransactionService;
    @Resource
    private CartService cartService;
    @Resource(name = "brainTreeConfigService")
    private BrainTreeConfigService brainTreeConfigService;
    @Resource(name = "brainTreePaymentFacadeImpl")
    private DefaultBrainTreePaymentFacade brainTreePaymentFacade;
    @Resource
    private ModelService modelService;

    public void storeIntentToCart() {
        CartModel cart = cartService.getSessionCart();
        if (cart.getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            BrainTreePaymentInfoModel paymentInfo = (BrainTreePaymentInfoModel) cart.getPaymentInfo();
            paymentInfo.setPayPalIntent(brainTreeConfigService.getIntent());
            getModelService().save(paymentInfo);
        } else if (cart.getPaymentInfo() instanceof InvoicePaymentInfoModel) {
            InvoicePaymentInfoModel paymentInfo = (InvoicePaymentInfoModel) cart.getPaymentInfo();
            getModelService().save(paymentInfo);
        }
    }

    public void storeShipsFromPostalCodeToCart(final String shipsFromPostalCode) {
        CartModel cart = cartService.getSessionCart();
        if (cart.getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            BrainTreePaymentInfoModel paymentInfo = (BrainTreePaymentInfoModel) cart.getPaymentInfo();
            paymentInfo.setShipsFromPostalCode(shipsFromPostalCode);
            getModelService().save(paymentInfo);
        } else if (cart.getPaymentInfo() instanceof InvoicePaymentInfoModel) {
            InvoicePaymentInfoModel paymentInfo = (InvoicePaymentInfoModel) cart.getPaymentInfo();
            getModelService().save(paymentInfo);
        }
    }

    public void storeCustomFieldsToCart(final Map<String, String> customFields) {
        CartModel cart = cartService.getSessionCart();
        if (cart.getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            BrainTreePaymentInfoModel paymentInfo = (BrainTreePaymentInfoModel) cart.getPaymentInfo();
            paymentInfo.setCustomFields(customFields);
            getModelService().save(paymentInfo);
            getModelService().save(cart);
        }
    }

    @Override
    public boolean authorizePayment(final String securityCode) {
        final CartModel cart = getCartService().getSessionCart();
        if (CheckoutPaymentType.ACCOUNT.equals(cart.getPaymentType())) {
            return super.authorizePayment(securityCode);
        }
        return brainTreeCheckoutFacade.authorizePayment(securityCode);
    }

    @Override
    protected CCPaymentInfoData getPaymentDetails() {
        return brainTreeCheckoutFacade.getPaymentDetails();
    }

    @Override
    public boolean setPaymentDetails(final String paymentInfoId) {
        return brainTreeCheckoutFacade.setPaymentDetails(paymentInfoId);
    }

    public boolean authorizePaymentForReplenishment() {
        return getBrainTreeTransactionService().createPaymentMethodTokenForOrderReplenishment();
    }

    @Override
    public <T extends AbstractOrderData> T placeOrder(PlaceOrderData placeOrderData) throws InvalidCartException {
        CartModel cart = brainTreeCheckoutFacade.getCartService().getSessionCart();
        String intent = StringUtils.EMPTY;
        if (cart.getPaymentInfo() != null && cart.getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            intent = ((BrainTreePaymentInfoModel) cart.getPaymentInfo()).getPayPalIntent();
        }

        if (placeOrderData.getReplenishmentOrder() != null && placeOrderData.getReplenishmentOrder()) {
            brainTreePaymentFacade.forceUnduplicateCartForReplenishment(cart.getCode());
        }

        if (BraintreeConstants.PAYPAL_INTENT_ORDER.equals(intent)) {

            if (placeOrderData.getTermsCheck() != null) {
                return placeOrderIfIntentOrder(placeOrderData);
            } else {
                return (T) placeOrder();
            }
        }

//		Due to B2B specific: save successful authorization even without actual request
        if (isNoTransactionAuthorized(cart)) {
            createFakeAuthorizationTransaction(cart);
        }

        return super.placeOrder(placeOrderData);
    }

    public OrderData placeOrderByCart(CartModel cartModel) throws InvalidCartException {
        if (cartModel != null) {
            beforePlaceOrder(cartModel);
            final OrderModel orderModel = placeOrder(cartModel);
            afterPlaceOrder(cartModel, orderModel);
            if (orderModel != null) {
                return getOrderConverter().convert(orderModel);
            }
        }
        return null;
    }

    public OrderModel placeOrderForLocalPaymentReversed(CartModel cartModel) throws InvalidCartException {
        if (cartModel != null) {
            beforePlaceOrder(cartModel);
            final OrderModel orderModel = placeOrder(cartModel);
            afterPlaceOrder(cartModel, orderModel);
            return orderModel;
        }
        return null;
    }

    @Override
    protected void beforePlaceOrder(CartModel cartModel) {
        super.beforePlaceOrder(cartModel);

        final boolean isQuoteOrder = !cartModel.getB2bcomments().isEmpty();
        if (isQuoteOrder) {
            cartModel.setStatus(OrderStatus.PENDING_QUOTE);
        } else {
            cartModel.setStatus(OrderStatus.CREATED);
        }

        this.brainTreeCheckoutFacade.beforePlaceOrder(cartModel);
    }

    @Override
    protected void afterPlaceOrder(CartModel cartModel, OrderModel orderModel) {
        if (orderModel != null) {
            getModelService().remove(cartModel);
            getModelService().refresh(orderModel);
        }
    }

    private boolean isNoTransactionAuthorized(final CartModel cart) {
        return cart.getPaymentTransactions().stream()
            .map(PaymentTransactionModel::getEntries)
            .flatMap(Collection::stream)
            .filter(isAuthorizationTypePredicate())
            .filter(isFakeTransactionPredicate().negate())
            .map(PaymentTransactionEntryModel::getTransactionStatus)
            .noneMatch(Predicate.isEqual(TransactionStatus.ACCEPTED.name()));
    }

    private Predicate<PaymentTransactionEntryModel> isAuthorizationTypePredicate() {
        return entry -> PaymentTransactionType.AUTHORIZATION.equals(entry.getType());
    }

    private Predicate<PaymentTransactionEntryModel> isFakeTransactionPredicate() {
        return entry -> FAKE_REQUEST_ID.equals(entry.getRequestId());
    }

    public <T extends AbstractOrderData> T placeOrderIfIntentOrder(PlaceOrderData placeOrderData)
        throws InvalidCartException {
        // term must be checked
        if (!placeOrderData.getTermsCheck().equals(Boolean.TRUE)) {
            throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_TERM_UNCHECKED));
        }

        if (isValidCheckoutCart(placeOrderData)) {
            // validate quote negotiation
            if (isNegotiateQuoteNotNullAndTrue(placeOrderData)) {
                if (StringUtils.isBlank(placeOrderData.getQuoteRequestDescription())) {
                    throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_NO_QUOTE_DESCRIPTION));
                } else {
                    final B2BCommentData b2BComment = new B2BCommentData();
                    b2BComment.setComment(placeOrderData.getQuoteRequestDescription());

                    final CartData cartData = new CartData();
                    cartData.setB2BComment(b2BComment);

                    updateCheckoutCart(cartData);
                }
            }

            // validate replenishment
            if (isReplenishmentOrderNotNullAndTrue(placeOrderData)) {
                if (placeOrderData.getReplenishmentStartDate() == null) {
                    throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_REPLENISHMENT_NO_STARTDATE));
                }

                if (placeOrderData.getReplenishmentRecurrence().equals(B2BReplenishmentRecurrenceEnum.WEEKLY)
                    && CollectionUtils.isEmpty(placeOrderData.getNDaysOfWeek())) {
                    throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_REPLENISHMENT_NO_FREQUENCY));
                }

                final TriggerData triggerData = new TriggerData();
                populateTriggerDataFromPlaceOrderData(placeOrderData, triggerData);

                return (T) scheduleOrder(triggerData);
            }

            return (T) super.placeOrder();
        }

        return null;
    }

    private boolean isReplenishmentOrderNotNullAndTrue(PlaceOrderData placeOrderData) {
        return placeOrderData.getReplenishmentOrder() != null
                && placeOrderData.getReplenishmentOrder().equals(Boolean.TRUE);
    }

    private boolean isNegotiateQuoteNotNullAndTrue(PlaceOrderData placeOrderData) {
        return placeOrderData.getNegotiateQuote() != null && placeOrderData.getNegotiateQuote().equals(Boolean.TRUE);
    }

    /**
     * This method should not be annotated by Override annotation because DefaultB2BCheckoutFacade class in hybris versions
     * before 6 does not contain this method with PLaceOrderData parameter. That's why it should be overloaded.
     *
     */
    protected boolean isValidCheckoutCart(final PlaceOrderData placeOrderData) {
        final CartData cartData = getCheckoutCart();
        final boolean valid = true;

        if (!cartData.isCalculated()) {
            throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_NOT_CALCULATED));
        }

        if (cartData.getDeliveryAddress() == null) {
            throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_DELIVERYADDRESS_INVALID));
        }

        if (cartData.getDeliveryMode() == null) {
            throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_DELIVERYMODE_INVALID));
        }

        final boolean accountPaymentType = CheckoutPaymentType.ACCOUNT.getCode()
            .equals(cartData.getPaymentType().getCode());
        if (!accountPaymentType && cartData.getPaymentInfo() == null) {
            throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_PAYMENTINFO_EMPTY));
        }

        if (Boolean.TRUE.equals(placeOrderData.getNegotiateQuote()) && !cartData.getQuoteAllowed()
            && isAvailableQuoteValidation()) {
            throw new EntityValidationException(getLocalizedString(CART_CHECKOUT_QUOTE_REQUIREMENTS_NOT_SATISFIED));
        }

        return valid;
    }

    private boolean isAvailableQuoteValidation() {
        Integer apiVersion = brainTreeCheckoutFacade.getBrainTreeConfigService().getHybrisBuildApiVersion();
        return apiVersion != null && apiVersion >= MIN_HYBRIS_API_VERSION_FOR_QUOTE_VALIDATION;
    }

    public void setCardPaymentType() {
        final CartModel cart = getCart();
        cart.setPaymentType(CheckoutPaymentType.CARD);
        getModelService().save(cart);
    }

    public BrainTreeTransactionService getBrainTreeTransactionService() {
        return brainTreeTransactionService;
    }

    public void setBrainTreeTransactionService(BrainTreeTransactionService brainTreeTransactionService) {
        this.brainTreeTransactionService = brainTreeTransactionService;
    }

    private void createFakeAuthorizationTransaction(CartModel cart) {

        final PaymentTransactionEntryModel paymentTransactionEntryModel = modelService
            .create(PaymentTransactionEntryModel.class);
        paymentTransactionEntryModel.setType(PaymentTransactionType.AUTHORIZATION);
        paymentTransactionEntryModel.setTransactionStatus(TransactionStatus.ACCEPTED.name());
        paymentTransactionEntryModel.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL.name());
        paymentTransactionEntryModel.setRequestId(FAKE_REQUEST_ID);
        final String code =
            BRAINTREE_PROVIDER_NAME + "_cart_" + cart.getCode() + "_stamp_" + System.currentTimeMillis();
        paymentTransactionEntryModel.setCode(code);

        savePaymentTransaction(paymentTransactionEntryModel, cart);
    }

    private void savePaymentTransaction(final PaymentTransactionEntryModel paymentTransactionEntry,
        final AbstractOrderModel cart) {
        final List<PaymentTransactionModel> paymentTransactions;
        List<PaymentTransactionEntryModel> paymentTransactionEntrys;
        PaymentTransactionModel braintreePaymentTransaction = null;
        paymentTransactions = Lists.newArrayList();

        braintreePaymentTransaction = modelService.create(PaymentTransactionModel.class);
        braintreePaymentTransaction.setRequestId(paymentTransactionEntry.getRequestId());
        braintreePaymentTransaction.setPaymentProvider(BRAINTREE_PROVIDER_NAME);
        if (cart.getPaymentInfo() != null && cart.getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            BrainTreePaymentInfoModel paymentInfo = (BrainTreePaymentInfoModel) cart.getPaymentInfo();
            braintreePaymentTransaction.setInfo(paymentInfo);
        }
        paymentTransactions.add(braintreePaymentTransaction);

        paymentTransactionEntrys = Lists.newArrayList();

        paymentTransactionEntrys.add(paymentTransactionEntry);
        braintreePaymentTransaction.setEntries(paymentTransactionEntrys);
        cart.setPaymentTransactions(paymentTransactions);

        modelService.saveAll(paymentTransactionEntry, braintreePaymentTransaction, cart);
    }

}
