/**
 *
 */
package com.braintree.transaction.service.impl;

import static com.braintree.constants.BraintreeConstants.*;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.customfield.service.BraintreeCustomFieldsService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.payment.PaymentService;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.config.ConfigurationService;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.model.ModelService;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.Currency;
import java.util.Date;
import java.util.List;

import org.apache.commons.configuration.Configuration;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import com.braintree.command.request.BrainTreeAuthorizationRequest;
import com.braintree.command.request.BrainTreeFindMerchantAccountRequest;
import com.braintree.command.result.BrainTreeAuthorizationResult;
import com.braintree.command.result.BrainTreeFindMerchantAccountResult;
import com.braintree.command.result.BrainTreeRefundTransactionResult;
import com.braintree.command.result.BrainTreeVoidResult;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.paypal.converters.impl.BraintreeBillingAddressConverter;
import com.google.common.collect.Lists;


@UnitTest
public class DefaultBrainTreeTransactionServiceTest {

    private static final String BT_CUSTOMER_ID = "BT_CUSTOMER_ID";

	private static final String NONCE = "NONCE";
	private static final String DEVICE_DATA = "DEVICE_DATA";
	private static final String PAYMENT_PROVIDER = "PAYMENT_PROVIDER";
	private static final Boolean USE_PAYMENT_METHOD_TOKEN = Boolean.TRUE;
	private static final String PAYMNET_METHOD_TOKEN = "PAYMNET_METHOD_TOKEN";
	private static final Boolean IS_PAYMENT_SAVED = Boolean.TRUE;
	private static final Boolean LIABILITY_SHIFTED = Boolean.TRUE;
	private static final Boolean THREE_D_SECURE_CONFIGURATION = Boolean.TRUE;
	private static final Boolean ADVANCED_FRAUD_TOOLS = Boolean.TRUE;
	private static final Boolean IS_SKIP_3D_SECURE_LIABILITY_RESULT = Boolean.TRUE;
	private static final String CREDIT_CARD_STATEMENT_NAME = "CREDIT_CARD_STATEMENT_NAME";
	private static final String MERCHANT_ACCOUNT_ID_FOR_CURRENT_SITE = "MERCHANT_ACCOUNT_ID_FOR_CURRENT_SITE";
	private static final String BRAINTREE_CHANNEL = "BRAINTREE_CHANNEL";
	private static final String PAYMENT_CODE = "PAYMENT_CODE";
	private static final String SHIPS_FROM_POSTAL_CODE = "SHIPS_FROM_POSTAL_CODE";

	private static final String CURRENCY_CODE_USD = "USD";
	private static final Double TOTAL_PRICE = Double.valueOf(100);
	private static final Double TOTAL_TAX = Double.valueOf(20);
	private static final Double DELIVERY_COST = Double.valueOf(10);
	private static final String REQUEST_ID = "REQUEST_ID";

    private static final TransactionStatusDetails TRANSACTION_STATUS_DETAILS = TransactionStatusDetails.SUCCESFULL;
    private static final TransactionStatus TRANSACTION_STATUS = TransactionStatus.ACCEPTED;

    private static final String NEW_ENTRY_CODE = "NEW_ENTRY_CODE";
    private static final Currency CURRENCY = Currency.getInstance(CURRENCY_CODE_USD);

	private static final String CART_CODE = "CART_CODE";
	private static final String ORDER_INTENT = "sale";
	private static final Boolean STORE_IN_VAULT = Boolean.TRUE;
	private static final String BRAINTREE_ADDRESS_ID = "BRAINTREE_ADDRESS_ID";
	private static final String POSTAL_CODE = "POSTAL_CODE";
	private static final Boolean PROPERTY_LEVEL2_LEVEL3_ENABLED = Boolean.FALSE;

    @Mock
    private CartService cartService;

    @Mock
    private ModelService modelService;

	@Mock
	private BrainTreePaymentService brainTreePaymentService;

    @Mock
    private PaymentService paymentService;

    @Mock
    private BraintreeBillingAddressConverter billingAddressConverter;


	@Mock
	private CommonI18NService commonI18NService;

	@Mock
	private BrainTreeConfigService brainTreeConfigService;

	@Mock
	private BraintreeCustomFieldsService customFieldsService;

    @Spy
    @InjectMocks
    private final DefaultBrainTreeTransactionService brainTreeTransactionServiceImpl = new DefaultBrainTreeTransactionService();

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
        final CurrencyModel currencyModel = mock(CurrencyModel.class);
        when(commonI18NService.getCurrentCurrency()).thenReturn(currencyModel);
    }

	@Test
	public void shouldCreateAuthTransactionSuccess()
	{
		final CartModel cart = mock(CartModel.class);
		final BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);
		final CustomerModel customerModel = mock(CustomerModel.class);
		when(cart.getEntries()).thenReturn(Collections.emptyList());
		when(cart.getCode()).thenReturn(CART_CODE);
		when(cart.getTotalTax()).thenReturn(TOTAL_TAX);
		when(cart.getDiscounts()).thenReturn(Collections.emptyList());
		when(cart.getDeliveryCost()).thenReturn(DELIVERY_COST);
		when(cart.getUser()).thenReturn(customerModel);
		when(customerModel.getBraintreeCustomerId()).thenReturn(BT_CUSTOMER_ID);
		when(cartService.getSessionCart()).thenReturn(cart);
		when(cart.getPaymentInfo()).thenReturn(paymentInfo);
		final CurrencyModel currencyModel = mock(CurrencyModel.class);
		when(currencyModel.getIsocode()).thenReturn(CURRENCY_CODE_USD);
		when(cart.getCurrency()).thenReturn(currencyModel);
		when(paymentInfo.getNonce()).thenReturn(NONCE);
		when(paymentInfo.getDeviceData()).thenReturn(DEVICE_DATA);
		when(paymentInfo.getPaymentProvider()).thenReturn(PAYMENT_PROVIDER);
		when(paymentInfo.getUsePaymentMethodToken()).thenReturn(USE_PAYMENT_METHOD_TOKEN);
		when(paymentInfo.getPaymentMethodToken()).thenReturn(PAYMNET_METHOD_TOKEN);
		when(Boolean.valueOf(paymentInfo.isSaved())).thenReturn(IS_PAYMENT_SAVED);
		when(paymentInfo.getLiabilityShifted()).thenReturn(LIABILITY_SHIFTED);
		when(paymentInfo.getThreeDSecureConfiguration()).thenReturn(THREE_D_SECURE_CONFIGURATION);
		when(paymentInfo.getAdvancedFraudTools()).thenReturn(ADVANCED_FRAUD_TOOLS);
		when(paymentInfo.getIsSkip3dSecureLiabilityResult()).thenReturn(IS_SKIP_3D_SECURE_LIABILITY_RESULT);
		when(paymentInfo.getCreditCardStatementName()).thenReturn(CREDIT_CARD_STATEMENT_NAME);
		when(paymentInfo.getMerchantAccountIdForCurrentSite()).thenReturn(MERCHANT_ACCOUNT_ID_FOR_CURRENT_SITE);
		when(paymentInfo.getBrainTreeChannel()).thenReturn(BRAINTREE_CHANNEL);
		when(paymentInfo.getShipsFromPostalCode()).thenReturn(SHIPS_FROM_POSTAL_CODE);

		final AddressModel shippingAddress = mock(AddressModel.class);
		final BillingInfo shippingInfo = mock(BillingInfo.class);
		when(cart.getDeliveryAddress()).thenReturn(shippingAddress);
		when(shippingAddress.getBrainTreeAddressId()).thenReturn(BRAINTREE_ADDRESS_ID);
		when(shippingAddress.getPostalcode()).thenReturn(POSTAL_CODE);
		when(billingAddressConverter.convert(shippingAddress)).thenReturn(shippingInfo);
		when(cart.getTotalPrice()).thenReturn(TOTAL_PRICE);

        final BrainTreeFindMerchantAccountResult brainTreeFindMerchantAccountResult = mock(
            BrainTreeFindMerchantAccountResult.class);
        when(Boolean.valueOf(brainTreeFindMerchantAccountResult.isMerchantAccountExist())).thenReturn(Boolean.TRUE);
        when(brainTreePaymentService.findMerchantAccount(any(BrainTreeFindMerchantAccountRequest.class)))
            .thenReturn(brainTreeFindMerchantAccountResult);

        when(paymentInfo.getCode()).thenReturn(PAYMENT_CODE);

		final PaymentTransactionEntryModel paymentTransactionEntry = mock(PaymentTransactionEntryModel.class);
		when(paymentTransactionEntry.getRequestId()).thenReturn(REQUEST_ID);
		when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(paymentTransactionEntry);
		final BrainTreeAuthorizationResult authorizationResult = mock(BrainTreeAuthorizationResult.class);
		when(authorizationResult.getTransactionStatus()).thenReturn(TRANSACTION_STATUS);
		when(authorizationResult.getTransactionStatusDetails()).thenReturn(TRANSACTION_STATUS_DETAILS);
		when(authorizationResult.getRequestId()).thenReturn(REQUEST_ID);
		when(Boolean.valueOf(authorizationResult.isSuccess())).thenReturn(Boolean.TRUE);
		when(authorizationResult.getCurrency()).thenReturn(CURRENCY);
		when(authorizationResult.getTotalAmount()).thenReturn(BigDecimal.valueOf(TOTAL_PRICE+TOTAL_TAX));
		when(paymentTransactionEntry.getAmount()).thenReturn(BigDecimal.valueOf(TOTAL_PRICE));

        when(brainTreePaymentService.authorize(any(BrainTreeAuthorizationRequest.class), any(CustomerModel.class)))
            .thenReturn(authorizationResult);

        final PaymentTransactionModel paymentTransaction = mock(PaymentTransactionModel.class);
        when(modelService.create(PaymentTransactionModel.class)).thenReturn(paymentTransaction);
        final List<PaymentTransactionEntryModel> paymentTransactionEntries = Lists.newArrayList();
        paymentTransactionEntries.add(paymentTransactionEntry);
        when(commonI18NService.getCurrency(CURRENCY_CODE_USD)).thenReturn(currencyModel);

		final ConfigurationService configurationService = mock(ConfigurationService.class);
		final Configuration configuration = mock(Configuration.class);
		when(customFieldsService.getDefaultCustomFieldsMap()).thenReturn(Collections.emptyMap());
		when(brainTreeConfigService.getIntent()).thenReturn(ORDER_INTENT);
		when(brainTreeConfigService.getStoreInVaultIgnoringIntent()).thenReturn(STORE_IN_VAULT);
		when(brainTreeConfigService.getConfigurationService()).thenReturn(configurationService);
		when(configurationService.getConfiguration()).thenReturn(configuration);
		when(configuration.getBoolean(PROPERTY_LEVEL2_LEVEL3)).thenReturn(PROPERTY_LEVEL2_LEVEL3_ENABLED);

		final boolean createResult = brainTreeTransactionServiceImpl.createAuthorizationTransaction();

        verify(paymentTransactionEntry).setType(PaymentTransactionType.AUTHORIZATION);
        verify(paymentTransactionEntry).setTransactionStatus(TRANSACTION_STATUS.toString());
        verify(paymentTransactionEntry).setTransactionStatusDetails(TRANSACTION_STATUS_DETAILS.toString());
        verify(paymentTransactionEntry).setRequestId(REQUEST_ID);
        verify(paymentTransactionEntry).setCurrency(currencyModel);
        verify(paymentTransactionEntry).setRequestToken(NONCE);
        verify(paymentTransactionEntry).setCode(anyString());
        verify(paymentTransactionEntry).setAmount(Mockito.<BigDecimal>anyObject());
        verify(paymentTransactionEntry).setTime(Mockito.<Date>anyObject());

        verify(paymentTransaction).setRequestId(REQUEST_ID);
        verify(paymentTransaction).setEntries(paymentTransactionEntries);
        verify(paymentTransaction).setRequestToken(NONCE);
        verify(paymentTransaction).setPaymentProvider(BRAINTREE_PROVIDER_NAME);
        final List<PaymentTransactionModel> paymentTransactions = Lists.newArrayList();
        paymentTransactions.add(paymentTransaction);
        verify(cart).setPaymentTransactions(paymentTransactions);

		verify(modelService).saveAll(paymentTransactionEntry, paymentTransaction, cart);

        assertTrue(createResult);
    }

    @Test
    public void shouldCreateCancelTransaction() {
        final PaymentTransactionEntryModel paymentTransactionEntryModelExpected = mock(
            PaymentTransactionEntryModel.class);
        final BrainTreeVoidResult brainTreeVoidResult = mock(BrainTreeVoidResult.class);
        final PaymentTransactionModel paymentTransactionModel = mock(PaymentTransactionModel.class);
        final CurrencyModel currencyModel = mock(CurrencyModel.class);
        when(paymentTransactionModel.getCurrency()).thenReturn(currencyModel);
        when(paymentService.getNewPaymentTransactionEntryCode(paymentTransactionModel, PaymentTransactionType.CANCEL))
            .thenReturn(NEW_ENTRY_CODE);
        when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(paymentTransactionEntryModelExpected);
        when(brainTreeVoidResult.getRequestId()).thenReturn(REQUEST_ID);
        when(brainTreeVoidResult.getTransactionStatus()).thenReturn(TRANSACTION_STATUS);
        final BigDecimal amount = BigDecimal.ONE;
        when(brainTreeVoidResult.getAmount()).thenReturn(amount);
        when(brainTreeVoidResult.getTransactionStatusDetails()).thenReturn(TRANSACTION_STATUS_DETAILS);

        final PaymentTransactionEntryModel paymentTransactionEntryModelActual = brainTreeTransactionServiceImpl
            .createCancelTransaction(paymentTransactionModel, brainTreeVoidResult);

        assertEquals(paymentTransactionEntryModelExpected, paymentTransactionEntryModelActual);

        verify(modelService).saveAll(paymentTransactionEntryModelExpected, paymentTransactionModel);
    }

	@Test
	public void shouldCreateRefundTransaction()
	{
		final PaymentTransactionEntryModel paymentTransactionEntryModelExpected = mock(PaymentTransactionEntryModel.class);
		final BrainTreeRefundTransactionResult brainTreeVoidResult = mock(BrainTreeRefundTransactionResult.class);
		final PaymentTransactionModel paymentTransactionModel = mock(PaymentTransactionModel.class);
		final CurrencyModel currencyModel = mock(CurrencyModel.class);
		final OrderModel orderModel = mock(OrderModel.class);
		when(paymentTransactionModel.getOrder()).thenReturn(orderModel);
		when(orderModel.getCurrency()).thenReturn(currencyModel);
		when(paymentService.getNewPaymentTransactionEntryCode(paymentTransactionModel, PaymentTransactionType.REFUND_STANDALONE))
				.thenReturn(NEW_ENTRY_CODE);
		when(modelService.create(PaymentTransactionEntryModel.class)).thenReturn(paymentTransactionEntryModelExpected);
		when(brainTreeVoidResult.getRequestId()).thenReturn(REQUEST_ID);
		when(brainTreeVoidResult.getTransactionStatus()).thenReturn(TRANSACTION_STATUS);
		final BigDecimal amount = BigDecimal.ONE;
		when(brainTreeVoidResult.getAmount()).thenReturn(amount);
		when(brainTreeVoidResult.getTransactionStatusDetails()).thenReturn(TRANSACTION_STATUS_DETAILS);

        final PaymentTransactionEntryModel paymentTransactionEntryModelActual = brainTreeTransactionServiceImpl
            .createRefundTransaction(paymentTransactionModel, brainTreeVoidResult);

        assertEquals(paymentTransactionEntryModelExpected, paymentTransactionEntryModelActual);
        verify(modelService).saveAll(paymentTransactionEntryModelExpected, paymentTransactionModel);
    }
}
