/**
 *
 */
package com.braintree.payment.info.service.impl;

import static com.braintree.constants.BraintreeConstants.ANDROID_PAY_CARD;
import static com.braintree.constants.BraintreeConstants.ANDROID_PAY_CARD_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.APPLE_PAY_CARD;
import static com.braintree.constants.BraintreeConstants.APPLE_PAY_ORIGIN_TYPE;
import static com.braintree.constants.BraintreeConstants.APPLE_PAY_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_ACCEPTED_PAYMENT_METHODS;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_CREDITCARD_PAYMENT;
import static com.braintree.constants.BraintreeConstants.CREDIT_CARD_DETAILS;
import static com.braintree.constants.BraintreeConstants.CREDIT_CARD_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.GOOGLE_PAY_ORIGIN_TYPE;
import static com.braintree.constants.BraintreeConstants.LOCAL_PAYMENT_DETAILS;
import static com.braintree.constants.BraintreeConstants.LOCAL_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.PAYPAL_ACCOUNT_DETAILS;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT_METHOD;
import static com.braintree.constants.BraintreeConstants.PAYPAL_TRANSACTION_DETAILS;
import static com.braintree.constants.BraintreeConstants.SRC_CARD;
import static com.braintree.constants.BraintreeConstants.VENMO_ACCOUNT_DETAILS;
import static com.braintree.constants.BraintreeConstants.VENMO_CHECKOUT;
import static com.braintree.constants.BraintreeConstants.VENMO_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.VISA_CHECKOUT_CARD_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.VISA_CHECKOUT_ORIGIN_TYPE;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.graphql.commands.response.BrainTreePaymentMethodOrigin;
import com.braintree.graphql.commands.response.BrainTreePaymentMethodSnapshot;
import com.braintree.method.BrainTreePaymentService;
import com.braintreegateway.CreditCard.CardType;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.model.ModelService;

import java.util.Collections;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.customer.dao.BrainTreeCustomerAccountDao;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.payment.info.dao.BrainTreePaymentInfoDao;
import com.braintree.payment.info.service.BraintreePaymentInfoService;

@UnitTest
public class DefaultBraintreeBraintreePaymentInfoServiceTest {

    private static final String CUSTOMER_ID = "CUSTOMER_ID";

    private static final String PAYMENT_METHOD_TOKEN = "PAYMENT_METHOD_TOKEN";
    private static final String PAYMENT_METHOD_GRAPHQL_TOKEN = "PAYMENT_METHOD_GRAPHQL_TOKEN";
    private static final String CORRECT_SOURCE = "CORRECT_SOURCE";
    private static final String UID = "UID";
    public static final boolean SAVE_PAYMENT_INFO = true;
    public static final Boolean USE_PAYMENT_METHOD_TOKEN = Boolean.TRUE;
    public static final Boolean BRAINTREE_3D_SECURE = Boolean.TRUE;
    public static final Boolean BRAINTREE_ADVANCED_FRAUD_TOOLS_ENABLED = Boolean.TRUE;
    public static final Boolean IS_SKIP_3D_SECURE_LIABILITY_RESULT = Boolean.TRUE;
    private static final String BRAINTREE_CREDIT_CARD_STATEMENT_NAME = "BRAINTREE_CREDIT_CARD_STATEMENT_NAME";
    public static final String MERCHANT_ACCOUNT_ID_FOR_CURRENT_SITE_AND_CURRENCY = "MERCHANT_ACCOUNT_ID_FOR_CURRENT_SITE_AND_CURRENCY";
    public static final String BRAINTREE_CHANNEL = "Hybris_BT";
    public static final String CARDHOLDER_NAME = "CARDHOLDER_NAME";
    public static final String CARD_NUMBER = "CARD_NUMBER";
    public static final String EXPIRATION_MONTH = "EXPIRATION_MONTH";
    public static final String EXPIRATION_YEAR = "EXPIRATION_YEAR";

    @Mock
    private BrainTreePaymentInfoDao brainTreePaymentInfoDao;

    @Mock
    private BrainTreeCustomerAccountDao brainTreeCustomerAccountDao;

    @Mock
    private ModelService modelService;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @InjectMocks
    private DefaultBraintreePaymentInfoService unit;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldRemovePaymentInfo() {
        // given
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        when(brainTreePaymentInfoDao.find(CUSTOMER_ID, PAYMENT_METHOD_TOKEN)).thenReturn(brainTreePaymentInfoModel);

        // when
        unit.remove(CUSTOMER_ID, PAYMENT_METHOD_TOKEN);

        // then
        verify(modelService).remove(brainTreePaymentInfoModel);
    }

    @Test
    public void shouldAddToCustomer() {
        // given
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        final CustomerModel customerModel = mock(CustomerModel.class);
        final AddressModel billingAddressModel = mock(AddressModel.class);

        when(brainTreeCustomerAccountDao.findCustomerByBrainTreeCustomerId(CUSTOMER_ID)).thenReturn(customerModel);
        when(customerModel.getUid()).thenReturn(UID);

        when(brainTreePaymentInfoModel.getCustomerId()).thenReturn(CUSTOMER_ID);
        when(brainTreePaymentInfoModel.getBillingAddress()).thenReturn(billingAddressModel);

        when(brainTreeConfigService.get3dSecureConfiguration()).thenReturn(BRAINTREE_3D_SECURE);
        when(brainTreeConfigService.getAdvancedFraudTools()).thenReturn(BRAINTREE_ADVANCED_FRAUD_TOOLS_ENABLED);
        when(brainTreeConfigService.resolveSkip3DSecureLiabilityResult()).thenReturn(IS_SKIP_3D_SECURE_LIABILITY_RESULT);
        when(brainTreeConfigService.getCreditCardStatementName()).thenReturn(BRAINTREE_CREDIT_CARD_STATEMENT_NAME);
        when(brainTreeConfigService.getMerchantAccountIdForCurrentSiteAndCurrency())
            .thenReturn(MERCHANT_ACCOUNT_ID_FOR_CURRENT_SITE_AND_CURRENCY);
        when(brainTreeConfigService.getBrainTreeChannel()).thenReturn(BRAINTREE_CHANNEL);

        // when
        unit.addToCustomer(brainTreePaymentInfoModel);

        // then
        verify(brainTreePaymentInfoModel).setUser(customerModel);
        verify(brainTreePaymentInfoModel).setCode(anyString());
        verify(brainTreePaymentInfoModel).setSaved(SAVE_PAYMENT_INFO);
        verify(brainTreePaymentInfoModel).setUsePaymentMethodToken(USE_PAYMENT_METHOD_TOKEN);
        verify(brainTreePaymentInfoModel).setThreeDSecureConfiguration(BRAINTREE_3D_SECURE);
        verify(brainTreePaymentInfoModel).setAdvancedFraudTools(BRAINTREE_ADVANCED_FRAUD_TOOLS_ENABLED);
        verify(brainTreePaymentInfoModel).setIsSkip3dSecureLiabilityResult(IS_SKIP_3D_SECURE_LIABILITY_RESULT);
        verify(brainTreePaymentInfoModel).setCreditCardStatementName(BRAINTREE_CREDIT_CARD_STATEMENT_NAME);
        verify(brainTreePaymentInfoModel)
            .setMerchantAccountIdForCurrentSite(MERCHANT_ACCOUNT_ID_FOR_CURRENT_SITE_AND_CURRENCY);
        verify(brainTreePaymentInfoModel).setBrainTreeChannel(BRAINTREE_CHANNEL);

        verify(modelService).save(billingAddressModel);
        verify(modelService).save(brainTreePaymentInfoModel);
        verify(modelService).refresh(customerModel);
    }

    @Test
    public void shouldUpdatePaymentInfo() {
        // given
        final BrainTreePaymentInfoModel paymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = mock(BrainTreePaymentInfoModel.class);
        final AddressModel billingAddressModel = mock(AddressModel.class);

        when(paymentInfoModel.getCustomerId()).thenReturn(CUSTOMER_ID);
        when(brainTreePaymentInfoDao.find(CUSTOMER_ID, PAYMENT_METHOD_TOKEN)).thenReturn(brainTreePaymentInfoModel);
        when(paymentInfoModel.getPaymentMethodToken()).thenReturn(PAYMENT_METHOD_TOKEN);
        when(paymentInfoModel.getCardNumber()).thenReturn(CARD_NUMBER);
        when(paymentInfoModel.getCardholderName()).thenReturn(CARDHOLDER_NAME);
        when(paymentInfoModel.getExpirationMonth()).thenReturn(EXPIRATION_MONTH);
        when(paymentInfoModel.getExpirationYear()).thenReturn(EXPIRATION_YEAR);
        when(paymentInfoModel.getBillingAddress()).thenReturn(billingAddressModel);

        // when
        unit.update(PAYMENT_METHOD_TOKEN, paymentInfoModel);

        // then
        verify(brainTreePaymentInfoModel).setPaymentMethodToken(PAYMENT_METHOD_TOKEN);
        verify(brainTreePaymentInfoModel).setCardNumber(CARD_NUMBER);
        verify(brainTreePaymentInfoModel).setCardholderName(CARDHOLDER_NAME);
        verify(brainTreePaymentInfoModel).setExpirationMonth(EXPIRATION_MONTH);
        verify(brainTreePaymentInfoModel).setExpirationYear(EXPIRATION_YEAR);
        verify(billingAddressModel).setOwner(brainTreePaymentInfoModel);
        verify(brainTreePaymentInfoModel).setBillingAddress(billingAddressModel);

        verify(modelService).save(billingAddressModel);
        verify(modelService).save(brainTreePaymentInfoModel);
        verify(modelService).refresh(brainTreePaymentInfoModel);
    }

    @Test
    public void disableShouldNotDisableMethodIfItIsNotBelongForCustomer() {
        BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);
        CustomerModel customerModel = Mockito.mock(CustomerModel.class);

        when(customerModel.getPaymentInfos()).thenReturn(Collections.emptyList());
        when(brainTreePaymentInfoDao.find(CUSTOMER_ID, PAYMENT_METHOD_TOKEN)).thenReturn(paymentInfoModel);
        when(brainTreeCustomerAccountDao.findCustomerByBrainTreeCustomerId(CUSTOMER_ID)).thenReturn(customerModel);

        unit.disable(CUSTOMER_ID, PAYMENT_METHOD_TOKEN);
        verify(brainTreePaymentInfoDao).find(CUSTOMER_ID, PAYMENT_METHOD_TOKEN);
        verify(brainTreeCustomerAccountDao).findCustomerByBrainTreeCustomerId(CUSTOMER_ID);
        verify(paymentInfoModel, never()).setDuplicate(Boolean.TRUE);
    }

    @Test
    public void disableShouldSetDuplicateToTrue() {
        BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);
        CustomerModel customerModel = Mockito.mock(CustomerModel.class);

        when(customerModel.getPaymentInfos()).thenReturn(Collections.singletonList(paymentInfoModel));
        when(brainTreePaymentInfoDao.find(CUSTOMER_ID, PAYMENT_METHOD_TOKEN)).thenReturn(paymentInfoModel);
        when(brainTreeCustomerAccountDao.findCustomerByBrainTreeCustomerId(CUSTOMER_ID)).thenReturn(customerModel);

        unit.disable(CUSTOMER_ID, PAYMENT_METHOD_TOKEN);
        verify(brainTreePaymentInfoDao).find(CUSTOMER_ID, PAYMENT_METHOD_TOKEN);
        verify(brainTreeCustomerAccountDao).findCustomerByBrainTreeCustomerId(CUSTOMER_ID);
        verify(paymentInfoModel).setDuplicate(Boolean.TRUE);
        verify(modelService).save(paymentInfoModel);
        verify(modelService).refresh(customerModel);
    }

    @Test
    public void getGraphQLTokenForPaymentMethodShouldExchangeTokenAndSaveForPaymentInfo() {
        BrainTreePaymentInfoModel paymentInfo = mock(BrainTreePaymentInfoModel.class);

        when(brainTreePaymentInfoDao.find(PAYMENT_METHOD_TOKEN)).thenReturn(paymentInfo);
        when(brainTreePaymentService.exchangeLegacyIdToGraphQLId(any())).thenReturn(PAYMENT_METHOD_GRAPHQL_TOKEN);

        assertEquals(PAYMENT_METHOD_GRAPHQL_TOKEN, unit.getGraphQLTokenForPaymentMethod(PAYMENT_METHOD_TOKEN));
        verify(paymentInfo).setPaymentMethodGraphQLToken(PAYMENT_METHOD_GRAPHQL_TOKEN);
        verify(modelService).save(paymentInfo);
    }

    @Test
    public void getGraphQLTokenForPaymentMethodShouldExchangeTokenEvenPaymentInfoNotExist() {
        when(brainTreePaymentService.exchangeLegacyIdToGraphQLId(any())).thenReturn(PAYMENT_METHOD_GRAPHQL_TOKEN);

        assertEquals(PAYMENT_METHOD_GRAPHQL_TOKEN, unit.getGraphQLTokenForPaymentMethod(PAYMENT_METHOD_TOKEN));
    }

    @Test
    public void getPaymentInstrumentTypeBySnapshotAndOriginShouldReturnPayPal() {
        BrainTreePaymentMethodSnapshot paymentMethodSnapshot = mock(BrainTreePaymentMethodSnapshot.class);
        BrainTreePaymentMethodOrigin origin = mock(BrainTreePaymentMethodOrigin.class);

        when(paymentMethodSnapshot.getOrigin()).thenReturn(origin);
        when(paymentMethodSnapshot.get__typename()).thenReturn(PAYPAL_TRANSACTION_DETAILS);

        assertEquals(PAYPAL_PAYMENT_INSTRUMENT_TYPE,
            unit.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));
    }

    @Test
    public void getPaymentInstrumentTypeBySnapshotAndOriginShouldReturnCreditCard() {
        BrainTreePaymentMethodSnapshot paymentMethodSnapshot = mock(BrainTreePaymentMethodSnapshot.class);
        BrainTreePaymentMethodOrigin origin = mock(BrainTreePaymentMethodOrigin.class);

        when(paymentMethodSnapshot.getOrigin()).thenReturn(origin);
        when(paymentMethodSnapshot.get__typename()).thenReturn(CREDIT_CARD_DETAILS);

        assertEquals(CREDIT_CARD_PAYMENT_INSTRUMENT_TYPE,
            unit.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));
    }

    @Test
    public void getPaymentInstrumentTypeBySnapshotAndOriginShouldReturnVenmo() {
        BrainTreePaymentMethodSnapshot paymentMethodSnapshot = mock(BrainTreePaymentMethodSnapshot.class);
        BrainTreePaymentMethodOrigin origin = mock(BrainTreePaymentMethodOrigin.class);

        when(paymentMethodSnapshot.getOrigin()).thenReturn(origin);
        when(paymentMethodSnapshot.get__typename()).thenReturn(VENMO_ACCOUNT_DETAILS);

        assertEquals(VENMO_PAYMENT_INSTRUMENT_TYPE,
            unit.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));
    }

    @Test
    public void getPaymentInstrumentTypeBySnapshotAndOriginShouldReturnAndroid() {
        BrainTreePaymentMethodSnapshot paymentMethodSnapshot = mock(BrainTreePaymentMethodSnapshot.class);
        BrainTreePaymentMethodOrigin origin = mock(BrainTreePaymentMethodOrigin.class);

        when(paymentMethodSnapshot.getOrigin()).thenReturn(origin);
        when(origin.getType()).thenReturn(GOOGLE_PAY_ORIGIN_TYPE);
        when(paymentMethodSnapshot.get__typename()).thenReturn(CREDIT_CARD_DETAILS);

        assertEquals(ANDROID_PAY_CARD_PAYMENT_INSTRUMENT_TYPE,
            unit.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));
    }

    @Test
    public void getPaymentInstrumentTypeBySnapshotAndOriginShouldReturnSRC() {
        BrainTreePaymentMethodSnapshot paymentMethodSnapshot = mock(BrainTreePaymentMethodSnapshot.class);
        BrainTreePaymentMethodOrigin origin = mock(BrainTreePaymentMethodOrigin.class);

        when(paymentMethodSnapshot.getOrigin()).thenReturn(origin);
        when(origin.getType()).thenReturn(VISA_CHECKOUT_ORIGIN_TYPE);
        when(paymentMethodSnapshot.get__typename()).thenReturn(CREDIT_CARD_DETAILS);

        assertEquals(VISA_CHECKOUT_CARD_PAYMENT_INSTRUMENT_TYPE,
            unit.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));
    }

    @Test
    public void getPaymentInstrumentTypeBySnapshotAndOriginShouldReturnApplePay() {
        BrainTreePaymentMethodSnapshot paymentMethodSnapshot = mock(BrainTreePaymentMethodSnapshot.class);
        BrainTreePaymentMethodOrigin origin = mock(BrainTreePaymentMethodOrigin.class);

        when(paymentMethodSnapshot.getOrigin()).thenReturn(origin);
        when(origin.getType()).thenReturn(APPLE_PAY_ORIGIN_TYPE);
        when(paymentMethodSnapshot.get__typename()).thenReturn(CREDIT_CARD_DETAILS);

        assertEquals(APPLE_PAY_PAYMENT_INSTRUMENT_TYPE,
            unit.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));
    }

    @Test
    public void getPaymentInstrumentTypeBySnapshotAndOriginShouldReturnLPM() {
        BrainTreePaymentMethodSnapshot paymentMethodSnapshot = mock(BrainTreePaymentMethodSnapshot.class);
        BrainTreePaymentMethodOrigin origin = mock(BrainTreePaymentMethodOrigin.class);

        when(paymentMethodSnapshot.getOrigin()).thenReturn(origin);
        when(paymentMethodSnapshot.get__typename()).thenReturn(LOCAL_PAYMENT_DETAILS);

        assertEquals(LOCAL_PAYMENT_INSTRUMENT_TYPE,
            unit.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));
    }

    @Test(expected = IllegalArgumentException.class)
    public void getPaymentInstrumentTypeBySnapshotAndOriginShouldThrowException() {
        BrainTreePaymentMethodSnapshot paymentMethodSnapshot = mock(BrainTreePaymentMethodSnapshot.class);
        BrainTreePaymentMethodOrigin origin = mock(BrainTreePaymentMethodOrigin.class);

        when(paymentMethodSnapshot.getOrigin()).thenReturn(origin);
        when(paymentMethodSnapshot.get__typename()).thenReturn(LOCAL_PAYMENT_INSTRUMENT_TYPE);

        assertEquals(LOCAL_PAYMENT_INSTRUMENT_TYPE,
            unit.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));
    }

    @Test
    public void getPaymentProviderByDetailsAndOriginShoudReturnCorrectProviderForPayPal() {
        assertEquals(PAYPAL_PAYMENT, unit.getPaymentProviderByDetailsAndOrigin(PAYPAL_ACCOUNT_DETAILS, null));
    }

    @Test
    public void getPaymentProviderByDetailsAndOriginShoudReturnCorrectProviderForVenmo() {
        assertEquals(VENMO_CHECKOUT, unit.getPaymentProviderByDetailsAndOrigin(VENMO_ACCOUNT_DETAILS, null));
    }

    @Test
    public void getPaymentProviderByDetailsAndOriginShoudReturnCorrectProviderForGooglePay() {
        assertEquals(ANDROID_PAY_CARD,
            unit.getPaymentProviderByDetailsAndOrigin(CREDIT_CARD_DETAILS, GOOGLE_PAY_ORIGIN_TYPE));
    }

    @Test
    public void getPaymentProviderByDetailsAndOriginShoudReturnCorrectProviderForSRC() {
        assertEquals(SRC_CARD,
            unit.getPaymentProviderByDetailsAndOrigin(CREDIT_CARD_DETAILS, VISA_CHECKOUT_ORIGIN_TYPE));
    }

    @Test
    public void getPaymentProviderByDetailsAndOriginShoudReturnCorrectProviderForApplePay() {
        assertEquals(APPLE_PAY_CARD,
            unit.getPaymentProviderByDetailsAndOrigin(CREDIT_CARD_DETAILS, APPLE_PAY_ORIGIN_TYPE));
    }

    @Test
    public void getPaymentProviderByDetailsAndOriginShoudReturnCorrectProviderForCreditCard() {
        assertEquals(BRAINTREE_CREDITCARD_PAYMENT,
            unit.getPaymentProviderByDetailsAndOrigin(CREDIT_CARD_DETAILS, null));
    }

    @Test
    public void getImageSourceByPaymentProviderAndCardTypeShouldReturnCorrectSourceForPayPal() {
        when(brainTreePaymentService
            .getImageSourceForPaymentMethod(PAYPAL_PAYMENT_METHOD, BRAINTREE_ACCEPTED_PAYMENT_METHODS))
            .thenReturn(CORRECT_SOURCE);

        assertEquals(CORRECT_SOURCE, unit.getImageSourceByPaymentProviderAndCardType(PAYPAL_PAYMENT, null));
    }


    @Test
    public void getImageSourceByPaymentProviderAndCardTypeShouldReturnCorrectSourceForVenmo() {
        when(brainTreePaymentService
            .getImageSourceForPaymentMethod(VENMO_CHECKOUT, BRAINTREE_ACCEPTED_PAYMENT_METHODS))
            .thenReturn(CORRECT_SOURCE);

        assertEquals(CORRECT_SOURCE, unit.getImageSourceByPaymentProviderAndCardType(VENMO_CHECKOUT, null));
    }


    @Test
    public void getImageSourceByPaymentProviderAndCardTypeShouldReturnCorrectSourceForGooglePay() {
        when(brainTreePaymentService
            .getImageSourceForPaymentMethod(ANDROID_PAY_CARD, BRAINTREE_ACCEPTED_PAYMENT_METHODS))
            .thenReturn(CORRECT_SOURCE);

        assertEquals(CORRECT_SOURCE, unit.getImageSourceByPaymentProviderAndCardType(ANDROID_PAY_CARD, null));
    }


    @Test
    public void getImageSourceByPaymentProviderAndCardTypeShouldReturnCorrectSourceForApplePay() {
        when(brainTreePaymentService
            .getImageSourceForPaymentMethod(APPLE_PAY_CARD, BRAINTREE_ACCEPTED_PAYMENT_METHODS))
            .thenReturn(CORRECT_SOURCE);

        assertEquals(CORRECT_SOURCE, unit.getImageSourceByPaymentProviderAndCardType(APPLE_PAY_CARD, null));
    }


    @Test
    public void getImageSourceByPaymentProviderAndCardTypeShouldReturnCorrectSourceForCreditCard() {
        when(brainTreePaymentService
            .getImageSourceForPaymentMethod(CardType.VISA.toString(), BRAINTREE_ACCEPTED_PAYMENT_METHODS))
            .thenReturn(CORRECT_SOURCE);

        assertEquals(CORRECT_SOURCE, unit.getImageSourceByPaymentProviderAndCardType(BRAINTREE_CREDITCARD_PAYMENT, CardType.VISA.toString()));
    }



}
