/**
 *
 */
package com.braintree.order.capture.partial.services;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import com.braintree.command.result.BrainTreeSaleTransactionResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.capture.partial.services.impl.DefaultBraintreePartialCaptureService;
import com.braintree.order.capture.partial.strategy.BraintreePartialOrderCalculationStrategy;
import com.braintree.transaction.service.BrainTreePaymentTransactionService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;

public class DefaultBraintreePartialCaptureServiceTest {

    private static final String AUTHORIZATION_TRANSACTION_ID = "testId";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "lastName";
    private static final String EMAIL = "email";
    private static final String POSTAL_CODE = "postalCode";
    private static final String LINE_1 = "line1";
    private static final String LINE_2 = "line2";

    private static final String TRANSACTION_ID = "testId";

    private AddressModel addressModel;

    @Mock
    private BrainTreeConfigService brainTreeConfigService;

    @Mock
    private BrainTreePaymentTransactionService brainTreePaymentTransactionService;

    @Mock
    private BraintreePartialOrderCalculationStrategy partialOrderCalculationStrategy;

    @Mock
    private ModelService modelService;

    @Mock
    private BrainTreeTransactionService brainTreeTransactionService;

    @Mock
    private BrainTreePaymentService brainTreePaymentService;

    @InjectMocks
    private DefaultBraintreePartialCaptureService unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
        addressModel = Mockito.mock(AddressModel.class);
        when(addressModel.getFirstname()).thenReturn(FIRST_NAME);
        when(addressModel.getLastname()).thenReturn(LAST_NAME);
        when(addressModel.getEmail()).thenReturn(EMAIL);
        when(addressModel.getPostalcode()).thenReturn(POSTAL_CODE);
        when(addressModel.getLine1()).thenReturn(LINE_1);
        when(addressModel.getLine2()).thenReturn(LINE_2);

        BrainTreeSaleTransactionResult result = Mockito.mock(BrainTreeSaleTransactionResult.class);
        when(result.isSuccess()).thenReturn(true);
        when(brainTreePaymentService.partialCaptureTransaction(any())).thenReturn(result);

        when(partialOrderCalculationStrategy.calculateCaptureAmount(any())).thenReturn(BigDecimal.TEN);
    }

    @Test(expected = BraintreeErrorException.class)
    public void partialCaptureWithOrderWhichNotHaveTransactionsShouldTrowException() throws BraintreeErrorException {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        BigDecimal bigDecimal = BigDecimal.TEN;
        CustomerModel userModel = Mockito.mock(CustomerModel.class);

        BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);

        when(orderModel.getUser()).thenReturn(userModel);
        when(orderModel.getPaymentAddress()).thenReturn(addressModel);
        when(orderModel.getDeliveryAddress()).thenReturn(addressModel);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);

        unit.partialCapture(orderModel, bigDecimal, AUTHORIZATION_TRANSACTION_ID);
    }

    @Test
    public void testPartialCaptureWithUncompletedOrder() throws BraintreeErrorException {

        OrderModel orderModel = Mockito.mock(OrderModel.class);
        BigDecimal bigDecimal = BigDecimal.TEN;
        CustomerModel userModel = Mockito.mock(CustomerModel.class);
        PaymentTransactionModel paymentTransactionModel = Mockito.mock(PaymentTransactionModel.class);
        List<PaymentTransactionModel> paymentTransactionModelList = new ArrayList<>();
        paymentTransactionModelList.add(paymentTransactionModel);
        BrainTreePaymentInfoModel paymentInfoModel = Mockito.mock(BrainTreePaymentInfoModel.class);

        when(orderModel.getUser()).thenReturn(userModel);
        when(orderModel.getPaymentAddress()).thenReturn(addressModel);
        when(orderModel.getDeliveryAddress()).thenReturn(addressModel);
        when(orderModel.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(orderModel.getPaymentTransactions()).thenReturn(paymentTransactionModelList);

        unit.partialCapture(orderModel, bigDecimal, AUTHORIZATION_TRANSACTION_ID);
        verify(brainTreePaymentTransactionService).setOrderStatus(orderModel, OrderStatus.PARTIAL_CAPTURE);
    }

    @Test
    public void testGetTransactionAmountWhenPossibleCaptureLessThanNotCaptured() {
        PaymentTransactionEntryModel entryModel = Mockito.mock(PaymentTransactionEntryModel.class);
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        when(entryModel.getAmount()).thenReturn(BigDecimal.TEN);
        when(entryModel.getRequestId()).thenReturn(TRANSACTION_ID);
        when(partialOrderCalculationStrategy.calculateTransaction(any(), any())).thenReturn(BigDecimal.ONE);

        Assert.assertEquals(BigDecimal.ONE, unit.getTransactionAmount(entryModel, orderModel));
    }

    @Test
    public void isPartialCapturePossibleShouldCallConfigService() {
        OrderModel orderModel = Mockito.mock(OrderModel.class);
        unit.isPartialCapturePossible(orderModel);
        verify(brainTreeConfigService).getMultiCaptureEnabled();
    }
}
