/**
 *
 */
package com.braintree.configuration.service;

import static com.braintree.constants.BraintreeConstants.BRAINTREE_3D_SECURE;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_ACCEPTED_CREDIT_CARD_PAYMENT_METHODS;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_ACCEPTED_PAYMENT_METHODS;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_CHANGE_PAYMENT_METHOD_ENABLED;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_DISABLE_FUNDING_ON_MY_ACCOUNT;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_ENVIRONMENT;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_IMAGES_PREFIX;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_MERCHANT_ACCOUNT_PREFIX;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_PAYMENT;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_PAYPAL_INTENT;
import static com.braintree.constants.BraintreeConstants.CARD_FUNDING_SOURCE;
import static com.braintree.constants.BraintreeConstants.CONFIGURATION_PROPERTY_DELIMETER;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_SUBMIT_FOR_SETTLEMENT;
import static com.braintree.constants.BraintreeConstants.ENVIRONMENT_SANDBOX;
import static com.braintree.constants.BraintreeConstants.PAYPAL_DISABLE_FUNDING;
import static com.braintree.constants.BraintreeConstants.PAYPAL_EXPRESS_ENABLED;
import static com.braintree.constants.BraintreeConstants.PAYPAL_INTENT_ORDER;
import static com.braintree.constants.BraintreeConstants.PAYPAL_INTENT_SALE;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT;
import static com.braintree.constants.BraintreeConstants.PAYPAL_STANDARD_ENABLED;
import static com.braintree.constants.BraintreeConstants.STORE_IN_VAULT;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import com.braintree.enums.BraintreePageType;
import com.braintreegateway.Environment;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.servicelayer.config.ConfigurationService;
import de.hybris.platform.servicelayer.i18n.CommonI18NService;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.site.BaseSiteService;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;


@UnitTest
public class BrainTreeConfigServiceTest {

    private static final String PAYMENT_METHOD1 = "PAYMENT_METHOD1";
    private static final String IMAGE_LINK1 = "IMAGE_LINK1";
    private static final String PAYMENT_METHOD2 = "PAYMENT_METHOD2";
    private static final String IMAGE_LINK2 = "IMAGE_LINK2";
    private static final String PAYMENT_METHODS_STRING_ONE_PAYMENT = PAYMENT_METHOD1;
    private static final String PAYMENT_METHODS_STRING_MORE_THAN_ONE_PAYMENT = PAYMENT_METHOD1 + ";" + PAYMENT_METHOD2;
    public static final String ENVIRONMENT_DEVELOPMENT = "development";

    private static final String MERCHANT_ACCOUNT_ID = "MERCHANT_ACCOUNT_ID";
    private static final String BRAINTREE_CUSTOMER_ID = "braintreeId";
    private static final String UID = "uid";
    private static final String ISO_CODE = "iso_code";

    @Mock
    private ConfigurationService configurationService;
    @Mock
    private BaseSiteService baseSiteService;
    @Mock
    private CommonI18NService commonI18NService;
    @Mock
    private UserService userService;

    @InjectMocks
    private final BrainTreeConfigService brainTreeConfigService = new BrainTreeConfigService();

    private Configuration configuration;

    @Before
    public void init() {
        MockitoAnnotations.initMocks(this);
        configuration = mock(Configuration.class);
        when(configurationService.getConfiguration()).thenReturn(configuration);
    }

    @Test
    public void shouldGetSettlementConfigParameter() {
        final Boolean expected = Boolean.TRUE;
        when(configuration.getString(BRAINTREE_SUBMIT_FOR_SETTLEMENT)).thenReturn(String.valueOf(expected));

        final Boolean actual = brainTreeConfigService.getSettlementConfigParameter();

        assertEquals(expected, actual);
    }

    @Test
    public void shouldGet3dSecureConfiguration() {
        final Boolean expected = Boolean.TRUE;
        when(Boolean.valueOf(configuration.getBoolean(BRAINTREE_3D_SECURE, false))).thenReturn(expected);

        final Boolean actual = brainTreeConfigService.get3dSecureConfiguration();

        assertEquals(expected, actual);
    }

    @Test
    public void shouldGetAcceptedPaymentMethodImagesNoAcceptedPaymentMethods() {
        final Map<String, String> acceptedPaymentMethodImagesExpected = new HashMap<>();
        when(configuration.getString(BRAINTREE_ACCEPTED_PAYMENT_METHODS)).thenReturn("");

        final Map<String, String> acceptedPaymentMethodImagesActual = brainTreeConfigService
            .getAcceptedPaymentMethodImages(BRAINTREE_ACCEPTED_CREDIT_CARD_PAYMENT_METHODS);

        assertEquals(acceptedPaymentMethodImagesExpected, acceptedPaymentMethodImagesActual);
    }

    @Test
    public void shouldGetAcceptedPaymentMethodImagesOneAcceptedPaymentMethods() {
        final Map<String, String> acceptedPaymentMethodImagesExpected = new HashMap<>();
        acceptedPaymentMethodImagesExpected.put(PAYMENT_METHOD1, IMAGE_LINK1);
        when(configuration.getString(BRAINTREE_ACCEPTED_PAYMENT_METHODS))
            .thenReturn(PAYMENT_METHODS_STRING_ONE_PAYMENT);
        when(configuration.getString(BRAINTREE_IMAGES_PREFIX + PAYMENT_METHOD1)).thenReturn(IMAGE_LINK1);

        final Map<String, String> acceptedPaymentMethodImagesActual = brainTreeConfigService
            .getAcceptedPaymentMethodImages(BRAINTREE_ACCEPTED_PAYMENT_METHODS);

        assertEquals(acceptedPaymentMethodImagesExpected, acceptedPaymentMethodImagesActual);
    }

    @Test
    public void shouldGetAcceptedPaymentMethodImagesMoreThanOneAcceptedPaymentMethods() {
        final Map<String, String> acceptedPaymentMethodImagesExpected = new HashMap<>();
        acceptedPaymentMethodImagesExpected.put(PAYMENT_METHOD1, IMAGE_LINK1);
        acceptedPaymentMethodImagesExpected.put(PAYMENT_METHOD2, IMAGE_LINK2);
        when(configuration.getString(BRAINTREE_ACCEPTED_PAYMENT_METHODS))
            .thenReturn(PAYMENT_METHODS_STRING_MORE_THAN_ONE_PAYMENT);
        when(configuration.getString(BRAINTREE_IMAGES_PREFIX + PAYMENT_METHOD1)).thenReturn(IMAGE_LINK1);
        when(configuration.getString(BRAINTREE_IMAGES_PREFIX + PAYMENT_METHOD2)).thenReturn(IMAGE_LINK2);

        final Map<String, String> acceptedPaymentMethodImagesActual = brainTreeConfigService
            .getAcceptedPaymentMethodImages(BRAINTREE_ACCEPTED_PAYMENT_METHODS);

        assertEquals(acceptedPaymentMethodImagesExpected, acceptedPaymentMethodImagesActual);
    }

    @Test
    public void shouldGetMerchantAccountIdByCurrentSiteNameAndCurrency() {
        // given
        final BaseSiteModel currentBaseSite = mock(BaseSiteModel.class);
        final CurrencyModel currencyModel = mock(CurrencyModel.class);
        when(baseSiteService.getCurrentBaseSite()).thenReturn(currentBaseSite);
        when(currentBaseSite.getUid()).thenReturn(UID);
        when(commonI18NService.getCurrentCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(ISO_CODE);
        when(configuration
            .getString(BRAINTREE_MERCHANT_ACCOUNT_PREFIX + UID + CONFIGURATION_PROPERTY_DELIMETER + ISO_CODE))
            .thenReturn(MERCHANT_ACCOUNT_ID);

        // when
        final String merchantAccountIdActual = brainTreeConfigService
            .getMerchantAccountIdForCurrentSiteAndCurrency();

        // then
        assertEquals(MERCHANT_ACCOUNT_ID, merchantAccountIdActual);
    }

    @Test
    public void shouldGetMerchantAccountIdByCurrentSiteNameAndCurrencyIsoCode() {
        // given
        final BaseSiteModel currentBaseSite = mock(BaseSiteModel.class);
        when(baseSiteService.getCurrentBaseSite()).thenReturn(currentBaseSite);
        when(currentBaseSite.getUid()).thenReturn(UID);
        when(configuration
            .getString(BRAINTREE_MERCHANT_ACCOUNT_PREFIX + UID + CONFIGURATION_PROPERTY_DELIMETER + ISO_CODE))
            .thenReturn(MERCHANT_ACCOUNT_ID);

        // when
        final String merchantAccountIdActual = brainTreeConfigService
            .getMerchantAccountIdForCurrentSiteAndCurrencyIsoCode(ISO_CODE);

        // then
        assertEquals(MERCHANT_ACCOUNT_ID, merchantAccountIdActual);
    }

    @Test
    public void shouldGetChangePaymentButtonEnabled() {
        final Boolean expected = Boolean.TRUE;

        when(configuration.getBoolean(BRAINTREE_CHANGE_PAYMENT_METHOD_ENABLED, true)).thenReturn(Boolean.TRUE);

        final Boolean actual = brainTreeConfigService.getChangePaymentMethodButtonEnabled();

        assertEquals(expected, actual);

    }

    @Test
    public void shouldGetChangePaymentButtonDisabled() {
        final Boolean expected = Boolean.FALSE;

        when(configuration.getBoolean(BRAINTREE_CHANGE_PAYMENT_METHOD_ENABLED, true)).thenReturn(Boolean.FALSE);

        final Boolean actual = brainTreeConfigService.getChangePaymentMethodButtonEnabled();

        assertEquals(expected, actual);

    }

    @Test
    public void shouldGetSandboxEnvironment() {
        when(configuration.getString(BRAINTREE_ENVIRONMENT)).thenReturn(ENVIRONMENT_SANDBOX);
        assertEquals(Environment.SANDBOX ,brainTreeConfigService.getEnvironmentType());
    }

    @Test
    public void shouldGetDevelopmentEnvironment() {
        when(configuration.getString(BRAINTREE_ENVIRONMENT)).thenReturn(ENVIRONMENT_DEVELOPMENT);
        assertEquals(Environment.DEVELOPMENT ,brainTreeConfigService.getEnvironmentType());
    }

    @Test
    public void getPayPalExpressEnabledShouldReturnFalseIfPayPalStandartEnabledAndExpressEnabled() {
        when(configuration.getBoolean(PAYPAL_STANDARD_ENABLED, true)).thenReturn(false);
        when(configuration.getBoolean(PAYPAL_EXPRESS_ENABLED, true)).thenReturn(true);
        assertFalse(brainTreeConfigService.getPayPalExpressEnabled());
    }

    @Test
    public void getPayPalExpressEnabledShouldReturnTrueIfPayPalStandartAndExpressEnabled() {
        when(configuration.getBoolean(PAYPAL_STANDARD_ENABLED, true)).thenReturn(true);
        when(configuration.getBoolean(PAYPAL_EXPRESS_ENABLED,true)).thenReturn(true);
        assertTrue(brainTreeConfigService.getPayPalExpressEnabled());
    }

    @Test
    public void getStoreInVaultForCurrentUserShouldReturnFalseIfCustomerAnonymous() {
        when(configuration.getString(STORE_IN_VAULT)).thenReturn(Boolean.TRUE.toString());
        when(userService.isAnonymousUser(any())).thenReturn(true);
        assertEquals(Boolean.FALSE.toString(), brainTreeConfigService.getStoreInVaultForCurrentUser());
    }

    @Test
    public void getStoreInVaultForCurrentUserShouldReturnFalseIfIntentOrder() {
        when(configuration.getString(STORE_IN_VAULT)).thenReturn(Boolean.TRUE.toString());
        when(configuration.getString(BRAINTREE_PAYPAL_INTENT)).thenReturn(PAYPAL_INTENT_ORDER);
        assertEquals(Boolean.FALSE.toString(), brainTreeConfigService.getStoreInVaultForCurrentUser());
    }

    @Test
    public void getStoreInVaultForCurrentUserShouldReturnTrue() {
        when(configuration.getString(STORE_IN_VAULT)).thenReturn(Boolean.TRUE.toString());
        when(configuration.getString(BRAINTREE_PAYPAL_INTENT)).thenReturn(PAYPAL_INTENT_SALE);
        assertEquals(Boolean.TRUE.toString(), brainTreeConfigService.getStoreInVaultForCurrentUser());
    }

    @Test
    public void getVaultingForCurrentUserShouldReturnTrue() {
        when(configuration.getString(STORE_IN_VAULT)).thenReturn(Boolean.TRUE.toString());
        when(configuration.getString(BRAINTREE_PAYPAL_INTENT)).thenReturn(PAYPAL_INTENT_ORDER);
        assertTrue(brainTreeConfigService.getVaultingForCurrentUser(BRAINTREE_PAYMENT));
    }

    @Test
    public void getVaultingForCurrentUserShouldReturnFalseForPayPalWithIntentOrder() {
        when(configuration.getString(STORE_IN_VAULT)).thenReturn(Boolean.TRUE.toString());
        when(configuration.getString(BRAINTREE_PAYPAL_INTENT)).thenReturn(PAYPAL_INTENT_ORDER);
        assertFalse(brainTreeConfigService.getVaultingForCurrentUser(PAYPAL_PAYMENT));
    }

    @Test
    public void getStoreInVaultForCardVaultingShouldReturnFalseIfBraintreeIdEmpty() {
        when(configuration.getString(STORE_IN_VAULT)).thenReturn(Boolean.TRUE.toString());
        assertEquals(Boolean.FALSE.toString(), brainTreeConfigService.getStoreInVaultForCardVaulting(StringUtils.EMPTY));
    }

    @Test
    public void getStoreInVaultForCardVaultingShouldReturnTrue() {
        when(configuration.getString(STORE_IN_VAULT)).thenReturn(Boolean.TRUE.toString());
        assertEquals(Boolean.TRUE.toString(), brainTreeConfigService.getStoreInVaultForCardVaulting(BRAINTREE_CUSTOMER_ID));
    }

    @Test
    public void getStoreInVaultForCurrentUserShouldReturnFalseIfIntentOrderAndBrainTreeCustomerIdExist() {
        when(configuration.getString(STORE_IN_VAULT)).thenReturn(Boolean.TRUE.toString());
        when(configuration.getString(BRAINTREE_PAYPAL_INTENT)).thenReturn(PAYPAL_INTENT_ORDER);
        assertEquals(Boolean.FALSE.toString(), brainTreeConfigService.getStoreInVaultForCurrentUser(BRAINTREE_CUSTOMER_ID));
    }

    @Test
    public void getDisableFundingShouldReturnCorrectValueForAccountPage() {
        when(configuration.getString(PAYPAL_DISABLE_FUNDING)).thenReturn(StringUtils.EMPTY);
        assertEquals(BRAINTREE_DISABLE_FUNDING_ON_MY_ACCOUNT, brainTreeConfigService.getDisableFunding(
            BraintreePageType.ACCOUNT));
    }

    @Test
    public void getDisableFundingShouldReturnCardIfChangePaymentButtonEnabled() {
        when(configuration.getString(PAYPAL_DISABLE_FUNDING)).thenReturn(StringUtils.EMPTY);
        when(configuration.getBoolean(BRAINTREE_CHANGE_PAYMENT_METHOD_ENABLED, true)).thenReturn(true);
        assertEquals(CARD_FUNDING_SOURCE, brainTreeConfigService.getDisableFunding(
            BraintreePageType.MARK));
    }

    @Test
    public void isOneOfPaymentMethodsEnabledShouldReturnFalse() {
        assertFalse(brainTreeConfigService.isOneOfPaymentMethodsEnabled());
    }

}
