package com.braintree.payment.info.service.impl;

import static com.braintree.constants.BraintreeConstants.ANDROID_PAY_CARD;
import static com.braintree.constants.BraintreeConstants.ANDROID_PAY_CARD_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.APPLE_PAY_CARD;
import static com.braintree.constants.BraintreeConstants.APPLE_PAY_ORIGIN_TYPE;
import static com.braintree.constants.BraintreeConstants.APPLE_PAY_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_ACCEPTED_PAYMENT_METHODS;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_CREDITCARD_PAYMENT;
import static com.braintree.constants.BraintreeConstants.CREDIT_CARD_DETAILS;
import static com.braintree.constants.BraintreeConstants.CREDIT_CARD_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.CREDIT_CARD_TRANSACTION_DETAILS;
import static com.braintree.constants.BraintreeConstants.GOOGLE_PAY_ORIGIN_TYPE;
import static com.braintree.constants.BraintreeConstants.LOCAL_PAYMENT_DETAILS;
import static com.braintree.constants.BraintreeConstants.LOCAL_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.PAYPAL_ACCOUNT_DETAILS;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT_METHOD;
import static com.braintree.constants.BraintreeConstants.PAYPAL_TRANSACTION_DETAILS;
import static com.braintree.constants.BraintreeConstants.SRC_CARD;
import static com.braintree.constants.BraintreeConstants.TYPE_PAYMENT_METHOD_PARAMETER;
import static com.braintree.constants.BraintreeConstants.VENMO_ACCOUNT_DETAILS;
import static com.braintree.constants.BraintreeConstants.VENMO_CHECKOUT;
import static com.braintree.constants.BraintreeConstants.VENMO_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.VISA_CHECKOUT_CARD_PAYMENT_INSTRUMENT_TYPE;
import static com.braintree.constants.BraintreeConstants.VISA_CHECKOUT_ORIGIN_TYPE;

import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.customer.dao.BrainTreeCustomerAccountDao;
import com.braintree.graphql.commands.request.BrainTreeExchangeLegacyIdRequest;
import com.braintree.graphql.commands.response.BrainTreePaymentMethodSnapshot;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.payment.info.dao.BrainTreePaymentInfoDao;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import de.hybris.platform.commercefacades.user.data.AddressData;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.TitleModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.search.FlexibleSearchService;
import de.hybris.platform.servicelayer.util.ServicesUtil;
import java.util.UUID;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * This class is a default implementation of the BraintreePaymentInfoService interface
 */
public class DefaultBraintreePaymentInfoService implements BraintreePaymentInfoService {

    private static final Logger LOG = Logger.getLogger(DefaultBraintreePaymentInfoService.class);

    private BrainTreePaymentInfoDao brainTreePaymentInfoDao;
    private BrainTreeCustomerAccountDao brainTreeCustomerAccountDao;
    private ModelService modelService;
    private BrainTreeConfigService brainTreeConfigService;
    private FlexibleSearchService flexibleSearchService;
    private BrainTreePaymentService brainTreePaymentService;

    @Override
    public void remove(final String customerId, final String paymentMethodToken) {
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = getBrainTreePaymentInfoDao().find(customerId,
            paymentMethodToken);
        if (brainTreePaymentInfoModel != null) {
            getModelService().remove(brainTreePaymentInfoModel);
        }
    }

    @Override
    public void disable(final String customerId, final String paymentMethodToken) {
        BrainTreePaymentInfoModel brainTreePaymentInfoModel = getBrainTreePaymentInfoDao().find(customerId,
            paymentMethodToken);
        CustomerModel customerModel = getBrainTreeCustomerAccountDao().findCustomerByBrainTreeCustomerId(customerId);
        if (customerModel != null && brainTreePaymentInfoModel != null
            && customerModel.getPaymentInfos().contains(brainTreePaymentInfoModel)) {
            brainTreePaymentInfoModel.setDuplicate(true);
            getModelService().save(brainTreePaymentInfoModel);
            getModelService().refresh(customerModel);
        } else {
            LOG.info("Payment Info " + brainTreePaymentInfoModel
                + " does not belong to the customer " + customerModel + " and will not be removed.");
        }
    }

    @Override
    public void addToCustomer(final BrainTreePaymentInfoModel paymentInfo) {
        ServicesUtil.validateParameterNotNull(paymentInfo, "paymentInfo must not be null");
        final CustomerModel customerModel = getBrainTreeCustomerAccountDao().findCustomerByBrainTreeCustomerId(
            paymentInfo.getCustomerId());
        if (customerModel != null) {
            paymentInfo.setUser(customerModel);
            paymentInfo.setCode(customerModel.getUid() + "_" + UUID.randomUUID());
            paymentInfo.setSaved(Boolean.TRUE.booleanValue());
            paymentInfo.setUsePaymentMethodToken(Boolean.TRUE);
            paymentInfo.setThreeDSecureConfiguration(getBrainTreeConfigService().get3dSecureConfiguration());
            paymentInfo.setAdvancedFraudTools(getBrainTreeConfigService().getAdvancedFraudTools());
            paymentInfo
                .setIsSkip3dSecureLiabilityResult(getBrainTreeConfigService().resolveSkip3DSecureLiabilityResult());
            paymentInfo.setCreditCardStatementName(getBrainTreeConfigService().getCreditCardStatementName());
            paymentInfo.setMerchantAccountIdForCurrentSite(getBrainTreeConfigService()
                .getMerchantAccountIdForCurrentSiteAndCurrency());
            paymentInfo.setBrainTreeChannel(getBrainTreeConfigService().getBrainTreeChannel());

            if (paymentInfo.getBillingAddress() != null) {
                final AddressModel billingAddress = paymentInfo.getBillingAddress();
                billingAddress.setOwner(paymentInfo);
                getModelService().save(billingAddress);
            }
            getModelService().save(paymentInfo);
            getModelService().refresh(customerModel);
        }
    }

    @Override
    public void update(final String paymentMethodToken, final BrainTreePaymentInfoModel paymentInfo) {
        ServicesUtil.validateParameterNotNull(paymentInfo, "paymentInfo must not be null");

        update(paymentInfo, null, paymentMethodToken);
    }

    @Override
    public void update(BrainTreePaymentInfoModel paymentInfo, AddressData addressData) {
        ServicesUtil.validateParameterNotNull(paymentInfo, "paymentInfo must not be null");
        String paymentMethodToken = paymentInfo.getPaymentMethodToken();

        update(paymentInfo, addressData, paymentMethodToken);
    }

    @Override
    public String getGraphQLTokenForPaymentMethod(String legacyToken) {
        BrainTreePaymentInfoModel paymentInfoModel = getBrainTreePaymentInfoDao().find(legacyToken);
        if (paymentInfoModel != null) {
            String methodGraphQLToken = paymentInfoModel.getPaymentMethodGraphQLToken();
            if (methodGraphQLToken == null) {
                BrainTreeExchangeLegacyIdRequest request = new BrainTreeExchangeLegacyIdRequest(legacyToken,
                    TYPE_PAYMENT_METHOD_PARAMETER);
                methodGraphQLToken = brainTreePaymentService.exchangeLegacyIdToGraphQLId(request);
                paymentInfoModel.setPaymentMethodGraphQLToken(methodGraphQLToken);
                modelService.save(paymentInfoModel);
            }
            return methodGraphQLToken;
        } else {
            BrainTreeExchangeLegacyIdRequest request = new BrainTreeExchangeLegacyIdRequest(legacyToken,
                TYPE_PAYMENT_METHOD_PARAMETER);
            return brainTreePaymentService.exchangeLegacyIdToGraphQLId(request);
        }
    }

    @Override
    public String getPaymentInstrumentTypeBySnapshotAndOrigin(BrainTreePaymentMethodSnapshot paymentMethodSnapshot) {
        String paymentDetailsSnapshot = paymentMethodSnapshot.get__typename();
        String originType = null;
        if (paymentMethodSnapshot.getOrigin() != null) {
            originType = paymentMethodSnapshot.getOrigin().getType();
        } else if (paymentMethodSnapshot.getCreditCard() != null && paymentMethodSnapshot.getCreditCard().getOrigin() != null) {
            originType = paymentMethodSnapshot.getCreditCard().getOrigin().getType();
        }
        return setPaymentInstrumentType(paymentDetailsSnapshot, originType);
    }

    private String setPaymentInstrumentType(String paymentDetailsSnapshot, String originType) {
        String paymentInstrumentType;
        if (PAYPAL_TRANSACTION_DETAILS.equals(paymentDetailsSnapshot)) {
            paymentInstrumentType = PAYPAL_PAYMENT_INSTRUMENT_TYPE;
        } else if ((CREDIT_CARD_DETAILS.equals(paymentDetailsSnapshot) || (CREDIT_CARD_TRANSACTION_DETAILS.equals(paymentDetailsSnapshot)))
                && (originType == null)) {
            paymentInstrumentType = CREDIT_CARD_PAYMENT_INSTRUMENT_TYPE;
        }else if (VENMO_ACCOUNT_DETAILS.equals(paymentDetailsSnapshot)) {
            paymentInstrumentType = VENMO_PAYMENT_INSTRUMENT_TYPE;
        } else if (GOOGLE_PAY_ORIGIN_TYPE.equals(originType)) {
            paymentInstrumentType = ANDROID_PAY_CARD_PAYMENT_INSTRUMENT_TYPE;
        } else if (VISA_CHECKOUT_ORIGIN_TYPE.equals(originType)) {
            paymentInstrumentType = VISA_CHECKOUT_CARD_PAYMENT_INSTRUMENT_TYPE;
        } else if (APPLE_PAY_ORIGIN_TYPE.equals(originType)) {
            paymentInstrumentType = APPLE_PAY_PAYMENT_INSTRUMENT_TYPE;
        } else if (LOCAL_PAYMENT_DETAILS.equals(paymentDetailsSnapshot)) {
            paymentInstrumentType = LOCAL_PAYMENT_INSTRUMENT_TYPE;
        } else {
            throw new IllegalArgumentException("Can't find payment instrument type for paymentDetailsSnapshot and originType");
        }
        return paymentInstrumentType;
    }

    @Override
    public String getPaymentProviderByDetailsAndOrigin(String detailsType, String originType) {
        String paymentProvider;
        if (PAYPAL_ACCOUNT_DETAILS.equalsIgnoreCase(detailsType)) {
            paymentProvider = PAYPAL_PAYMENT;
        }  else if (VENMO_ACCOUNT_DETAILS.equalsIgnoreCase(detailsType)) {
            paymentProvider = VENMO_CHECKOUT;
        } else if (GOOGLE_PAY_ORIGIN_TYPE.equalsIgnoreCase(originType)) {
            paymentProvider = ANDROID_PAY_CARD;
        } else if (VISA_CHECKOUT_ORIGIN_TYPE.equalsIgnoreCase(originType)) {
            paymentProvider = SRC_CARD;
        } else if (APPLE_PAY_ORIGIN_TYPE.equalsIgnoreCase(originType)) {
            paymentProvider = APPLE_PAY_CARD;
        } else {
            paymentProvider = BRAINTREE_CREDITCARD_PAYMENT;
        }
        return paymentProvider;
    }

    @Override
    public String getImageSourceByPaymentProviderAndCardType(String paymentProvider, String cardType) {
        String imageSource = "";
        if (PAYPAL_PAYMENT.equalsIgnoreCase(paymentProvider)) {
            imageSource = brainTreePaymentService
                .getImageSourceForPaymentMethod(PAYPAL_PAYMENT_METHOD, BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        } else if (VENMO_CHECKOUT.equalsIgnoreCase(paymentProvider)) {
            imageSource = brainTreePaymentService
                .getImageSourceForPaymentMethod(VENMO_CHECKOUT, BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        } else if (ANDROID_PAY_CARD.equalsIgnoreCase(paymentProvider)) {
            imageSource = brainTreePaymentService.getImageSourceForPaymentMethod(ANDROID_PAY_CARD,
                BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        } else if (APPLE_PAY_CARD.equalsIgnoreCase(paymentProvider)) {
            imageSource = brainTreePaymentService.getImageSourceForPaymentMethod(APPLE_PAY_CARD,
                BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        } else if (StringUtils.isNotEmpty(cardType)) {
            imageSource = brainTreePaymentService.getImageSourceForPaymentMethod(cardType,
                BRAINTREE_ACCEPTED_PAYMENT_METHODS);
        }
        return imageSource;
    }

    private void update(BrainTreePaymentInfoModel paymentInfo, AddressData addressData, String paymentMethodToken) {
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = populateBraintreePaymentInfo(paymentInfo,
            paymentMethodToken);
        AddressModel billingAddress = paymentInfo.getBillingAddress();
        if (brainTreePaymentInfoModel != null && billingAddress != null) {
            updateBillingAddress(brainTreePaymentInfoModel, billingAddress, addressData);
            getModelService().save(brainTreePaymentInfoModel);
            getModelService().refresh(brainTreePaymentInfoModel);
        }
    }

    private BrainTreePaymentInfoModel populateBraintreePaymentInfo(final BrainTreePaymentInfoModel paymentInfo,
        final String originalPaymentMethodToken) {
        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = getBrainTreePaymentInfoDao()
            .find(paymentInfo.getCustomerId(),
                originalPaymentMethodToken);
        if (brainTreePaymentInfoModel != null) {
            brainTreePaymentInfoModel.setPaymentMethodToken(paymentInfo.getPaymentMethodToken());
            brainTreePaymentInfoModel.setCardholderName(paymentInfo.getCardholderName());
            brainTreePaymentInfoModel.setCardNumber(paymentInfo.getCardNumber());
            brainTreePaymentInfoModel.setExpirationMonth(paymentInfo.getExpirationMonth());
            brainTreePaymentInfoModel.setExpirationYear(paymentInfo.getExpirationYear());
            brainTreePaymentInfoModel.setIsDefault(paymentInfo.isIsDefault());
            return brainTreePaymentInfoModel;
        }
        return null;
    }

    private void updateBillingAddress(final BrainTreePaymentInfoModel brainTreePaymentInfoModel,
        final AddressModel billingAddress, final AddressData addressData) {
        if (addressData != null && addressData.getTitleCode() != null) {
            final TitleModel title = new TitleModel();
            title.setCode(addressData.getTitleCode());
            billingAddress.setTitle(getFlexibleSearchService().getModelByExample(title));
        }
        billingAddress.setOwner(brainTreePaymentInfoModel);
        brainTreePaymentInfoModel.setBillingAddress(billingAddress);
        getModelService().save(billingAddress);
    }

    public BrainTreePaymentInfoDao getBrainTreePaymentInfoDao() {
        return brainTreePaymentInfoDao;
    }

    public void setBrainTreePaymentInfoDao(final BrainTreePaymentInfoDao brainTreePaymentInfoDao) {
        this.brainTreePaymentInfoDao = brainTreePaymentInfoDao;
    }

    public ModelService getModelService() {
        return modelService;
    }

    public void setModelService(final ModelService modelService) {
        this.modelService = modelService;
    }

    public BrainTreeCustomerAccountDao getBrainTreeCustomerAccountDao() {
        return brainTreeCustomerAccountDao;
    }

    public void setBrainTreeCustomerAccountDao(final BrainTreeCustomerAccountDao brainTreeCustomerAccountDao) {
        this.brainTreeCustomerAccountDao = brainTreeCustomerAccountDao;
    }

    public BrainTreeConfigService getBrainTreeConfigService() {
        return brainTreeConfigService;
    }

    public void setBrainTreeConfigService(final BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }

    public FlexibleSearchService getFlexibleSearchService() {
        return flexibleSearchService;
    }

    public void setFlexibleSearchService(FlexibleSearchService flexibleSearchService) {
        this.flexibleSearchService = flexibleSearchService;
    }

    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    public void setBrainTreePaymentService(BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }
}
