package com.braintree.order.capture.partial.services.impl;

import com.braintree.command.request.BrainTreeSaleTransactionRequest;
import com.braintree.command.result.BrainTreeSaleTransactionResult;
import com.braintree.configuration.service.BrainTreeConfigService;
import com.braintree.exceptions.BraintreeErrorException;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.capture.partial.process.BraintreeOrderCompleteProcessService;
import com.braintree.order.capture.partial.services.BraintreePartialCaptureService;
import com.braintree.order.capture.partial.strategy.BraintreePartialOrderCalculationStrategy;
import com.braintree.transaction.service.BrainTreePaymentTransactionService;
import com.braintree.transaction.service.BrainTreeTransactionService;
import com.google.common.base.Preconditions;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.CardInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.log4j.Logger;
import de.hybris.platform.payment.enums.PaymentTransactionType;

import java.util.function.Predicate;

import java.math.BigDecimal;
import java.util.List;

import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

/**
 * This class is a default implementation of the BraintreePartialCaptureService interface
 */
public class DefaultBraintreePartialCaptureService implements BraintreePartialCaptureService {

    private static final Logger LOG = Logger.getLogger(DefaultBraintreePartialCaptureService.class);

    private static final String ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE = "orderModel can not be null";
    private BraintreePartialOrderCalculationStrategy partialOrderCalculationStrategy;
    private BrainTreeConfigService brainTreeConfigService;
    private BrainTreePaymentService brainTreePaymentService;
    private BrainTreePaymentTransactionService brainTreePaymentTransactionService;
    private BrainTreeTransactionService brainTreeTransactionService;
    private BraintreeOrderCompleteProcessService orderCompleteProcessService;
    private ModelService modelService;


    @Override
    public PaymentTransactionEntryModel partialCapture(OrderModel orderModel, BigDecimal captureAmount,
        final String authorizeTransactionID)
        throws BraintreeErrorException {
        validateParameterNotNull(orderModel, ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE);

        BrainTreeSaleTransactionRequest request = createRequest(orderModel,
            captureAmount, authorizeTransactionID);
        BrainTreeSaleTransactionResult brainTreeSaleTransactionResult = brainTreePaymentService
            .partialCaptureTransaction(request);

        PaymentTransactionEntryModel transactionEntry = createTransaction(orderModel, brainTreeSaleTransactionResult,
            captureAmount, authorizeTransactionID);

        if (brainTreeSaleTransactionResult.isSuccess()) {
            if (isOrderComplete(orderModel)) {
                getBrainTreePaymentTransactionService().setOrderStatus(orderModel, OrderStatus.PAYMENT_CAPTURED);
                getBrainTreePaymentTransactionService().continueOrderProcess(orderModel);
            } else {
                getBrainTreePaymentTransactionService().setOrderStatus(orderModel, OrderStatus.PARTIAL_CAPTURE);
            }
        }

        return transactionEntry;

    }

    private boolean isOrderComplete(OrderModel orderModel) {
        BigDecimal amountForCapture = getPossibleAmountForCapture(orderModel);
        return BigDecimal.ZERO.compareTo(amountForCapture) >= 0;
    }

    private PaymentTransactionEntryModel createTransaction(OrderModel orderModel,
        BrainTreeSaleTransactionResult brainTreeSaleTransactionResult,
        final BigDecimal requestedAmount, final String authorizeTransactionID) throws BraintreeErrorException {
        final List<PaymentTransactionModel> paymentTransactions = orderModel.getPaymentTransactions();

        final PaymentTransactionType transactionType = PaymentTransactionType.PARTIAL_CAPTURE;
        if (CollectionUtils.isNotEmpty(paymentTransactions)) {
            PaymentTransactionModel transactionModel = paymentTransactions.iterator().next();
            return brainTreeTransactionService
                .createPaymentTransaction(transactionModel, brainTreeSaleTransactionResult,
                    requestedAmount, transactionType, authorizeTransactionID);
        }
        throw new BraintreeErrorException("Order doesn't have payment transactions");
    }


    private BrainTreeSaleTransactionRequest createRequest(OrderModel orderModel, BigDecimal captureAmount,
        String authorizeTransactionID) {
        CustomerModel customer = (CustomerModel) orderModel.getUser();

        BrainTreeSaleTransactionRequest request = new BrainTreeSaleTransactionRequest(customer.getUid(),
            createCard(orderModel.getPaymentAddress()), null, captureAmount,
            createBillingInfo(orderModel.getDeliveryAddress()));

        final BrainTreePaymentInfoModel brainTreePaymentInfoModel = (BrainTreePaymentInfoModel) orderModel
            .getPaymentInfo();

        request.setUsePaymentMethodToken(Boolean.TRUE);
        request.setPaymentMethodToken(brainTreePaymentInfoModel.getPaymentMethodToken());
        request.setRequestId(authorizeTransactionID);
        return request;
    }

    private String getRequestId(final OrderModel orderModel) {
        String requestId = "";
        for (final PaymentTransactionModel paymentTransaction : orderModel.getPaymentTransactions()) {
            if (paymentTransaction.getRequestId() != null && !paymentTransaction.getRequestId().isEmpty()) {
                requestId = paymentTransaction.getRequestId();
            }
        }
        return requestId;
    }

    private BillingInfo createBillingInfo(final AddressModel addressModel) {
        final BillingInfo shippingInfo = new BillingInfo();
        shippingInfo.setFirstName(addressModel.getFirstname());
        shippingInfo.setLastName(addressModel.getLastname());
        shippingInfo.setEmail(addressModel.getEmail());
        shippingInfo.setPostalCode(addressModel.getPostalcode());
        shippingInfo.setStreet1(addressModel.getLine1());
        shippingInfo.setStreet2(addressModel.getLine2());
        return shippingInfo;
    }

    private CardInfo createCard(final AddressModel addressModel) {
        final CardInfo cardInfo = new CardInfo();
        cardInfo.setBillingInfo(createBillingInfo(addressModel));
        return cardInfo;
    }

    @Override
    public BigDecimal getPossibleAmountForCapture(OrderModel orderModel) {
        validateParameterNotNull(orderModel, ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE);
        modelService.refresh(orderModel);
        return partialOrderCalculationStrategy.calculateCaptureAmount(orderModel);
    }

    @Override
    public BigDecimal getPossibleAmountForCaptureForAuthorizedTransaction
        (PaymentTransactionModel paymentTransactionModel, BigDecimal totalAmount) {
        return partialOrderCalculationStrategy.calculateTransaction(paymentTransactionModel, totalAmount.doubleValue());
    }

    @Override
    public boolean isPartialCapturePossible(OrderModel orderModel) {
        validateParameterNotNull(orderModel, ORDER_MODEL_CAN_NOT_BE_NULL_MESSAGE);
        boolean result = brainTreeConfigService.getMultiCaptureEnabled();
        LOG.warn("isPartialCapturePossible, result: " + result);
        return result;
    }

    /**
     * This method is used to get Captured Amount
     * @param orderModel order model
     * @param transactionID transaction id
     * @return captured amount for authorized transaction
     */
    public BigDecimal getCapturedAmountForAuthorizedTransaction(final OrderModel orderModel,
        final String transactionID) {
        validateParameterNotNullStandardMessage("transactionID", transactionID);

        return orderModel.getPaymentTransactions().stream()
            .flatMap(paymentTransaction -> paymentTransaction.getEntries().stream())
            .filter(entry -> transactionID.equals(entry.getSubscriptionID()))
            .filter(getCaptureTransactionTypePredicate())
            .map(PaymentTransactionEntryModel::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * This method is used to get Transaction Amount
     * @param currentModel current PaymentTransactionEntryModel
     * @param orderModel order model
     * @return transaction amount
     */
    public BigDecimal getTransactionAmount(final PaymentTransactionEntryModel currentModel,
        final OrderModel orderModel) {
        validateParameterNotNullStandardMessage("currentModel", currentModel);
        validateParameterNotNullStandardMessage("orderModel", orderModel);

        BigDecimal possibleAmountForCapture = getPossibleAmountForCaptureForAuthorizedTransaction(
            currentModel.getPaymentTransaction(), currentModel.getAmount());
        BigDecimal amountCapturedForTransaction = getCapturedAmountForAuthorizedTransaction(orderModel,
            currentModel.getRequestId());
        BigDecimal currentPaymentEntryAmount = currentModel.getAmount();
        BigDecimal notCaptured = currentPaymentEntryAmount.subtract(amountCapturedForTransaction);

        if (notCaptured.compareTo(BigDecimal.ZERO) > 0) {
            return possibleAmountForCapture.compareTo(notCaptured) > 0 ? notCaptured : possibleAmountForCapture;
        }

        return BigDecimal.ZERO;
    }

    public Predicate<PaymentTransactionEntryModel> getCaptureTransactionTypePredicate() {
        return transaction -> (PaymentTransactionType.CAPTURE.equals(transaction.getType())
            || PaymentTransactionType.PARTIAL_CAPTURE.equals(transaction.getType()))
            && TransactionStatus.ACCEPTED.toString().equals(transaction.getTransactionStatus());
    }

    public Predicate<PaymentTransactionEntryModel> getAuthorizeTransactionTypePredicate() {
        return transaction -> PaymentTransactionType.AUTHORIZATION.equals(transaction.getType())
            && TransactionStatus.ACCEPTED.toString().equals(transaction.getTransactionStatus());
    }

    public Predicate<PaymentTransactionEntryModel> getRefundTransactionTypePredicate() {
        return transaction -> PaymentTransactionType.REFUND_PARTIAL.equals(transaction.getType())
            && TransactionStatus.ACCEPTED.toString().equals(transaction.getTransactionStatus());
    }

    protected void validateParameterNotNull(Object parameter, String nullMessage) {
        Preconditions.checkArgument(parameter != null, nullMessage);
    }

    public BraintreePartialOrderCalculationStrategy getPartialOrderCalculationStrategy() {
        return partialOrderCalculationStrategy;
    }

    public void setPartialOrderCalculationStrategy(
        BraintreePartialOrderCalculationStrategy partialOrderCalculationStrategy) {
        this.partialOrderCalculationStrategy = partialOrderCalculationStrategy;
    }

    public BrainTreeConfigService getBrainTreeConfigService() {
        return brainTreeConfigService;
    }

    public void setBrainTreeConfigService(BrainTreeConfigService brainTreeConfigService) {
        this.brainTreeConfigService = brainTreeConfigService;
    }

    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    public void setBrainTreePaymentService(BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }

    public BrainTreePaymentTransactionService getBrainTreePaymentTransactionService() {
        return brainTreePaymentTransactionService;
    }

    public void setBrainTreePaymentTransactionService(
        BrainTreePaymentTransactionService brainTreePaymentTransactionService) {
        this.brainTreePaymentTransactionService = brainTreePaymentTransactionService;
    }

    public BrainTreeTransactionService getBrainTreeTransactionService() {
        return brainTreeTransactionService;
    }

    public void setBrainTreeTransactionService(BrainTreeTransactionService brainTreeTransactionService) {
        this.brainTreeTransactionService = brainTreeTransactionService;
    }

    public ModelService getModelService() {
        return modelService;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public BraintreeOrderCompleteProcessService getOrderCompleteProcessService() {
        return orderCompleteProcessService;
    }

    public void setOrderCompleteProcessService(BraintreeOrderCompleteProcessService orderCompleteProcessService) {
        this.orderCompleteProcessService = orderCompleteProcessService;
    }
}
