/**
 *
 */
package com.braintree.graphql.commands.impl;

import static com.braintree.constants.BraintreeConstants.INPUT_PARAMETER;
import static com.braintree.constants.BraintreeConstants.RESULT_DATA;
import static com.braintree.constants.BraintreeConstants.RESULT_ERRORS;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import com.braintree.command.request.BrainTreeAuthorizationRequest;
import com.braintree.command.request.BrainTreeSaleTransactionRequest;
import com.braintree.command.request.BrainTreeTokenizeCreditCardRequest;
import com.braintree.command.result.BrainTreeSaleTransactionResult;
import com.braintree.command.result.BrainTreeTokenizeCardResult;
import com.braintree.commands.BrainTreeSaleCommand;
import com.braintree.commands.impl.AbstractCommand;
import com.braintree.commands.impl.DefaultBrainTreeSaleTransactionCommand;
import com.braintree.constants.BraintreeConstants;
import com.braintree.customer.service.BrainTreeCustomerAccountService;
import com.braintree.graphql.commands.request.BrainTreeAddressInput;
import com.braintree.graphql.commands.request.BrainTreeChargePaymentMethodInput;
import com.braintree.graphql.commands.request.BrainTreeTransactionCustomerDetailsInput;
import com.braintree.graphql.commands.request.BrainTreeTransactionDescriptorInput;
import com.braintree.graphql.commands.request.BrainTreeTransactionInput;
import com.braintree.graphql.commands.request.BrainTreeTransactionShippingInput;
import com.braintree.graphql.commands.request.BrainTreeTransactionTaxInput;
import com.braintree.graphql.commands.request.BrainTreeVaultPaymentMethodAfterTransactingInput;
import com.braintree.graphql.commands.request.BrainTreeVaultPaymentMethodCriteria;
import com.braintree.graphql.commands.response.BrainTreeErrorDefinition;
import com.braintree.graphql.commands.response.BrainTreePaymentMethodSnapshot;
import com.braintree.graphql.commands.response.BrainTreeTransaction;
import com.braintree.graphql.commands.response.BrainTreeTransactionPayload;
import com.braintree.method.BrainTreePaymentService;
import com.braintree.payment.info.service.BraintreePaymentInfoService;
import com.braintreegateway.Transaction;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.CardInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

/**
 * This class extends AbstractCommand, implements BrainTreeSaleCommand and is used in GraphQL API.
 */
public class DefaultBrainTreeGraphQLSaleTransactionCommand extends AbstractCommand implements
    BrainTreeSaleCommand {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeSaleTransactionCommand.class);

    private static final String DEFINITION_FILE_NAME_CHARGE_PAYPAL = "chargePayPalAccount";
    private static final String DEFINITION_FILE_NAME_AUTHORIZATION_PAYPAL = "authorizePayPalAccount";

    private BrainTreeCustomerAccountService brainTreeCustomerAccountService;
    private BraintreePaymentInfoService paymentInfoService;
    private BrainTreePaymentService brainTreePaymentService;

    @Override
    public BrainTreeSaleTransactionResult perform(final BrainTreeSaleTransactionRequest saleRequest) {
        LOG.error("---> " + this);
        validateParameterNotNullStandardMessage("Sale transaction request", saleRequest);
        try {
            String fileDefinition = getMutationFileDefinition(saleRequest);
            Map<String, Object> result = makeGraphQlCall(fileDefinition,
                createVariablesMap(saleRequest));
            ArrayList<Map<String, Object>> mapErrors = (ArrayList<Map<String, Object>>) result.get(RESULT_ERRORS);

            BrainTreeSaleTransactionResult braintreeResult;
            if (mapErrors == null) {
                braintreeResult = translateResponse((Map<String, Object>) result.get(RESULT_DATA), fileDefinition);
            } else {
                List<BrainTreeErrorDefinition> errors = mapErrors.stream()
                    .map(stringObjectMap -> objectMapper.convertValue(stringObjectMap, BrainTreeErrorDefinition.class))
                    .collect(Collectors.toList());
                braintreeResult = translateErrorResponse(errors);
            }

            braintreeResult.setRequestBody(parseToXML(createVariablesMap(saleRequest)));
            braintreeResult.setResponseBody(parseToXML(result));
            return braintreeResult;
        } catch (final Exception exception) {
            LOG.error("Error during sale transaction. Request: " + saleRequest, exception);
            throw new AdapterException(exception.getMessage(), exception);
        }
    }

    private BrainTreeSaleTransactionResult translateErrorResponse(List<BrainTreeErrorDefinition> errors) {
        final BrainTreeSaleTransactionResult response = new BrainTreeSaleTransactionResult(false);

        response.setTransactionStatus(TransactionStatus.REJECTED);

        if (errors != null && errors.size() > 0) {
            BrainTreeErrorDefinition errorDefinition = errors.get(0);
            getLoggingHandler().getLogger().info(
                String.format("BT sale transaction(new transaction) with error: %s %s",
                    errorDefinition.getExtensions().getLegacyCode(),
                    errorDefinition.getMessage()));

            if (errorDefinition.getExtensions().getLegacyCode() != null) {
                response.setErrorCode(errorDefinition.getExtensions().getLegacyCode());
            }
            response.setErrorMessage(errorDefinition.getMessage());

        }
        return response;
    }

    private BrainTreeSaleTransactionResult translateResponse(Map<String, Object> data, String mutationName) {
        final BrainTreeSaleTransactionResult result = new BrainTreeSaleTransactionResult(true);
        BrainTreeTransactionPayload payload = objectMapper
            .convertValue(data.get(mutationName), BrainTreeTransactionPayload.class);

        if (payload.getTransaction() != null && payload.getTransaction().getStatus() != null
            && payload.getTransaction().getStatus().isSuccess()) {
            BrainTreeTransaction transaction = payload.getTransaction();
            result.setMerchantTransactionCode(transaction.getMerchantAccountId());
            result.setRequestId(transaction.getLegacyId());
            result.setTransactionGraphQLId(transaction.getId());
            result.setTransactionId(transaction.getLegacyId());
            result.setRequestToken(transaction.getLegacyId());
            result.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);
            result.setTransactionStatus(TransactionStatus.ACCEPTED);
            result.setAmount(new BigDecimal(transaction.getAmount().getValue()));
            result.setCurrencyIsoCode(transaction.getAmount().getCurrencyCode());
            result.setCreatedAt(transaction.getCreatedAt().getTime());

            BrainTreePaymentMethodSnapshot paymentMethodSnapshot = transaction.getPaymentMethodSnapshot();
            result.setPaymentInstrumentType(
                paymentInfoService.getPaymentInstrumentTypeBySnapshotAndOrigin(paymentMethodSnapshot));
        } else {
            result.setSuccess(false);
            result.setErrorMessage(
                getMessageByStatus(payload.getTransaction().getStatusHistory(), payload.getTransaction().getStatus()));
            result.setErrorCode(getErrorCodeFromStatusHistory(payload.getTransaction().getStatusHistory()));
            result.setTransactionStatusDetails(TransactionStatusDetails.BANK_DECLINE);
        }
        return result;
    }

    private Map<String, Object> createVariablesMap(BrainTreeSaleTransactionRequest saleRequest) {
        Map<String, Object> map = new HashMap<>();
        BrainTreeChargePaymentMethodInput input = new BrainTreeChargePaymentMethodInput();
        BrainTreeTransactionInput transaction = new BrainTreeTransactionInput();
        input.setTransaction(transaction);

        transaction.setAmount(saleRequest.getTotalAmount());

        BrainTreeTransactionTaxInput tax = new BrainTreeTransactionTaxInput();
        tax.setTaxAmount(saleRequest.getTaxAmount());
        transaction.setTax(tax);

        if (saleRequest.getCustomerId() != null) {
            transaction
                .setCustomerId(brainTreeCustomerAccountService.getGraphQLIdForCustomer(saleRequest.getCustomerId()));
        }
        transaction.setChannel(getBrainTreeConfigService().getBrainTreeChannel());
        transaction.setMerchantAccountId(saleRequest.getMerchantAccountIdForCurrentSite());

        if (!saleRequest.getUsePaymentMethodToken() && Boolean
            .parseBoolean(getBrainTreeConfigService().getStoreInVault())) {
            BrainTreeVaultPaymentMethodAfterTransactingInput vault = new BrainTreeVaultPaymentMethodAfterTransactingInput();
            vault.setWhen(BrainTreeVaultPaymentMethodCriteria.ON_SUCCESSFUL_TRANSACTION);
            input.getTransaction().setVaultPaymentMethodAfterTransacting(vault);
        }

        setCustomFields(transaction, saleRequest.getCustomFields());

        setAdditionalParameters(saleRequest, input);

        addShippingInfo(saleRequest, input);
        if (saleRequest.getCustomerId() == null && saleRequest.getCustomerEmail() != null) {
            addCustomerInfo(saleRequest, input);
        }

        map.put(INPUT_PARAMETER, input);
        return map;
    }

    private void addCustomerInfo(BrainTreeSaleTransactionRequest saleRequest, BrainTreeChargePaymentMethodInput input) {
        BrainTreeTransactionCustomerDetailsInput customerDetailsInput = new BrainTreeTransactionCustomerDetailsInput();
        customerDetailsInput.setEmail(saleRequest.getCustomerEmail());
        input.getTransaction().setCustomerDetails(customerDetailsInput);
    }

    private void addShippingInfo(BrainTreeSaleTransactionRequest saleRequest, BrainTreeChargePaymentMethodInput input) {
        BrainTreeTransactionShippingInput shippingInput = new BrainTreeTransactionShippingInput();
        BrainTreeAddressInput addressInput = new BrainTreeAddressInput();
        BillingInfo shippingInfo = saleRequest.getShippingInfo();
        if (shippingInfo.getState() != null) {
            addressInput.setRegion(shippingInfo.getState());
        } else {
            addressInput.setRegion(shippingInfo.getRegion());
        }
        addressInput.setCountryCode(shippingInfo.getCountry());
        addressInput.setFirstName(shippingInfo.getFirstName());
        addressInput.setStreetAddress(shippingInfo.getStreet1());
        addressInput.setLastName(shippingInfo.getLastName());
        addressInput.setPostalCode(shippingInfo.getPostalCode());
        addressInput.setLocality(shippingInfo.getCity());
        shippingInput.setShippingAddress(addressInput);
        input.getTransaction().setShipping(shippingInput);
    }

    private void setAdditionalParameters(BrainTreeSaleTransactionRequest saleRequest,
        BrainTreeChargePaymentMethodInput input) {
        if (BooleanUtils.isTrue(saleRequest.getUsePaymentMethodToken())) {
            if (saleRequest.getPaymentMethodToken() == null) {
                getLoggingHandler().getLogger().error("Error: PaymentMethodToken is null!");
                throw new IllegalArgumentException("Error during using existing payment.");
            }
            input.setPaymentMethodId(
                paymentInfoService.getGraphQLTokenForPaymentMethod(saleRequest.getPaymentMethodToken()));
        } else {
            if (StringUtils.isNotBlank(saleRequest.getMethodNonce())) {
                input.setPaymentMethodId(saleRequest.getMethodNonce());
            } else if (StringUtils.isNotBlank(saleRequest.getPaymentMethodToken())) {
                input.setPaymentMethodId(
                    paymentInfoService.getGraphQLTokenForPaymentMethod(saleRequest.getPaymentMethodToken()));
            } else if (saleRequest.getCard() != null) {
                addCardInfo(saleRequest, input);
            }
        }

        if (StringUtils.isNotBlank(getBrainTreeConfigService().getCreditCardStatementName())) {
            BrainTreeTransactionDescriptorInput descriptor = new BrainTreeTransactionDescriptorInput();
            descriptor.setName(saleRequest.getCreditCardStatementName());
            input.getTransaction().setDescriptor(descriptor);
        }
    }

    private void addCardInfo(BrainTreeSaleTransactionRequest saleRequest, BrainTreeChargePaymentMethodInput input) {
        BrainTreeTokenizeCreditCardRequest request = new BrainTreeTokenizeCreditCardRequest();
        CardInfo card = saleRequest.getCard();
        request.setCardHolderName(card.getCardHolderFullName());
        request.setCardNumber(card.getCardNumber());
        request.setCvv(card.getCv2Number());
        request.setExpirationMonth(card.getExpirationMonth());
        request.setExpirationYear(card.getExpirationYear());
        BrainTreeTokenizeCardResult tokenizeCardResult = brainTreePaymentService.tokenizeCreditCard(request);
        input.setPaymentMethodId(tokenizeCardResult.getMethodGraphQLId());
    }

    private String getMutationFileDefinition(BrainTreeSaleTransactionRequest saleRequest) {
        if (getBrainTreeConfigService().getMultiCaptureEnabled() || isAvailableSubmitForSettlement(saleRequest)) {
            return DEFINITION_FILE_NAME_CHARGE_PAYPAL;
        }
        return DEFINITION_FILE_NAME_AUTHORIZATION_PAYPAL;
    }

    private boolean isAvailableSubmitForSettlement(final BrainTreeAuthorizationRequest brainTreeAuthorizationRequest) {
        return !BraintreeConstants.BRAINTREE_PAYMENT.equalsIgnoreCase(brainTreeAuthorizationRequest.getPaymentType())
            && BraintreeConstants.PAYPAL_INTENT_SALE.equalsIgnoreCase(getBrainTreeConfigService().getIntent());
    }

    public BrainTreeCustomerAccountService getBrainTreeCustomerAccountService() {
        return brainTreeCustomerAccountService;
    }

    public void setBrainTreeCustomerAccountService(
        BrainTreeCustomerAccountService brainTreeCustomerAccountService) {
        this.brainTreeCustomerAccountService = brainTreeCustomerAccountService;
    }

    public BraintreePaymentInfoService getPaymentInfoService() {
        return paymentInfoService;
    }

    public void setPaymentInfoService(BraintreePaymentInfoService paymentInfoService) {
        this.paymentInfoService = paymentInfoService;
    }

    public BrainTreePaymentService getBrainTreePaymentService() {
        return brainTreePaymentService;
    }

    public void setBrainTreePaymentService(BrainTreePaymentService brainTreePaymentService) {
        this.brainTreePaymentService = brainTreePaymentService;
    }
}
