/**
 *
 */
package com.braintree.commands.impl;

import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNullStandardMessage;

import com.braintree.command.request.BrainTreeCustomerRequest;
import com.braintree.command.result.BrainTreeFindCustomerResult;
import com.braintree.command.result.BrainTreeFindCustomersResult;
import com.braintree.command.result.BrainTreePaymentDetailsResult;
import com.braintree.commands.BrainTreeFindCustomerCommand;
import com.braintree.constants.BrainTreeSearchTextOptions;
import com.braintree.model.BraintreeCustomerDetailsModel;
import com.braintreegateway.Customer;
import com.braintreegateway.CustomerSearchRequest;
import com.braintreegateway.PaymentMethod;
import com.braintreegateway.ResourceCollection;
import com.braintreegateway.exceptions.NotFoundException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;


public class DefaultBrainTreeFindCustomerCommand extends
    AbstractCommand<BrainTreeCustomerRequest, BrainTreeFindCustomerResult> implements
    BrainTreeFindCustomerCommand {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeFindCustomerCommand.class);

    private Converter<PaymentMethod, BrainTreePaymentDetailsResult> paymentMethodToPaymentDetailsConverter;
    private Converter<Customer, BraintreeCustomerDetailsModel> customerDetailsPopulator;

    @Override
    public BrainTreeFindCustomerResult perform(final BrainTreeCustomerRequest request) {
        validateParameterNotNullStandardMessage("Find Customer Request", request);
        final String customerId = request.getCustomerId();
        validateParameterNotNullStandardMessage("customerId", customerId);
        try {
            final Customer customer = getBraintreeGateway().customer().find(customerId);
            return translateResponse(customer);
        } catch (final NotFoundException exception) {
            LOG.error("[BT Find Customer] Can't find BrainTree customer with id: " + customerId, exception);
            return new BrainTreeFindCustomerResult(false);
        } catch (final Exception exception) {
            LOG.error("[BT Find Customer] Error during try to find customer: " + customerId, exception);
            throw new IllegalArgumentException(exception.getMessage());
        }
    }

    protected BrainTreeFindCustomerResult translateResponse(Customer customer) {
        if (customer != null) {
            final BrainTreeFindCustomerResult result = new BrainTreeFindCustomerResult(true);
            result.setId(customer.getId());
            result.setGraphQLId(customer.getGraphQLId());
            result.setEmail(customer.getEmail());
            result.setFirstName(customer.getFirstName());
            result.setLastName(customer.getLastName());
            List<BrainTreePaymentDetailsResult> paymentMethods = customer.getPaymentMethods()
                .stream()
                .map(paymentMethodToPaymentDetailsConverter::convert)
                .collect(Collectors.toList());
            result.setPaymentMethods(paymentMethods);
            return result;
        }
        return new BrainTreeFindCustomerResult(false);
    }


    /*
     * (non-Javadoc)
     *
     * @see
     * com.braintree.commands.BrainTreeFindCustomerCommand#process(com.braintree.command.request.BrainTreeCustomerRequest
     * )
     */
    @Override
    public BrainTreeFindCustomersResult process(final BrainTreeCustomerRequest request) {
        LOG.info("process, request: " + request);
        try {
            final CustomerSearchRequest customerSearchRequest = translateRequest(request);
            final ResourceCollection<Customer> customers = getBraintreeGateway().customer()
                .search(customerSearchRequest);
            return translateResponse(customers);
        } catch (final NotFoundException exception) {
            LOG.error("Can't find BrainTree Customer, message: " + exception.getMessage(), exception);
            return new BrainTreeFindCustomersResult();
        } catch (final Exception exception) {
            LOG.error("Error while search for Customer, message: " + exception.getMessage(), exception);
            throw new IllegalArgumentException(exception.getMessage());
        }
    }

    protected BrainTreeFindCustomersResult translateResponse(ResourceCollection<Customer> customers) {
        final BrainTreeFindCustomersResult result = new BrainTreeFindCustomersResult();
        int searchLimit = getBrainTreeConfigService().getBackofficeSearchLimit();
        if (customers.getMaximumSize() > searchLimit) {
            String message = "[BT Transaction search]Too many results! Limit is " + searchLimit +
                " Please type transaction id or customer email.";
            LOG.warn(message);
            result.setExceedLimit(true);
            return result;
        }
        final List<BraintreeCustomerDetailsModel> customerList = new ArrayList<>(customers.getMaximumSize());
        customers.forEach(customer -> customerList.add(customerDetailsPopulator.convert(customer)));
        result.setCustomerList(customerList);
        return result;
    }

    private CustomerSearchRequest translateRequest(final BrainTreeCustomerRequest request) {
        final CustomerSearchRequest customerSearchRequest = new CustomerSearchRequest();
        if (request != null) {
            if (StringUtils.isNotBlank(request.getCustomerId())) {
                customerSearchRequest.id().is(request.getCustomerId());
            }
            if (StringUtils.isNotBlank(request.getCustomerEmail())) {
                switch (BrainTreeSearchTextOptions.valueOf(request.getCustomerEmailOperator())) {
                    case CONTAINS:
                        customerSearchRequest.email().contains(request.getCustomerEmail());
                        break;

                    case STARTS_WITH:
                        customerSearchRequest.email().startsWith(request.getCustomerEmail());
                        break;

                    case ENDS_WITH:
                        customerSearchRequest.email().endsWith(request.getCustomerEmail());
                        break;

                    default:
                        customerSearchRequest.email().is(request.getCustomerEmail());
                        break;
                }
            }
        }
        return customerSearchRequest;
    }

    public Converter<PaymentMethod, BrainTreePaymentDetailsResult> getPaymentMethodToPaymentDetailsConverter() {
        return paymentMethodToPaymentDetailsConverter;
    }

    public void setPaymentMethodToPaymentDetailsConverter(
        Converter<PaymentMethod, BrainTreePaymentDetailsResult> paymentMethodToPaymentDetailsConverter) {
        this.paymentMethodToPaymentDetailsConverter = paymentMethodToPaymentDetailsConverter;
    }

    public Converter<Customer, BraintreeCustomerDetailsModel> getCustomerDetailsPopulator() {
        return customerDetailsPopulator;
    }

    public void setCustomerDetailsPopulator(
        Converter<Customer, BraintreeCustomerDetailsModel> customerDetailsPopulator) {
        this.customerDetailsPopulator = customerDetailsPopulator;
    }
}
