/**
 *
 */
package com.braintree.commands.impl;

import static com.braintree.constants.BraintreeConstants.CARD_NUMBER_MASK;

import com.braintree.command.request.BrainTreeCreateCreditCardPaymentMethodRequest;
import com.braintree.command.result.BrainTreeBillingAddressResult;
import com.braintree.command.result.BrainTreePaymentMethodResult;
import com.braintree.commands.BrainTreeCreateCreditCardPaymentMethodCommand;
import com.braintree.constants.BraintreeConstants;
import com.braintree.enums.BrainTreePaymentMethod;
import com.braintreegateway.Address;
import com.braintreegateway.AmexExpressCheckoutCard;
import com.braintreegateway.AndroidPayCard;
import com.braintreegateway.ApplePayCard;
import com.braintreegateway.CreditCard;
import com.braintreegateway.CreditCardVerification;
import com.braintreegateway.PayPalAccount;
import com.braintreegateway.PaymentMethod;
import com.braintreegateway.PaymentMethodRequest;
import com.braintreegateway.Result;
import com.braintreegateway.ValidationError;
import com.braintreegateway.VisaCheckoutCard;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;


public class DefaultBrainTreeCreateCreditCardPaymentMethodCommand extends
    AbstractCommand<BrainTreeCreateCreditCardPaymentMethodRequest, BrainTreePaymentMethodResult> implements
    BrainTreeCreateCreditCardPaymentMethodCommand {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeCreateCreditCardPaymentMethodCommand.class);

    @Override
    public BrainTreePaymentMethodResult perform(final BrainTreeCreateCreditCardPaymentMethodRequest request) {
        final PaymentMethodRequest braintreeRequest = translateRequest(request);

        LOG.info("braintreeRequest: " + braintreeRequest);
        LOG.info("braintreeRequest.xml: " + braintreeRequest.toXML()); // toQueryString

        final Result<? extends PaymentMethod> result = getBraintreeGateway().paymentMethod().create(braintreeRequest);

        LOG.info("result: " + result);
        LOG.info("result.message: " + result.getMessage());
        LOG.info("result.errors: " + result.getErrors());
        LOG.info("result.parameters: " + result.getParameters());

        if (result.isSuccess()) {
            return translateResponse(result);
        } else {
            return translateErrorResponse(braintreeRequest, result);
        }
    }

    private PaymentMethodRequest translateRequest(BrainTreeCreateCreditCardPaymentMethodRequest request) {
        final PaymentMethodRequest paymentMethodRequest = new PaymentMethodRequest();
        paymentMethodRequest.paymentMethodNonce(request.getPaymentMethodNonce()).customerId(request.getCustomerId())
            .cardholderName(request.getCardHolderName());
        paymentMethodRequest.options().makeDefault(request.getIsDefault()).done();

        if (StringUtils.isNotBlank(request.getBillingAddressId())) {
            paymentMethodRequest.billingAddressId(request.getBillingAddressId());
        }
        paymentMethodRequest.options().verifyCard(getBrainTreeConfigService().getVerifyCard());
        paymentMethodRequest.options().verificationMerchantAccountId(
            getBrainTreeConfigService().getMerchantAccountIdForCurrentSiteAndCurrency());
        return paymentMethodRequest;
    }

    private BrainTreePaymentMethodResult translateResponse(Result<? extends PaymentMethod> btResult) {
        final BrainTreePaymentMethodResult result = new BrainTreePaymentMethodResult();
        if (btResult.getTarget() != null) {
            result.setSuccess(btResult.isSuccess());
            if (btResult.isSuccess()) {
                result.setTransactionStatus(TransactionStatus.ACCEPTED);
                result.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);
                if (btResult.getTarget() instanceof PayPalAccount) {
                    parsePayPalAccount(result, (PayPalAccount) btResult.getTarget());
                } else if (btResult.getTarget() instanceof CreditCard) {
                    parseCreditCard(result, (CreditCard) btResult.getTarget());
                } else if (btResult.getTarget() instanceof ApplePayCard) {
                    parseApplePay(result, (ApplePayCard) btResult.getTarget());
                } else if (btResult.getTarget() instanceof AndroidPayCard) {
                    parseAndroidPay(result, (AndroidPayCard) btResult.getTarget());
                } else if (btResult.getTarget() instanceof VisaCheckoutCard) {
                    parseSrcCard(result, (VisaCheckoutCard) btResult.getTarget());
                } else if(btResult.getTarget() instanceof AmexExpressCheckoutCard) {
                    parseAmex(result, (AmexExpressCheckoutCard) btResult.getTarget());
                }
            }
        }
        return result;
    }

    private void parseAmex(BrainTreePaymentMethodResult result, AmexExpressCheckoutCard target) {
        result.setPaymentMethodToken(target.getToken());
        result.setBin(target.getBin());
        result.setCreatedAt(target.getCreatedAt());
        result.setCustomerId(target.getCustomerId());
        result.setExpirationMonth(target.getExpirationMonth());
        result.setExpirationYear(target.getExpirationYear());
        result.setPaymentProvider(BrainTreePaymentMethod.AMEXEXPRESSCHECKOUTCARD.toString());
    }

    private void parseSrcCard(BrainTreePaymentMethodResult result, VisaCheckoutCard target) {
        result.setPaymentMethodToken(target.getToken());
        result.setCreatedAt(target.getCreatedAt());
        result.setExpirationMonth(target.getExpirationMonth());
        result.setExpirationYear(target.getExpirationYear());
        result.setCardType(target.getCardType());
        result.setPaymentProvider(BrainTreePaymentMethod.VISACHECKOUTCARD.toString());
    }

    private void parseAndroidPay(BrainTreePaymentMethodResult result, AndroidPayCard target) {
        result.setPaymentMethodToken(target.getToken());
        result.setBin(target.getBin());
        result.setLast4(target.getLast4());
        result.setCreatedAt(target.getCreatedAt());
        result.setExpirationMonth(target.getExpirationMonth());
        result.setExpirationYear(target.getExpirationYear());
        result.setPaymentProvider(BrainTreePaymentMethod.ANDROIDPAYCARD.toString());
    }

    private void parseApplePay(BrainTreePaymentMethodResult result, ApplePayCard target) {
        result.setPaymentMethodToken(target.getToken());
        result.setLast4(target.getLast4());
        result.setCreatedAt(target.getCreatedAt());
        result.setExpirationMonth(target.getExpirationMonth());
        result.setExpirationYear(target.getExpirationYear());
        result.setPaymentProvider(BrainTreePaymentMethod.APPLEPAYCARD.toString());
    }

    private void parseCreditCard(BrainTreePaymentMethodResult result, CreditCard card) {
        result.setPaymentMethodToken(card.getToken());
        result.setCardType(card.getCardType());
        result.setExpirationMonth(card.getExpirationMonth());
        result.setExpirationYear(card.getExpirationYear());
        result.setCardNumber(String.format(CARD_NUMBER_MASK, card.getLast4()));
        result.setImageSource(card.getImageUrl());
        result.setCardholderName(card.getCardholderName());
        result.setBin(card.getBin());
        result.setLast4(card.getLast4());
        result.setCustomerId(card.getCustomerId());
        result.setPaymentProvider(BrainTreePaymentMethod.CREDITCARD.toString());
        parseBillingAddress(result, card.getBillingAddress());
        getLoggingHandler().handleResult("[PAYMENT METHOD]", card);
    }

    private void parseBillingAddress(BrainTreePaymentMethodResult result, Address billingAddress) {
        BrainTreeBillingAddressResult addressResult = new BrainTreeBillingAddressResult();
        addressResult.setAddressId(billingAddress.getId());
        addressResult.setCompany(billingAddress.getCompany());
        addressResult.setCountyCode(billingAddress.getCountryCodeAlpha2());
        addressResult.setCountyCodeAlpha3(billingAddress.getCountryCodeAlpha3());
        addressResult.setStreetAddress(billingAddress.getStreetAddress());
        addressResult.setExtendedAddress(billingAddress.getExtendedAddress());
        addressResult.setRegion(billingAddress.getRegion());
        addressResult.setLocality(billingAddress.getLocality());
        addressResult.setPostalCode(billingAddress.getPostalCode());
        addressResult.setFirstName(billingAddress.getFirstName());
        addressResult.setLastName(billingAddress.getLastName());
        result.setBillingAddressResult(addressResult);
    }

    private void parsePayPalAccount(BrainTreePaymentMethodResult result, final PayPalAccount payPal) {
        result.setPaymentMethodToken(payPal.getToken());
        result.setImageSource(payPal.getImageUrl());
        result.setEmail(payPal.getEmail());
        result.setCardType(StringUtils.EMPTY);
        result.setExpirationMonth(StringUtils.EMPTY);
        result.setExpirationYear(StringUtils.EMPTY);
        result.setCardNumber(StringUtils.EMPTY);
        result.setCardholderName(StringUtils.EMPTY);
        result.setCreatedAt(payPal.getCreatedAt());
        result.setPaymentProvider(BrainTreePaymentMethod.PAYPAL.toString());
        getLoggingHandler().handleResult("[PAYPAL ACCOUNT]", payPal);
    }

    private BrainTreePaymentMethodResult translateErrorResponse(final PaymentMethodRequest request,
        final Result<? extends PaymentMethod> result) {
        final BrainTreePaymentMethodResult response = new BrainTreePaymentMethodResult();
        response.setSuccess(result.isSuccess());
        if (result.getErrors() != null) {
            final List<ValidationError> allDeepValidationErrors = result.getErrors().getAllDeepValidationErrors();
            if (allDeepValidationErrors != null && allDeepValidationErrors.size() > 0) {
                final ValidationError validationError = allDeepValidationErrors.get(0);
                getLoggingHandler().getLogger().info(
                    String
                        .format("Cannot create Payment method for user(%s) with error: %s %s", request.getCustomerId(),
                            validationError.getCode(), validationError.getMessage()));

                if (validationError.getCode() != null) {
                    response.setErrorCode(validationError.getCode().toString());
                }
                response.setErrorMessage(validationError.getMessage());
            }
            final CreditCardVerification creditCardVerification = result.getCreditCardVerification();
            if (creditCardVerification != null) {
                response.setErrorCode(creditCardVerification.getProcessorResponseCode());
                response.setErrorMessage(getLocalizedErrorMessage(BraintreeConstants.GENERAL_VALIDATION_ERROR_MESSAGE));
                getLoggingHandler().handleCardVerificationError(creditCardVerification);
            }

            getLoggingHandler().handleCardVerificationError(creditCardVerification);
            getLoggingHandler().handleErrors(result.getErrors().getAllDeepValidationErrors());
            getLoggingHandler().handleErrors(result.getErrors().getAllValidationErrors());
        }
        return response;
    }

}
