package com.braintree.commands.impl;

import static com.braintree.constants.BraintreeConstants.ANDROID_PAY_CARD;
import static com.braintree.constants.BraintreeConstants.BRAINTREE_CREDITCARD_PAYMENT;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT;
import static com.braintree.constants.BraintreeConstants.SRC_CARD;

import com.braintree.command.request.BrainTreeAuthorizationRequest;
import com.braintree.command.request.beans.BrainTreeLineItemBean;
import com.braintree.command.result.BrainTreeAuthorizationResult;
import com.braintree.command.result.BrainTreePaymentDetailsResult;
import com.braintree.constants.BraintreeConstants;
import com.braintreegateway.AndroidPayDetails;
import com.braintreegateway.CreditCard;
import com.braintreegateway.Result;
import com.braintreegateway.Transaction;
import com.braintreegateway.TransactionLineItem;
import com.braintreegateway.TransactionLineItemRequest;
import com.braintreegateway.TransactionOptionsRequest;
import com.braintreegateway.TransactionRequest;
import com.braintreegateway.ValidationError;
import com.braintreegateway.VisaCheckoutCardDetails;
import de.hybris.platform.order.CartService;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.AuthorizationCommand;
import de.hybris.platform.payment.commands.request.AuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.AvsStatus;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.CvnStatus;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.servicelayer.config.ConfigurationService;
import de.hybris.platform.servicelayer.util.ServicesUtil;
import java.util.Currency;
import java.util.Date;
import java.util.List;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;


public class DefaultBrainTreeAuthorizationCommand extends AbstractAuthorizationCommand
    implements AuthorizationCommand {

    private static final Logger LOG = Logger.getLogger(DefaultBrainTreeAuthorizationCommand.class);

    private static final String PROPERTY_LEVEL2_LEVEL3 = "braintree.enable.level2.level3.data";

    private CartService cartService;
    private ConfigurationService configurationService;


    @Override
    public AuthorizationResult perform(final AuthorizationRequest authorizationRequest) {
        LOG.info("configured intent: " + getBrainTreeConfigService().getIntent());

        final TransactionRequest transactionRequest = translateRequest(authorizationRequest);

        final Result<Transaction> braintreeReply = getBraintreeGateway().transaction().sale(transactionRequest);

        LOG.info("braintreeReply: " + braintreeReply);

        final BrainTreeAuthorizationResult authorizationResult = translateResponse(braintreeReply);
        authorizationResult.setRequestBody(transactionRequest.toXML());
        authorizationResult.setResponseBody(parseToXML(braintreeReply));

        if (braintreeReply.isSuccess()) {
            authorizationResult.setShouldGetMethodTokenFromBraintree(
                shouldGetPaymentMethodTokenFromBraintree(authorizationResult, authorizationRequest));
        }

        return authorizationResult;
    }

    private TransactionRequest translateRequest(final AuthorizationRequest authorizationRequest) {
        TransactionRequest request = null;

        if (authorizationRequest instanceof BrainTreeAuthorizationRequest) {
            final BrainTreeAuthorizationRequest brainTreeAuthorizationRequest = (BrainTreeAuthorizationRequest) authorizationRequest;

            request = new TransactionRequest().customerId(brainTreeAuthorizationRequest.getCustomerId())
                .amount(authorizationRequest.getTotalAmount());

            setAdditionalParameters(brainTreeAuthorizationRequest, request);

            LOG.info(
                "if Level2 enabled: " + configurationService.getConfiguration()
                    .getBoolean(PROPERTY_LEVEL2_LEVEL3, Boolean.FALSE));
            //                validate Level2 fields, if valid then apply to request
            if (configurationService.getConfiguration().getBoolean(PROPERTY_LEVEL2_LEVEL3, Boolean.FALSE) &&
                isLevel2Applicable(brainTreeAuthorizationRequest)) {
                request.taxAmount(roundNumberToTwoDecimalPlaces(brainTreeAuthorizationRequest.getTaxAmountAuthorize()));

                //                    validate Level3 fields, if valid then apply to request
                if (isLevel3Applicable(brainTreeAuthorizationRequest)) {
                    LOG.info("Apply Level3 data, order: " + brainTreeAuthorizationRequest.getPurchaseOrderNumber());

                    request.shippingAmount(
                        roundNumberToTwoDecimalPlaces(brainTreeAuthorizationRequest.getShippingAmount()));
                    request.discountAmount(
                        roundNumberToTwoDecimalPlaces(brainTreeAuthorizationRequest.getDiscountAmount()));
                    request.shipsFromPostalCode(brainTreeAuthorizationRequest.getShipsFromPostalCode());

                    request.shippingAddress().postalCode(brainTreeAuthorizationRequest.getShippingPostalCode())
                        .countryCodeAlpha3(brainTreeAuthorizationRequest.getShippingCountryCodeAlpha3())
                        .done();

                    for (final BrainTreeLineItemBean bean : brainTreeAuthorizationRequest.getLineItems()) {
                        final TransactionLineItemRequest t = request.lineItem();

                        t.name(bean.getName()).kind(TransactionLineItem.Kind.DEBIT).quantity(bean.getQuantity())
                            .unitAmount(bean.getUnitAmount())
                            .unitOfMeasure(bean.getUnitOfMeasure()).totalAmount(bean.getTotalAmount())
                            .taxAmount(bean.getTaxAmount()).discountAmount(bean.getDiscountAmount())
                            .productCode(bean.getProductCode()).commodityCode(bean.getCommodityCode()).done();
                    }
                }

            }

            getLoggingHandler().handleAuthorizationRequest(brainTreeAuthorizationRequest);

        } else {
            final String errorMessage = "[BT Authorization Error] Authorization Request must be Brain Tree type!";
            getLoggingHandler().getLogger().error(errorMessage);
            throw new AdapterException(errorMessage);
        }
        return request;
    }

    private void setAdditionalParameters(final BrainTreeAuthorizationRequest brainTreeAuthorizationRequest,
        final TransactionRequest request) {
        final TransactionOptionsRequest options = request.options();
        final boolean isAlreadyVaulted = brainTreeAuthorizationRequest.getUsePaymentMethodToken();
        if (isAlreadyVaulted) {
            ServicesUtil.validateParameterNotNull(brainTreeAuthorizationRequest.getPaymentMethodToken(),
                "Error: PaymentMethodToken is null!");
            request.paymentMethodToken(brainTreeAuthorizationRequest.getPaymentMethodToken());
            request.customerId(null);
        } else {
            request.paymentMethodNonce(brainTreeAuthorizationRequest.getMethodNonce());
        }

        Boolean submitForSettlement = brainTreeAuthorizationRequest.getSubmitForSettlement();

        if (isAvailableSubmitForSettlement(brainTreeAuthorizationRequest)
            || isAvailableSubmitForSettlementForPaymentProvider(
            brainTreeAuthorizationRequest)) {
            submitForSettlement = Boolean.TRUE;
        }

        if (isVenmoPayment(brainTreeAuthorizationRequest)
            && StringUtils.isNotEmpty(brainTreeAuthorizationRequest.getVenmoProfileId())) {
            request.options().venmo().profileId(brainTreeAuthorizationRequest.getVenmoProfileId());
        }

        final String storeInVaultForCard =
            getBrainTreeConfigService().getStoreInVaultForCardVaulting(brainTreeAuthorizationRequest.getCustomerId());
        if (isAlreadyVaulted) {
            options.storeInVault(Boolean.FALSE);
        } else if (Boolean.parseBoolean(storeInVaultForCard)) {
            options.storeInVaultOnSuccess(Boolean.TRUE);
        } else {
            options.storeInVault(Boolean.FALSE);
        }

        options.submitForSettlement(submitForSettlement);

        if ((BraintreeConstants.PAY_PAL_EXPRESS_CHECKOUT.equals(brainTreeAuthorizationRequest.getPaymentType())
                || PAYPAL_PAYMENT.equals(brainTreeAuthorizationRequest.getPaymentType()))
                && Boolean.FALSE.equals(Boolean.valueOf(brainTreeAuthorizationRequest.isStoreInVault()))
                && Boolean.TRUE.equals(brainTreeAuthorizationRequest.getAdvancedFraudTools())) {
            request.deviceData(brainTreeAuthorizationRequest.getDeviceData());
        }
        if (BraintreeConstants.BRAINTREE_PAYMENT.equals(brainTreeAuthorizationRequest.getPaymentType())) {
            if (isThreeDSRequired(brainTreeAuthorizationRequest)) {
                boolean threeDSecureRequired = true;
                if (BooleanUtils.isTrue(brainTreeAuthorizationRequest.getIsSkip3dSecureLiabilityResult())
                    && BooleanUtils.isFalse(brainTreeAuthorizationRequest.getLiabilityShifted())) {
                    threeDSecureRequired = false;
                }
                options.threeDSecure().required(threeDSecureRequired);
            }
            if (Boolean.TRUE.equals(brainTreeAuthorizationRequest.getAdvancedFraudTools())) {
                request.deviceData(brainTreeAuthorizationRequest.getDeviceData());
            }
        }

        if ((StringUtils.isNotBlank(brainTreeAuthorizationRequest.getCreditCardStatementName()))) {
            request.descriptor().name(brainTreeAuthorizationRequest.getCreditCardStatementName()).done();
        }

        final BillingInfo shippingInfo = brainTreeAuthorizationRequest.getShippingInfo();
        final BillingInfo billingInfo = brainTreeAuthorizationRequest.getBillingInfo();

        if (StringUtils.isNotEmpty(brainTreeAuthorizationRequest.getBrainTreeBilligAddressId())) {
            request.billingAddressId(brainTreeAuthorizationRequest.getBrainTreeBilligAddressId());
        } else {
            request.billingAddress().countryCodeAlpha2(billingInfo.getCountry()).region(billingInfo.getState())
                .firstName(billingInfo.getFirstName()).lastName(billingInfo.getLastName())
                .streetAddress(billingInfo.getStreet1())
                .extendedAddress(billingInfo.getStreet2()).locality(billingInfo.getCity())
                .postalCode(billingInfo.getPostalCode());
        }

        if (StringUtils.isNotEmpty(brainTreeAuthorizationRequest.getBrainTreeAddressId())) {
            request.shippingAddressId(brainTreeAuthorizationRequest.getBrainTreeAddressId());
        } else {
            request.shippingAddress().countryCodeAlpha2(shippingInfo.getCountry()).region(shippingInfo.getState())
                .firstName(shippingInfo.getFirstName()).lastName(shippingInfo.getLastName())
                .streetAddress(shippingInfo.getStreet1()).extendedAddress(shippingInfo.getStreet2())
                .locality(shippingInfo.getCity()).postalCode(shippingInfo.getPostalCode());
        }

        request.channel(getBrainTreeConfigService().getBrainTreeChannel());
        brainTreeAuthorizationRequest.setBrainTreeChannel(getBrainTreeConfigService().getBrainTreeChannel());
        request.merchantAccountId(brainTreeAuthorizationRequest.getMerchantAccountIdForCurrentSite());
        setCustomFields(brainTreeAuthorizationRequest, request);

        setPayee(options);

    }

    private void setPayee(TransactionOptionsRequest options) {
        final String payee = StringUtils.EMPTY;
        if (StringUtils.isEmpty(getBrainTreeConfigService().getIntent()) || isIntentSale()) {
            options.payeeEmail(payee);
        }
    }

    private void setCustomFields(BrainTreeAuthorizationRequest brainTreeAuthorizationRequest,
        TransactionRequest request) {
        if (brainTreeAuthorizationRequest.getCustomFields() != null && !brainTreeAuthorizationRequest.getCustomFields()
            .isEmpty()) {
            for (final String key : brainTreeAuthorizationRequest.getCustomFields().keySet()) {
                request.customField(key, brainTreeAuthorizationRequest.getCustomFields().get(key));
            }
        }
    }

    private BrainTreeAuthorizationResult translateResponse(final Result<Transaction> braintreeReply) {
        final BrainTreeAuthorizationResult result = new BrainTreeAuthorizationResult();
        result.setTransactionStatus(TransactionStatus.REJECTED);
        if (braintreeReply == null) {
            return result;
        }
        result.setSuccess(braintreeReply.isSuccess());

        final Transaction transaction = braintreeReply.getTarget();

        if (braintreeReply.isSuccess()) {
            processSuccessReply(result, transaction);
        } else if (braintreeReply.getErrors() != null) {
            processErrorReply(braintreeReply, result);
        }

        getLoggingHandler().handleResult(AUTHORIZATION_TRANSACTION, transaction);

        return result;
    }

    private void processSuccessReply(final BrainTreeAuthorizationResult result,
        final Transaction transaction) {
        if (transaction != null) {
            result.setAuthorizationCode(transaction.getProcessorAuthorizationCode());
            result.setAvsStatus(AvsStatus.MATCHED);
            result.setCvnStatus(CvnStatus.MATCHED);

            if (transaction.getAmount() != null) {
                result.setTotalAmount(transaction.getAmount());
            }

            result.setAuthorizationTime(transaction.getCreatedAt().getTime());
            result.setCurrency(Currency.getInstance(transaction.getCurrencyIsoCode()));
            result.setMerchantTransactionCode(transaction.getMerchantAccountId());
            result.setRequestId(transaction.getId());
            result.setRequestGraphQLId(transaction.getGraphQLId());
            result.setRequestToken(transaction.getId());
            result.setTransactionStatus(TransactionStatus.ACCEPTED);
            result.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);
            result.setAuthorizationExpiresAt(transaction.getAuthorizationExpiresAt());

            BrainTreePaymentDetailsResult paymentDetails = new BrainTreePaymentDetailsResult();
            paymentDetails.setPaymentInstrumentType(transaction.getPaymentInstrumentType());
            if (transaction.getPayPalDetails() != null) {
                paymentDetails.setPaymentProvider(PAYPAL_PAYMENT);
                if (transaction.getPayPalDetails().getToken() != null) {
                    paymentDetails.setPaymentMethodToken(transaction.getPayPalDetails().getToken());
                } else if (transaction.getPayPalDetails().getImplicitlyVaultedPaymentMethodToken() != null) {
                    paymentDetails
                        .setPaymentMethodToken(transaction.getPayPalDetails().getImplicitlyVaultedPaymentMethodToken());
                }
            } else if (transaction.getAndroidPayDetails() != null) {
                AndroidPayDetails androidPayDetails = transaction.getAndroidPayDetails();
                paymentDetails.setPaymentProvider(ANDROID_PAY_CARD);
                paymentDetails.setPaymentMethodToken(androidPayDetails.getToken());
                paymentDetails.setCardType(androidPayDetails.getCardType());
                paymentDetails.setLast4(androidPayDetails.getLast4());
                paymentDetails.setExpirationMonth(androidPayDetails.getExpirationMonth());
                paymentDetails.setExpirationYear(androidPayDetails.getExpirationYear());
                paymentDetails.setImageUrl(androidPayDetails.getImageUrl());
            } else if (transaction.getVisaCheckoutCardDetails() != null) {
                VisaCheckoutCardDetails visaCheckoutCardDetails = transaction.getVisaCheckoutCardDetails();
                paymentDetails.setPaymentProvider(SRC_CARD);
                paymentDetails.setPaymentMethodToken(visaCheckoutCardDetails.getToken());
                paymentDetails.setCardType(visaCheckoutCardDetails.getCardType());
                paymentDetails.setLast4(visaCheckoutCardDetails.getLast4());
                paymentDetails.setExpirationMonth(visaCheckoutCardDetails.getExpirationMonth());
                paymentDetails.setExpirationYear(visaCheckoutCardDetails.getExpirationYear());
                paymentDetails.setImageUrl(visaCheckoutCardDetails.getImageUrl());
            } else {
                CreditCard creditCard = transaction.getCreditCard();
                paymentDetails.setPaymentProvider(BRAINTREE_CREDITCARD_PAYMENT);
                paymentDetails.setPaymentMethodToken(creditCard.getToken());
                paymentDetails.setCardType(creditCard.getCardType());
                paymentDetails.setLast4(creditCard.getLast4());
                paymentDetails.setExpirationMonth(creditCard.getExpirationMonth());
                paymentDetails.setExpirationYear(creditCard.getExpirationYear());
                paymentDetails.setImageUrl(creditCard.getImageUrl());
            }
            result.setPaymentDetails(paymentDetails);
        } else {
            result.setTransactionStatusDetails(TransactionStatusDetails.BANK_DECLINE);
        }
    }

    private void processErrorReply(final Result<Transaction> braintreeReply,
        final BrainTreeAuthorizationResult result) {
        if (null == braintreeReply.getTransaction()) {
            result.setRequestId("N/A");
            result.setAuthorizationTime(new Date()); // in this case timestamp not available in the response
        } else {
            result.setRequestId(braintreeReply.getTransaction().getId());
            result.setRequestGraphQLId(braintreeReply.getTransaction().getGraphQLId());
            result.setAuthorizationTime(braintreeReply.getTransaction().getCreatedAt().getTime());
            result.setTotalAmount(braintreeReply.getTransaction().getAmount());
            result.setCurrency(Currency.getInstance(braintreeReply.getTransaction().getCurrencyIsoCode()));
        }

        final StringBuilder errorMessage = new StringBuilder("[ERROR AUTHORIZATION] ");
        final StringBuilder errorMessageReason = new StringBuilder();
        if (braintreeReply.getErrors().getAllDeepValidationErrors() != null
            && braintreeReply.getErrors().getAllDeepValidationErrors().size() > 0) {
            result.setTransactionStatusDetails(getCodeTranslator()
                .translateReasonCode(
                    braintreeReply.getErrors().getAllDeepValidationErrors().get(0).getCode().code));

            List<ValidationError> errors = braintreeReply.getErrors().getAllDeepValidationErrors();
            errorMessage.append(getLoggingHandler().handleErrors(errors));
            errorMessageReason
                .append(getErrorTranslator()
                    .getMessage(braintreeReply.getErrors().getAllDeepValidationErrors().get(0)));
        }
        if (result.getTransactionStatusDetails() == null) {
            result.setTransactionStatusDetails(TransactionStatusDetails.NO_AUTHORIZATION_FOR_SETTLEMENT);
            errorMessage.append(braintreeReply.getMessage());
            errorMessageReason.append(braintreeReply.getMessage());
        }
        LOG.error(errorMessageReason.toString());
    }

    public CartService getCartService() {
        return cartService;
    }

    public void setCartService(CartService cartService) {
        this.cartService = cartService;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }
}
