/**
 *
 */
package com.braintree.commands.impl;

import com.braintree.command.request.BrainTreeSubmitForSettlementTransactionRequest;
import com.braintree.constants.BraintreeConstants;
import com.braintree.model.BrainTreePaymentInfoModel;
import com.braintree.order.service.BraintreeOrderRetrievalService;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.payment.commands.CaptureCommand;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.commands.result.CaptureResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import java.math.BigDecimal;
import java.util.List;
import org.apache.log4j.Logger;

/**
 * This abstract class is for capture in subordinate classes.
 */
public abstract class AbstractCaptureCommand extends AbstractCommand implements CaptureCommand {

    private static final Logger LOG = Logger.getLogger(AbstractCaptureCommand.class);

    protected CaptureRequest request;

    protected BraintreeOrderRetrievalService orderRetrievalService;

    @Override
    public CaptureResult perform(final CaptureRequest request) {
        this.request = request;
        LOG.debug("Capture command implementation: " + this.getClass());
        final CaptureResult result;
        final String transactionID = request.getRequestId();

        final AbstractOrderModel order = orderRetrievalService.getOrderForTransaction(transactionID);

        if (isAvailableMultiCapture(order)) {
            return createReviewNeededResponse();
        } else {
            if (isIntentSale(order) || isAvailableSubmitForSettlement(order)) {
                result = createSettledResponse(request.getRequestId());
            } else {
                getLoggingHandler().getLogger()
                    .info(String.format("[CAPTURING FOR TRANSACTION CODE: %s]", transactionID));
                final BrainTreeSubmitForSettlementTransactionRequest captureRequest =
                    new BrainTreeSubmitForSettlementTransactionRequest(transactionID);
                captureRequest.setTransactionId(transactionID);

                final String orderId = orderRetrievalService.getOrderCodeForTransaction(transactionID);

                if (orderId != null) {
                    getLoggingHandler().getLogger().info(String.format("[ORDER CODE: %s]", orderId));
                    captureRequest.setOrderId(orderId);
                }

                final BigDecimal amount = getSubmitForSettlementAmountForOrder(order);
                if (amount != null && amount.compareTo(BigDecimal.ZERO) > 0) {
                    captureRequest.setAmount(amount);
                }

                result = makeApiCall(captureRequest);
            }
        }

        return result;
    }

    protected abstract CaptureResult makeApiCall(BrainTreeSubmitForSettlementTransactionRequest request);

    protected CaptureResult createReviewNeededResponse() {
        final CaptureResult captureResult = new CaptureResult();
        captureResult.setTransactionStatus(TransactionStatus.REVIEW);
        captureResult.setTransactionStatusDetails(TransactionStatusDetails.REVIEW_NEEDED);
        captureResult.setCurrency(request.getCurrency());
        captureResult.setRequestId(request.getRequestId());
        captureResult.setRequestToken(request.getRequestToken());
        captureResult.setMerchantTransactionCode(request.getMerchantTransactionCode());
        captureResult.setTotalAmount(BigDecimal.ZERO);
        return captureResult;
    }

    protected CaptureResult createSettledResponse(final String transactionID) {
        getLoggingHandler().getLogger().info(String
            .format(
                "[SETTLEMENT FOR TRANSACTION ID: %s WILL BE SETTLED IMMEDIATELY BY BRAINTREE] See transaction details.",
                transactionID));
        LOG.info("WILL BE SETTLED IMMEDIATELY BY BRAINTREE See transaction transactionID: " + transactionID);
        final CaptureResult result = new CaptureResult();

        result.setCurrency(request.getCurrency());
        result.setMerchantTransactionCode(request.getMerchantTransactionCode());
        result.setRequestId(request.getRequestId());
        result.setRequestToken(request.getRequestToken());
        result.setTransactionStatus(TransactionStatus.ACCEPTED);
        result.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL);
        result.setTotalAmount(request.getTotalAmount());

        return result;
    }

    protected BigDecimal getSubmitForSettlementAmountForOrder(AbstractOrderModel orderModel) {
        BigDecimal amount = null;
        final List<PaymentTransactionModel> paymentTransactionModels = orderModel.getPaymentTransactions();
        for (final PaymentTransactionModel transaction : paymentTransactionModels) {
            for (final PaymentTransactionEntryModel transactionEntry : transaction.getEntries()) {
                if (PaymentTransactionType.AUTHORIZATION.equals(transactionEntry.getType())) {
                    amount = transactionEntry.getSubmittedForSettlementAmount();
                    break;
                }
            }
            if (amount != null) {
                break;
            }
        }
        return amount;
    }

    protected boolean isIntentSale(AbstractOrderModel order) {
        if (order != null && order.getPaymentInfo() != null && order
            .getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            if (isNotPayPalPayment(order)) {
                return false;
            }
            final String intent = ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPayPalIntent();
            return BraintreeConstants.PAYPAL_INTENT_SALE.equalsIgnoreCase(intent);
        }
        return false;
    }

    protected boolean isIntentOrder(AbstractOrderModel order) {
        if (order != null && order.getPaymentInfo() != null && order
            .getPaymentInfo() instanceof BrainTreePaymentInfoModel) {
            final String intent = ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPayPalIntent();
            if (BraintreeConstants.PAYPAL_INTENT_ORDER.equalsIgnoreCase(intent)) {
                return true;
            }
        }
        return false;
    }

    protected boolean isNotPayPalPayment(AbstractOrderModel order) {
        final boolean isAccountPaymentSystems = isApplePay(order) || isVenmoPayment(order) || isGooglePayPayment(order);
        return isCreditCardPayment(order) || isSrcPayment(order) || isAccountPaymentSystems;
    }

    protected boolean isAvailableMultiCapture(AbstractOrderModel order) {
        final String paymentProvider = ((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPaymentProvider();

        return (BraintreeConstants.BRAINTREE_PAYMENT.equals(paymentProvider) || isIntentOrder(order))
            && !getBrainTreeConfigService().getSettlementConfigParameter()
            && getBrainTreeConfigService().getMultiCaptureEnabled();
    }

    public BraintreeOrderRetrievalService getOrderRetrievalService() {
        return orderRetrievalService;
    }

    public void setOrderRetrievalService(final BraintreeOrderRetrievalService orderRetrievalService) {
        this.orderRetrievalService = orderRetrievalService;
    }

    protected boolean isApplePay(AbstractOrderModel order) {
        return BraintreeConstants.APPLE_PAY_PAYMENT
            .equalsIgnoreCase(((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPaymentProvider());
    }

    protected boolean isVenmoPayment(AbstractOrderModel order) {
        return BraintreeConstants.VENMO_CHECKOUT
            .equalsIgnoreCase(((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPaymentProvider());
    }

    protected boolean isCreditCardPayment(AbstractOrderModel order) {
        return BraintreeConstants.BRAINTREE_CREDITCARD_PAYMENT
            .equalsIgnoreCase(((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPaymentProvider());
    }

    protected boolean isGooglePayPayment(AbstractOrderModel order) {
        return BraintreeConstants.ANDROID_PAY_CARD
            .equalsIgnoreCase(((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPaymentProvider());
    }

    protected boolean isSrcPayment(AbstractOrderModel order) {
        return BraintreeConstants.SRC_CARD
            .equalsIgnoreCase(((BrainTreePaymentInfoModel) order.getPaymentInfo()).getPaymentProvider());
    }


    protected boolean isAvailableSubmitForSettlement(AbstractOrderModel order) {
        final boolean isSubmitForSettlement = getBrainTreeConfigService().getSettlementConfigParameter();
        final boolean isAccountPaymentSystems = isApplePay(order) || isGooglePayPayment(order) || isVenmoPayment(order);
        final boolean isAnyPayment = isCreditCardPayment(order) || isSrcPayment(order) || isAccountPaymentSystems;
        return isAnyPayment && isSubmitForSettlement;
    }

}
