/**
 *
 */
package com.braintree.commands.impl;

import static com.braintree.constants.BraintreeConstants.ANDROID_PAY_CARD;
import static com.braintree.constants.BraintreeConstants.PAYPAL_PAYMENT;
import static com.braintree.constants.BraintreeConstants.PAY_PAL_EXPRESS_CHECKOUT;
import static com.braintree.constants.BraintreeConstants.SRC_CARD;

import com.braintree.command.request.BrainTreeAuthorizationRequest;
import com.braintree.command.result.BrainTreeAuthorizationResult;
import com.braintree.constants.BraintreeConstants;
import com.braintree.enums.BrainTreePaymentMethod;
import de.hybris.platform.payment.commands.AuthorizationCommand;
import de.hybris.platform.payment.commands.request.AuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.servicelayer.config.ConfigurationService;
import java.math.BigDecimal;
import java.math.RoundingMode;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;

/**
 * This abstract class is for authorization in subordinate classes.
 */
public abstract class AbstractAuthorizationCommand extends AbstractCommand<AuthorizationRequest, AuthorizationResult>
    implements AuthorizationCommand {

    protected static final String AUTHORIZATION_TRANSACTION = "[AUTHORIZATION TRANSACTION] ";

    protected ConfigurationService configurationService;

    protected boolean shouldGetPaymentMethodTokenFromBraintree(BrainTreeAuthorizationResult authorizationResult,
        AuthorizationRequest authorizationRequest) {
        if (authorizationRequest instanceof BrainTreeAuthorizationRequest) {
            final BrainTreeAuthorizationRequest brainTreeAuthorizationRequest = (BrainTreeAuthorizationRequest) authorizationRequest;
            final boolean isAlreadyVaulted = brainTreeAuthorizationRequest.getUsePaymentMethodToken();
            final boolean storeInVaultForCard =
                Boolean.parseBoolean(getBrainTreeConfigService()
                    .getStoreInVaultForCardVaulting(brainTreeAuthorizationRequest.getCustomerId()));
            final boolean isPayPalPayment = brainTreeAuthorizationRequest.getPaymentType()
                .equalsIgnoreCase(PAY_PAL_EXPRESS_CHECKOUT) || brainTreeAuthorizationRequest.getPaymentType()
                .equalsIgnoreCase(PAYPAL_PAYMENT);
            final boolean isNotAlreadyVaultedAndIsPayPalPayment = !isAlreadyVaulted && isPayPalPayment;
            return storeInVaultForCard && getBrainTreeConfigService()
                .getChangePaymentMethodButtonEnabled() && isNotAlreadyVaultedAndIsPayPalPayment
                && authorizationResult.getPaymentDetails().getPaymentMethodToken() == null;
        }
        return false;
    }

    protected boolean isLevel2Applicable(final BrainTreeAuthorizationRequest request) {
        final boolean isOrderNumberValid = StringUtils.isAsciiPrintable(request.getPurchaseOrderNumber())
            && (request.getPurchaseOrderNumber().length() <= 12);
        final boolean isTaxAmountValid = request.getTaxAmountAuthorize().compareTo(0d) >= 0;

        return isOrderNumberValid && isTaxAmountValid;
    }

    protected boolean isLevel3Applicable(final BrainTreeAuthorizationRequest request) {
        final String paymentVia = request.getPaymentType();

        if (paymentVia.equalsIgnoreCase(BrainTreePaymentMethod.PAYPAL.toString()) ||
            PAY_PAL_EXPRESS_CHECKOUT.equalsIgnoreCase(paymentVia) ||
            PAYPAL_PAYMENT.equalsIgnoreCase(paymentVia)) {
            //            Level3 disabled intentionally for PayPal payments, please contact Braintree for details
            return false;
        }
        return true;
    }

    protected static BigDecimal roundNumberToTwoDecimalPlaces(final Double number) {
        return BigDecimal.valueOf(number).setScale(2, RoundingMode.HALF_UP);
    }

    protected boolean isAvailableSubmitForSettlement(
        final BrainTreeAuthorizationRequest brainTreeAuthorizationRequest) {
        final boolean isAccountPaymentSystems = !isApplePay(brainTreeAuthorizationRequest)
                && !isGooglePayPayment(brainTreeAuthorizationRequest) && !isVenmoPayment(brainTreeAuthorizationRequest);
        final boolean isIntentSaleAndCCPaymentSystems = isIntentSale() && !isCreditCard(brainTreeAuthorizationRequest)
                && !isSrcPayment(brainTreeAuthorizationRequest);
        return
            (isCreditCard(brainTreeAuthorizationRequest) && getBrainTreeConfigService().getSettlementConfigParameter())
                || (isAccountPaymentSystems && isIntentSaleAndCCPaymentSystems);
    }

    protected boolean isIntentSale() {
        return BraintreeConstants.PAYPAL_INTENT_SALE.equalsIgnoreCase(getBrainTreeConfigService().getIntent());
    }

    protected boolean isApplePay(BrainTreeAuthorizationRequest brainTreeAuthorizationRequest) {
        return BraintreeConstants.APPLE_PAY_PAYMENT.equalsIgnoreCase(brainTreeAuthorizationRequest.getPaymentType());
    }

    protected boolean isCreditCard(BrainTreeAuthorizationRequest brainTreeAuthorizationRequest) {
        return BraintreeConstants.BRAINTREE_PAYMENT.equalsIgnoreCase(brainTreeAuthorizationRequest.getPaymentType());
    }

    protected boolean isVenmoPayment(BrainTreeAuthorizationRequest brainTreeAuthorizationRequest) {
        return BraintreeConstants.VENMO_CHECKOUT.equalsIgnoreCase(brainTreeAuthorizationRequest.getPaymentType());
    }

    protected boolean isGooglePayPayment(BrainTreeAuthorizationRequest brainTreeAuthorizationRequest) {
        return ANDROID_PAY_CARD.equalsIgnoreCase(brainTreeAuthorizationRequest.getPaymentType());
    }

    protected boolean isSrcPayment(BrainTreeAuthorizationRequest brainTreeAuthorizationRequest) {
        return SRC_CARD.equalsIgnoreCase(brainTreeAuthorizationRequest.getPaymentType());
    }

    protected boolean isAvailableSubmitForSettlementForPaymentProvider(
        BrainTreeAuthorizationRequest brainTreeAuthorizationRequest) {
        final boolean isPaymentSystems = isApplePay(brainTreeAuthorizationRequest)
                || isGooglePayPayment(brainTreeAuthorizationRequest) || isVenmoPayment(brainTreeAuthorizationRequest)
                || isSrcPayment(brainTreeAuthorizationRequest);
        return isPaymentSystems && getBrainTreeConfigService().getSettlementConfigParameter();
    }

    protected boolean isThreeDSRequired(BrainTreeAuthorizationRequest brainTreeAuthorizationRequest) {
        return BooleanUtils.isTrue(brainTreeAuthorizationRequest.getThreeDSecureConfiguration())
            && (!(brainTreeAuthorizationRequest.getUsePaymentMethodToken()))
            && getBrainTreeConfigService().get3dSecureConfiguration();
    }

    public ConfigurationService getConfigurationService() {
        return configurationService;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }
}
