import {Component, ElementRef, OnInit, ViewChild} from '@angular/core';
import {combineLatest, Observable} from 'rxjs';
import {map} from 'rxjs/operators';
import {
  GooglePayCheckoutService,
  PageType,
  CheckoutData,
  PayPalButtonStyle,
  PaypalCheckoutService,
  BraintreePaymentDetailsService,
  ApplePayCheckoutService, BraintreeSrcCheckoutService
} from 'braintree-spartacus-core';
import {BraintreeCheckoutConfigService} from '../../../services/braintree-checkout-config.service';
import {
  OCC_USER_ID_ANONYMOUS,
  UserIdService
} from '@spartacus/core';

@Component({
  selector: 'bt-cart',
  templateUrl: './braintree-cart.component.html'
})
export class BraintreeCartComponent implements OnInit {

  @ViewChild('paypalButtonContainer') paypalButtonContainer: ElementRef;
  @ViewChild('applePayButtonContainer') applePayButtonContainer: ElementRef;

  loadedCheckoutData: Observable<CheckoutData>;
  pageType = PageType.CART;

  constructor(
    protected paypalService: PaypalCheckoutService,
    protected applePayService: ApplePayCheckoutService,
    protected googlePayService: GooglePayCheckoutService,
    protected srcCheckoutService: BraintreeSrcCheckoutService,
    protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
    protected braintreeCheckoutServ: BraintreeCheckoutConfigService,
    protected userIdService: UserIdService
) { }

  ngOnInit(): void {
      this.loadedCheckoutData = combineLatest(
        [this.braintreePaymentDetailsService.getPayPalButtonStyles('cart'),
          this.braintreePaymentDetailsService.loadPaymentDetails('full', PageType.CART)])
        .pipe(
          map(
            ([buttonStyle, checkoutData]) => {
              this.initializeCartPage(checkoutData, buttonStyle);
              return checkoutData;
            }));
  }

  isPaymentAllowed(): boolean{
    return !this.isAnonymousUser() || this.braintreeCheckoutServ.isAllowedGuestExpressCheckout();
  }

  initializeCartPage(checkoutData: CheckoutData, buttonStyle: PayPalButtonStyle): void {
    if (this.isPaymentAllowed() && checkoutData.payPalPaymentMethod.payPalExpressEnabled) {
      this.paypalService.initializePayPalCheckout(
        this.paypalButtonContainer.nativeElement,
        checkoutData,
        buttonStyle,
        this.pageType
      );
    } else{
      this.paypalButtonContainer.nativeElement.remove();
    }
    if (this.isPaymentAllowed() && checkoutData.googlePayPaymentMethod.googlePayEnabled) {
      this.googlePayService.loadGooglePay(() => {
        this.googlePayService.initialiseGooglePay(checkoutData, this.pageType);
      });
    }
    if (this.isPaymentAllowed() && checkoutData.srcPaymentMethod.srcEnabled) {
      this.srcCheckoutService.initializeSrc(checkoutData, this.pageType);
    }
    if (this.isPaymentAllowed() && checkoutData.applePayPaymentMethod.applePayEnabled === 'true'
          && this.applePayService.isAvailableApplePay()) {
      this.applePayService.initalizeApplePay(checkoutData, this.pageType, this.applePayButtonContainer.nativeElement);
    } else {
      this.applePayButtonContainer.nativeElement.remove();
    }
  }

  isAnonymousUser(): boolean{
    let userId;
    this.userIdService
        .getUserId()
        .subscribe((occUserId) => (userId = occUserId))
        .unsubscribe();
    return OCC_USER_ID_ANONYMOUS === userId;
  }
}
