import {
  ConverterService, OCC_USER_ID_ANONYMOUS,
  OccEndpointsService,
  Order,
  PAYMENT_DETAILS_NORMALIZER,
  PaymentDetails
} from '@spartacus/core';
import {Injectable} from '@angular/core';
import {HttpClient, HttpHeaders, HttpParams} from '@angular/common/http';
import {Observable} from 'rxjs';
import {
  ApplePayPaymentRequest,
  BraintreeCreditCardPaymentDetails,
  BraintreePaymentDetails,
  Fields,
  GooglePayPaymentRequest,
  LocalPaymentRequest, PageType,
  PayPalPaymentRequest, SrcPaymentRequest,
  VenmoPaymentData
} from '../../braintree-checkout/models';

import {PaymentDetailsAdapter} from '../../braintree-checkout/connectors';
import {
  APPLE_PAY_PAYMENT_DETAILS_SERIALIZER,
  BRAINTREE_PAYMENT_DETAILS_NORMALIZER,
  CREDIT_CARD_PAYMENT_DETAILS_SERIALIZER,
  GOOGLE_PAY_PAYMENT_DETAILS_SERIALIZER,
  LOCALE_PAYMENT_METHOD_PAYMENT_DETAILS_SERIALIZER,
  PAYPAL_PAYMENT_DETAILS_SERIALIZER, SRC_PAYMENT_DETAILS_SERIALIZER,
  VENMO_PAYMENT_DETAILS_SERIALIZER
} from '../../braintree-checkout/braintree-checkout-store/converters/converters';
import {BraintreeOcc} from '../models/occ-payment-method.model';

@Injectable()
export class BraintreePaymentDetailsAdapter implements PaymentDetailsAdapter
{
  constructor(
    protected http: HttpClient,
    protected occEndpoints: OccEndpointsService,
    protected converter: ConverterService
  ) {}

  protected getEndpoint(userId?: string): string{
    return this.occEndpoints.getEndpoint('/users/' + userId + '/braintree/paymentInfo');
  }

  public saveVenmoPaymentDetails(
    userId: string,
    selectedAddressCode: string,
    venmoPayment: VenmoPaymentData,
    cartId?: string,
    shouldBeSaved?: boolean,
    deviceData?: string,
    fields?: Fields
  ): Observable<PaymentDetails>{

  const  occVenmoPayment: BraintreeOcc.VenmoPaymentData = this.converter.convert(venmoPayment, VENMO_PAYMENT_DETAILS_SERIALIZER);

  const url = this.getEndpoint(userId) + '/venmo';

  let params: HttpParams = new HttpParams()
      .set('shouldBeSaved', shouldBeSaved.toString())
      .set('selectedAddressCode', selectedAddressCode)
      .set('fields', fields);

  if (deviceData !== undefined){
    params = params.set('deviceData', deviceData);
    }

    if (cartId !== undefined){
      params = params.set('cartGuid', cartId);
    }

  const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

  return this.http.post(url, occVenmoPayment, {headers, params})
      .pipe(this.converter.pipeable(PAYMENT_DETAILS_NORMALIZER));

  }

  public savePayPalPaymentDetails(
    credit: boolean,
    payPalRequest: PayPalPaymentRequest,
    pageType: PageType,
    userId: string,
    cartId?: string,
    shouldBeSaved?: boolean,
    deviceData?: string,
    fields?: Fields,
    fundingSource?: string
  ): Observable<PaymentDetails>{

    payPalRequest = this.converter.convert(payPalRequest, PAYPAL_PAYMENT_DETAILS_SERIALIZER);
    const url = `${this.getEndpoint(
      userId)}/payPal`;

    let params: HttpParams = new HttpParams().set('shouldBeSaved', shouldBeSaved.toString())
      .set('credit', credit.toString())
      .set('fields', fields)
      .set('pageType', pageType);

    if (deviceData !== undefined){
      params = params.set('deviceData', deviceData);
    }

    if (cartId !== undefined){
      params = params.set('cartGuid', cartId);
    }

    if (fundingSource !== undefined){
      params = params.set('fundingSource', fundingSource);
    }

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url, payPalRequest, {headers, params})
      .pipe(this.converter.pipeable(PAYMENT_DETAILS_NORMALIZER));
  }

  public saveGooglePayPaymentDetails(
    userId: string,
    googlePayRequest: GooglePayPaymentRequest,
    shouldBeSaved: boolean,
    cartId?: string,
    deviceData?: string,
    fields?: Fields
  ): Observable<PaymentDetails>{

    googlePayRequest = this.converter.convert(googlePayRequest, GOOGLE_PAY_PAYMENT_DETAILS_SERIALIZER);

    const url = `${this.getEndpoint(userId)}/googlePay`;

    let params: HttpParams = new HttpParams().set('shouldBeSaved', shouldBeSaved.toString())
      .set('fields', fields);

    if (cartId !== undefined){
        params = params.set('cartGuid', cartId);
    }

    if (deviceData !== undefined){
      params = params.set('deviceData', deviceData);
    }

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url, googlePayRequest, {headers, params})
      .pipe(this.converter.pipeable(PAYMENT_DETAILS_NORMALIZER));
  }

  saveSrcPaymentDetails(
      userId: string,
      srcRequest: SrcPaymentRequest,
      shouldBeSaved: boolean,
      cartId?: string,
      deviceData?: string,
      fields?: Fields
  ): Observable<PaymentDetails> {

    srcRequest = this.converter.convert(srcRequest, SRC_PAYMENT_DETAILS_SERIALIZER);

    const url = `${this.getEndpoint(userId)}/srcPayment`;

    let params: HttpParams = new HttpParams().set('shouldBeSaved', shouldBeSaved.toString())
    .set('fields', fields);

    if (cartId !== undefined){
      params = params.set('cartGuid', cartId);
    }

    if (deviceData !== undefined){
      params = params.set('deviceData', deviceData);
    }

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url, srcRequest, {headers, params})
    .pipe(this.converter.pipeable(PAYMENT_DETAILS_NORMALIZER));
  }



  public saveCreditCardPaymentDetails(
    userId: string,
    paymentDetails: BraintreeCreditCardPaymentDetails,
    selectedAddressCode: string,
    cartId?: string,
    shouldBeSaved?: boolean,
    deviceData?: string,
    fields?: Fields
  ): Observable<PaymentDetails>{

    paymentDetails = this.converter.convert(paymentDetails, CREDIT_CARD_PAYMENT_DETAILS_SERIALIZER);

    const url = `${this.getEndpoint(userId)}/creditCard`;

    let params: HttpParams = new HttpParams().set('shouldBeSaved', shouldBeSaved.toString())
      .set('selectedAddressCode', selectedAddressCode)
      .set('fields', fields);

    if (deviceData !== undefined){
      params = params.set('deviceData', deviceData);
    }

    if (cartId !== undefined){
      params = params.set('cartGuid', cartId);
    }

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url, paymentDetails, {headers, params})
      .pipe(this.converter.pipeable(PAYMENT_DETAILS_NORMALIZER));
  }

  public saveApplePayPaymentDetails(
    userId: string,
    applePayRequest: ApplePayPaymentRequest,
    cartId?: string,
    deviceData?: string,
    fields?: Fields
  ): Observable<PaymentDetails>{

    applePayRequest = this.converter.convert(applePayRequest, APPLE_PAY_PAYMENT_DETAILS_SERIALIZER);

    const url = `${this.getEndpoint(userId)}/applePay`;

    let params: HttpParams = new HttpParams()
      .set('fields', fields);

    if (deviceData !== undefined){
      params = params.set('deviceData', deviceData);
    }

    if (cartId !== undefined){
      params = params.set('cartGuid', cartId);
    }

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url, applePayRequest, {headers, params})
      .pipe(this.converter.pipeable(PAYMENT_DETAILS_NORMALIZER));
  }

  public processLocalPayment(
    userId: string,
    cartId: string,
    localPaymentRequest: LocalPaymentRequest,
    deviceData?: string,
    fields?: Fields
  ): Observable<Order>{
    localPaymentRequest = this.converter.convert(localPaymentRequest, LOCALE_PAYMENT_METHOD_PAYMENT_DETAILS_SERIALIZER);

    const url = this.occEndpoints.getEndpoint('/users/' + userId + '/carts/' + cartId + '/braintree/paymentInfo/localPayment');

    let params: HttpParams = new HttpParams()
      .set('fields', fields);

    if (deviceData !== undefined){
      params = params.set('deviceData', deviceData);
    }

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url, localPaymentRequest, {headers, params});
  }

  public savePaymentIdForLPM(
    userId: string,
    cartId: string,
    paymentId: string
  ): Observable<{}> {

    const url = this.occEndpoints
      .getEndpoint('/users/' + userId + '/carts/' + cartId + '/braintree/localPaymentMethods/savePaymentId');

    const params: HttpParams = new HttpParams()
      .set('paymentId', paymentId);

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url,  {}, {headers, params});

  }

  public processLPMFallback(
    localPaymentRequest: LocalPaymentRequest,
    currencyFromFallbackURL: string,
    deviceData?: string,
    fields?: Fields,
  ): Observable<Order> {

    const url = this.occEndpoints
      .getEndpoint('/braintree/localPaymentMethods/processFallback');

    const params: HttpParams = new HttpParams()
        .set('deviceData', deviceData)
        .set('currency', currencyFromFallbackURL)
        .set('fields', fields);

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.post(url, localPaymentRequest, {headers, params});
  }

  public getBraintreePaymentDetails(
    userId: string,
    saved: boolean,
    fields?: Fields
  ): Observable<BraintreePaymentDetails[]> {

    const url = this.occEndpoints.getEndpoint('/users/' + userId + '/braintree');


    const params: HttpParams = new HttpParams()
    .set('saved', saved.toString())
    .set('fields', fields);

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.get(url, {headers, params})
    .pipe(this.converter.pipeableMany(BRAINTREE_PAYMENT_DETAILS_NORMALIZER));
  }

  public getBraintreePaymentDetailsById(
    userId: string,
    paymentId: string,
    fields?: Fields
  ): Observable<BraintreePaymentDetails> {

    const url = this.occEndpoints.getEndpoint('/users/' + userId + '/braintree/' + paymentId);


    const params: HttpParams = new HttpParams()
    .set('fields', fields);

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.get(url, {headers, params})
    .pipe(this.converter.pipeable(BRAINTREE_PAYMENT_DETAILS_NORMALIZER));
  }

  public selectBraintreePaymentMethod(
    userId: string,
    cartId: string,
    selectedPaymentMethodId: string,
    selectedPaymentMethodNonce: string
  ): Observable<{}> {

    const url = this.occEndpoints.getEndpoint('/users/' + userId + '/carts/' + cartId + '/braintree/choose-cc');

    const params: HttpParams = new HttpParams()
    .set('selectedPaymentMethodId', selectedPaymentMethodId)
    .set('selectedPaymentMethodNonce', selectedPaymentMethodNonce);

    const headers = new HttpHeaders({
      'Content-Type': 'application/json',
    });

    return this.http.put(url, {}, {headers, params});
  }

}
