import {Injectable} from '@angular/core';
import * as braintree from 'braintree-web';
import {PageType} from '../../models/braintree-payment-methods.model';
import {CheckoutData} from '../../models/braintree-payment-data.model';
import {BraintreeUtilsService} from '../utils/braintree-utils.service';
import {BraintreeCheckoutService} from '../checkout/braintree-checkout.service';

@Injectable({
  providedIn: 'root'
})
export class GooglePayCheckoutService {

  protected googlePaymentsClient;
  protected googlePayPaymentInstance;
  protected googleButtonType: PageType;
  protected checkoutData: CheckoutData;
  protected deviceData: string;

  constructor(
    protected braintreeUtils: BraintreeUtilsService,
    protected braintreeCheckout: BraintreeCheckoutService
  ) {
  }

  initialiseGooglePay(checkoutData: CheckoutData, pageType: PageType): void {
    this.googleButtonType = pageType;
    this.checkoutData = checkoutData;

    this.getGooglePaymentClient(checkoutData.configurationData.environment);

    this.braintreeUtils.createClientInstance(checkoutData.configurationData, (client, deviceData) => {
      this.deviceData = deviceData;
      this.processGooglePayPayment(client);
    });
  }

  private processGooglePayPayment(client): void {
    const paymentsClient = this.googlePaymentsClient;

    braintree.googlePayment.create({
      client,
      googlePayVersion: 2,
      googleMerchantId: this.checkoutData.googlePayPaymentMethod.googleMerchantId
    }).then((googlePaymentInstance) => {
      return paymentsClient.isReadyToPay({
        // see https://developers.google.com/pay/api/web/reference/object#IsReadyToPayRequest for all options
        apiVersion: 2,
        apiVersionMinor: 0,
        allowedPaymentMethods: googlePaymentInstance.createPaymentDataRequest().allowedPaymentMethods,
        existingPaymentMethodRequired: true
      }).then((isReadyToPay) => {
        if (isReadyToPay.result) {
          this.googlePayPaymentInstance = googlePaymentInstance;
          this.renderGooglePayButton(paymentsClient, googlePaymentInstance);

        } else {
          console.log('result false');
        }
      }).catch((err) => {
        // Handle creation errors
        console.error('An error occurred during creation:', err.message);
      });
    });
  }

  private renderGooglePayButton(paymentsClient, googlePaymentInstance): void {
    const buttonContainers = document.getElementsByClassName('google_pay_container');
    const button = paymentsClient
      .createButton({
        onClick: () => this.onGooglePaymentButtonClicked(paymentsClient, googlePaymentInstance),
        buttonSizeMode: 'fill'
      });
    Array.from(buttonContainers).forEach((item) => {
      if (!item.hasChildNodes()) {
        item.appendChild(button);
      }
    });
  }

  private onGooglePaymentButtonClicked(paymentsClient, googlePayPaymentInstance): void {
    const paymentDataRequest = this.getGooglePaymentDataRequest(googlePayPaymentInstance);

    // See all available options at https://developers.google.com/pay/api/web/reference/object
    const cardPaymentMethod = paymentDataRequest.allowedPaymentMethods[0];
    paymentDataRequest.emailRequired = true;
    cardPaymentMethod.parameters.billingAddressRequired = true;
    cardPaymentMethod.parameters.billingAddressParameters = {
      format: 'FULL',
      phoneNumberRequired: true
    };

    if (PageType.CART === this.googleButtonType) {
      paymentDataRequest.shippingAddressRequired = true;
    }

    let googlePayPaymentData;

    paymentsClient.loadPaymentData(paymentDataRequest).then((paymentData) => {
      googlePayPaymentData = paymentData;
      return googlePayPaymentInstance.parseResponse(paymentData);
    }).then((result) => {
      this.braintreeCheckout.processGooglePayResponse(
        result,
        googlePayPaymentData,
        this.deviceData,
        this.googleButtonType
      );
    }).catch((err) => {
      // Handle errors
      console.error('An error occurred:', err.message);
    });
  }

  private getGooglePaymentClient(environment: string): void {
    if (typeof this.googlePaymentsClient === 'undefined' || this.googlePaymentsClient === '') {
      const localEnvironment = environment.toUpperCase() === 'PRODUCTION' ? 'PRODUCTION' : 'TEST';
      this.googlePaymentsClient = new (window as any).google.payments.api.PaymentsClient({
        environment: localEnvironment
      });
    }
  }

  private getGooglePaymentDataRequest(googlePayPaymentInstance): any {
    let paymentDataRequest;
    if (this.googleButtonType === PageType.MY_ACCOUNT) {
      paymentDataRequest = googlePayPaymentInstance.createPaymentDataRequest({
        transactionInfo: {
          countryCode: this.checkoutData.googlePayPaymentMethod.googlePayCountryCode,
          currencyCode: this.checkoutData.configurationData.currency,
          totalPriceStatus: 'FINAL',
          totalPrice: '0.0'
        }
      });
    } else {
      paymentDataRequest = googlePayPaymentInstance.createPaymentDataRequest({
        transactionInfo: {
          countryCode: this.checkoutData.googlePayPaymentMethod.googlePayCountryCode,
          currencyCode: this.checkoutData.configurationData.currency,
          totalPriceStatus: 'FINAL',
          totalPrice: JSON.stringify(this.braintreeUtils.getTotalAmountBeforeRendering())
        }
      });
    }
    return paymentDataRequest;
  }

  loadGooglePay(onload?: Function): void{
    if ((window as any).google === 'undefined' || (window as any).google === undefined){
      this.braintreeUtils.loadSdk('https://pay.google.com/gp/p/js/pay.js', onload);
    }else if (onload !== undefined){
      onload();
    }
  }
}
