import { HttpRequest, HttpErrorResponse } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { Router } from '@angular/router';
import {
  BadRequestHandler,
  GlobalMessageService,
  GlobalMessageType,
  HttpErrorHandler,
  HttpResponseStatus,
  Priority
} from '@spartacus/core';

@Injectable({
  providedIn: 'root',
})
export class BraintreeErrorHandlerService extends BadRequestHandler {
  responseStatus = HttpResponseStatus.BAD_REQUEST;

  protected countryError = 'No country with the code';
  protected regionError = 'No region with the code';

  constructor(
    protected globalMessageService: GlobalMessageService,
    protected router: Router
    ){
    super(globalMessageService);
  }

  handleError(
    request: HttpRequest<any>,
    errorResponse: HttpErrorResponse
  ): void {
    super.handleError(request, errorResponse);
    if (errorResponse.error?.errors?.length > 0) {
      errorResponse.error?.errors.forEach((error) => {
        if (error.type === 'PaymentAuthorizationError') {
          this.globalMessageService.add(
            { key: 'error.authorization.failed' },
            GlobalMessageType.MSG_TYPE_ERROR
          );
        }else if (error.type === 'ConversionError' && this.checkErrorMessage(error.message)){
          this.globalMessageService.add(
            { key: 'error.address.provide' },
            GlobalMessageType.MSG_TYPE_WARNING
          );
          this.router.navigate(['/checkout/shipping-address']);
        } else if (error.type === 'BraintreeDeletePaymentDetailsWithNotCapturedOrderError') {
          this.globalMessageService.add(
              { key: 'error.braintree_delete_payment_details_with_not_captured_order' },
              GlobalMessageType.MSG_TYPE_ERROR
          );
        }
      });
    }
  }

  checkErrorMessage(message: string): boolean {
    return message.includes(this.regionError) || message.includes(this.countryError);
  }

  getPriority(): Priority {
    return Priority.NORMAL;
  }
}
