import {Injectable} from '@angular/core';
import {
  ActiveCartService,
  Address,
  CheckoutActions,
  CheckoutConnector,
  CheckoutDetails,
  MultiCartService,
  PaymentDetails,
  StateWithCheckout,
  UserIdService
} from '@spartacus/core';
import {Router} from '@angular/router';
import {BraintreeCheckoutConnector} from '../../connectors';
import {first, map} from 'rxjs/operators';
import {Observable} from 'rxjs';
import {Store} from '@ngrx/store';
import {PriceValueData} from '../../models';

@Injectable({
  providedIn: 'root',
})
export class BraintreeExpressCheckoutService {

  constructor(
      protected router: Router,
      protected activeCartService: ActiveCartService,
      protected userIdService: UserIdService,
      protected braintreeCheckoutConnector: BraintreeCheckoutConnector,
      protected multiCartService: MultiCartService,
      protected checkoutConnector: CheckoutConnector,
      protected checkoutStore: Store<StateWithCheckout>
  ) {
  }

  processExpressCheckout(
      address: Address,
      paymentDetails: PaymentDetails
  ): void {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();

    let cartId;
    this.activeCartService
    .getActiveCartId()
    .subscribe((activeCartId) => (cartId = activeCartId))
    .unsubscribe();

    if (userId && cartId) {
      this.braintreeCheckoutConnector.proceedExpressCheckout(paymentDetails.id, userId, cartId, address)
      .subscribe(value => {
        this.loadCheckoutDetails(userId, cartId).pipe(first())
        .subscribe(data => {
          this.multiCartService.reloadCart(cartId);
          this.router.navigate(['/checkout/review-order']);
        });
      });
    }
  }

  countDeliveryPrice(
      countryCode: string
  ): Observable<PriceValueData> {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();

    let cartId;
    this.activeCartService
    .getActiveCartId()
    .subscribe((activeCartId) => (cartId = activeCartId))
    .unsubscribe();

    if (userId && cartId) {
      return this.braintreeCheckoutConnector.countDeliveryPrice(countryCode, userId, cartId);
    }
  }

  private loadCheckoutDetails(userId: string, cartId: string): Observable<CheckoutDetails> {
    return this.checkoutConnector.loadCheckoutDetails(userId, cartId)
    .pipe(
        map(
            (data: CheckoutDetails) => {
              this.checkoutStore.dispatch(new CheckoutActions.LoadCheckoutDetailsSuccess(data));
              return data;
            }
        )
    );
  }

}
