import {ChangeDetectorRef, Component, ElementRef, EventEmitter, Input, OnInit, Output, ViewChild} from '@angular/core';
import {BraintreeDropInService, BraintreePaymentDetailsService, CheckoutData, PageType} from 'braintree-spartacus-core';
import {BehaviorSubject, combineLatest, Observable} from 'rxjs';
import {map, switchMap, tap} from 'rxjs/operators';
import {
  Address,
  CheckoutDeliveryService,
  Country, GlobalMessageService,
  GlobalMessageType,
  Region,
  UserAddressService,
  UserPaymentService
} from '@spartacus/core';
import {Card} from '@spartacus/storefront';
import {FormBuilder, FormGroup, Validators} from '@angular/forms';

@Component({
  selector: 'bt-drop-in',
  templateUrl: './braintree-drop-in.component.html'
})
export class BraintreeDropInComponent implements OnInit {

  @ViewChild('dropInButtonContainer') dropInButton: ElementRef;
  @ViewChild('dropInSubmitButton') dropInSubmitButton: ElementRef;

  @Input()
  pageType = PageType.BILLING;

  @Input()
  paymentMethodsCount: number;

  @Output()
  backToPaymentMethods = new EventEmitter<any>();

  loadedCheckoutData$: Observable<CheckoutData>;
  shippingAddress$: Observable<Address>;
  checkoutData: CheckoutData;
  hideButtons = true;
  sameAsShippingAddress = true;
  isCreditCard = false;
  isDropInError = false;

  billingAddressForm: FormGroup = this.fb.group({
    firstName: ['', Validators.required],
    lastName: ['', Validators.required],
    line1: ['', Validators.required],
    line2: [''],
    town: ['', Validators.required],
    region: this.fb.group({
      isocodeShort: [null, Validators.required],
    }),
    country: this.fb.group({
      isocode: [null, Validators.required],
    }),
    postalCode: ['', Validators.required],
  });

  regions$: Observable<Region[]>;
  countries$: Observable<Country[]>;
  selectedCountry$: BehaviorSubject<string> = new BehaviorSubject<string>('');

  constructor(
      protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
      protected braintreeDropInService: BraintreeDropInService,
      protected checkoutDeliveryService: CheckoutDeliveryService,
      protected fb: FormBuilder,
      protected userPaymentService: UserPaymentService,
      protected userAddressService: UserAddressService,
      protected globalMessageService: GlobalMessageService,
      private cdr: ChangeDetectorRef
  ) {
  }

  ngOnInit(): void {
    this.shippingAddress$ = this.checkoutDeliveryService.getDeliveryAddress();
    this.loadedCheckoutData$ = combineLatest(
        [this.braintreeDropInService.getDropInButtonStyle('Billing'),
          this.braintreePaymentDetailsService.loadPaymentDetailsForBillingPage('full', this.pageType)])
    .pipe(
        map(([dropInButtonStyle, checkoutData]) => {
          this.checkoutData = checkoutData;
          if (checkoutData.payPalPaymentMethod.payPalStandardEnabled
              || checkoutData.googlePayPaymentMethod.googlePayEnabled
              || (checkoutData.applePayPaymentMethod.applePayEnabled === 'true')
              || checkoutData.creditCardPaymentMethod.hostedFieldsEnable
              || checkoutData.venmoPaymentMethod.venmoEnabled) {
            this.braintreeDropInService.initializeDropIn(this.dropInButton.nativeElement, this.dropInSubmitButton.nativeElement,
                checkoutData, dropInButtonStyle, this.pageType, (dropInInstance, dropInCreatingError) => {

                  if (dropInCreatingError) {
                    this.isDropInError = true;
                    this.hideButtons = true;
                    this.cdr.detectChanges();
                    return;
                  }

                  dropInInstance.on('changeActiveView', (event) => {
                    const activePaymentMethod = dropInInstance._model._activePaymentMethod;
                    this.isCreditCard = (event.newViewId === 'methods' && activePaymentMethod !== undefined
                        && activePaymentMethod.type === 'CreditCard' && !dropInInstance._model.isInEditMode()) || event.newViewId === 'card';
                    this.cdr.detectChanges();
                  });

                  dropInInstance.on('paymentMethodRequestable', () => {
                    const activePaymentMethod = dropInInstance._model._activePaymentMethod;
                    this.isCreditCard = dropInInstance._model._activePaymentViewId === 'card' || 
                        (activePaymentMethod !== undefined && activePaymentMethod.type === 'CreditCard');
                    this.cdr.detectChanges();
                  });

                  if (dropInInstance._model._activePaymentViewId === 'card'
                      || (dropInInstance._model._activePaymentMethod !== undefined
                          && dropInInstance._model._activePaymentMethod.type === 'CreditCard')) {
                    this.isCreditCard = true;
                    this.cdr.detectChanges();
                  }

                },
                () => {
                  if (this.sameAsShippingAddress) {
                    return undefined;
                  } else {
                    if (this.billingAddressForm.valid) {
                      return this.billingAddressForm.value;
                    } else {
                      this.billingAddressForm.markAllAsTouched();
                      this.cdr.detectChanges();
                      return false;
                    }
                  }
                });
            this.hideButtons = false;
          }
          return checkoutData;
        })
    );

    this.countries$ = this.userPaymentService.getAllBillingCountries().pipe(
        tap((countries) => {
          // If the store is empty fetch countries. This is also used when changing language.
          if (Object.keys(countries).length === 0) {
            this.userPaymentService.loadBillingCountries();
          }
        })
    );

    this.regions$ = this.selectedCountry$.pipe(
        switchMap((country) => this.userAddressService.getRegions(country)),
        tap((regions) => {
          const regionControl = this.billingAddressForm.get(
              'region.isocodeShort'
          );
          if (regions.length > 0) {
            regionControl.enable();
          } else {
            regionControl.disable();
          }
        })
    );

  }

  back(): void {
    this.backToPaymentMethods.emit();
  }

  isBillingPage(): boolean {
    return this.pageType === PageType.BILLING;
  }

  isBackButtonAvailable(): boolean {
    return !this.isBillingPage();
  }

  getAddressCardContent(address: Address): Card {
    let region = '';
    if (address.region && address.region.isocode) {
      region = address.region.isocode + ', ';
    }

    return {
      textBold: address.firstName + ' ' + address.lastName,
      text: [
        address.line1,
        address.line2,
        address.town + ', ' + region + address.country.isocode,
        address.postalCode,
        address.phone,
      ],
    };
  }

  countrySelected(country: Country): void {
    this.billingAddressForm.get('country.isocode').setValue(country.isocode);
    this.selectedCountry$.next(country.isocode);
  }

  toggleSameAsShippingAddress(): void {
    this.sameAsShippingAddress = !this.sameAsShippingAddress;
  }

  private showErrorMessage(message): void {
    this.globalMessageService.add(
        {
          key: message,
        },
        GlobalMessageType.MSG_TYPE_ERROR
    );
  }

  areThereAnyPaymentMethodsAvailable(checkoutData: any): boolean {
    return !!(checkoutData.payPalPaymentMethod.payPalStandardEnabled || checkoutData.googlePayPaymentMethod.googlePayEnabled
        || (checkoutData.applePayPaymentMethod.applePayEnabled === 'true') || checkoutData.creditCardPaymentMethod.hostedFieldsEnable
        || checkoutData.venmoPaymentMethod.venmoEnabled);

  }
}
