import {Injectable} from '@angular/core';
import {HttpClient, HttpParams} from '@angular/common/http';
import {Observable} from 'rxjs';
import {BraintreeAssets, CheckoutData, DropInButtonStyle, PayPalButtonStyle} from '../../braintree-checkout/models';
import {ConverterService, OccEndpointsService} from '@spartacus/core';
import {PaymentAdapter} from '../../braintree-checkout/connectors';
import {
  BUTTON_STYLE_NORMALIZER,
  CHECKOUT_DATA_NORMALIZER, DROP_IN_STYLE_NORMALIZER
} from '../../braintree-checkout/braintree-checkout-store/converters/converters';

@Injectable()
export class BraintreePaymentAdapter
  implements PaymentAdapter
{
  constructor(
    protected http: HttpClient,
    protected occEndpoints: OccEndpointsService,
    protected converter: ConverterService
  ) {}

  protected getEndpoint(userId): string{
     return this.occEndpoints.getEndpoint('/users/' + userId + '/braintree/payment');
  }

  public getPayPalButtonStyles(
    userId: string,
    buttonType?: string
  ): Observable<PayPalButtonStyle>{
    const params = new HttpParams()
      .set('buttonType', buttonType);
    const url = `${this.getEndpoint(userId)}/button`;

    return this.http.get( url, { params })
      .pipe(this.converter.pipeable(BUTTON_STYLE_NORMALIZER));
  }

  public getDropInButtonStyle(
      userId: string,
      buttonType?: string
  ): Observable<DropInButtonStyle>{
    const params = new HttpParams()
        .set('buttonType', buttonType);
    const url = `${this.getEndpoint(userId)}/dropInButton`;

    return this.http.get( url, { params })
        .pipe(this.converter.pipeable(DROP_IN_STYLE_NORMALIZER));
  }

  public getBraintreeAssets(): Observable<BraintreeAssets>{
    const url = this.occEndpoints.getEndpoint(  '/braintree/payment/assets');

    return this.http.get(url);
  }

  public loadPaymentDetails(
    paymentMethod: string,
    userId: string,
    cartId?: string,
    pageType?: string
  ): Observable<CheckoutData> {
    let params = new HttpParams()
      .set('paymentMethod', paymentMethod);
    if (cartId !== undefined){
      params = params.set('cartGuid', cartId);
    }

    if (pageType !== undefined){
      params = params.set('pageType', pageType);
    }

    return this.http.get( this.getEndpoint(userId), { params })
      .pipe(this.converter.pipeable(CHECKOUT_DATA_NORMALIZER));
  }
}
