import {Injectable} from '@angular/core';
import * as braintree from 'braintree-web';
import {
  VAULT_FLOW,
  ON_SUCCESS,
  TRUE,
  FALSE,
  CHECKOUT_FLOW,
  INTENT_AUTHORIZE,
  INTENT_CAPTURE,
  INTENT_ORDER,
  INTENT_SALE,
  HOSTED_FIELDS_FIELDS_EMPTY,
  HOSTED_FIELDS_FIELDS_INVALID,
  HOSTED_FIELDS_FAILED_TOKENIZATION,
  HOSTED_FIELDS_TOKENIZATION_NETWORK_ERROR,
} from '../../models/braintree.constants';
import {PageType} from '../../models/braintree-payment-methods.model';
import {CheckoutData} from '../../models/braintree-payment-data.model';
import {PayPalConfiguration} from '../../models/braintree-payment-data.model';

import {ActiveCartService, GlobalMessageService, GlobalMessageType, OccConfig} from '@spartacus/core';
import {BraintreePaymentMethodsErrorHandlerService} from '../errorHandling';

export interface BaseOccUrlProperties {
  baseUrl?: boolean;
  prefix?: boolean;
  baseSite?: boolean;
}

@Injectable({
  providedIn: 'root',
})
export class BraintreeUtilsService {

  constructor(
      protected globalMessageService: GlobalMessageService,
      protected activeCartService: ActiveCartService,
      protected config: OccConfig,
      protected braintreeErrorHandlerService: BraintreePaymentMethodsErrorHandlerService
  ) {
  }

  createClientInstance(configurationData: PayPalConfiguration, onProcess): any {
    if (
        typeof configurationData.client_id !== 'undefined' &&
        configurationData.client_id !== '' &&
        typeof braintree !== 'undefined'
    ) {
      braintree.client.create(
          {
            authorization: configurationData.client_id,
          },
          (clientErr, clientInstance) => {
            if (clientErr) {
              console.log(clientErr);
              return;
            }
            if (typeof configurationData.advancedFraudTools !== 'undefined') {
              this.createDataCollector(
                  {
                    client: clientInstance,
                    paypal: JSON.parse(configurationData.advancedFraudTools),
                  },
                  (deviceData) => onProcess(clientInstance, deviceData)
              );
            }
          }
      );
    } else {
      this.globalMessageService.add(
          {key: 'error.authorization.client'},
          GlobalMessageType.MSG_TYPE_ERROR
      );
    }
  }

  loadSdk(url: string, onload?: Function): void {
    const body = document.body as HTMLDivElement;
    const script = document.createElement('script');
    script.innerHTML = '';
    script.src = url;
    script.async = false;
    script.defer = true;
    script.onload = () => {
      if (onload) {
        onload();
      }
    };
    body.appendChild(script);
  }

  loadPayPalSDK(
      checkoutData: CheckoutData,
      paypalCheckoutInstance,
      paypalIntent,
      commit,
      flow,
      pageType: PageType,
      onLoad,
      amount?: number
  ): any {
    paypalCheckoutInstance.getClientId().then((id) => {
      const vault = flow === VAULT_FLOW;
      const importRequest: { [k: string]: any } = {};
      if (PageType.MY_ACCOUNT === pageType) {
        importRequest.intent = 'tokenize';
      } else {
        importRequest.intent =
            paypalIntent === 'sale' || vault ? 'capture' : paypalIntent;
      }
      let disableFunding = !checkoutData.payPalPaymentMethod.creditEnabled
      || PageType.MY_ACCOUNT === pageType
          ? 'credit'
          : '';
      importRequest['client-id'] = id;
      importRequest.commit = commit;
      importRequest.components = 'buttons,messages';
      if (checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton && amount) {
        importRequest.dataAttributes = {
          amount
        };
      }

      if (vault) {
        importRequest.vault = vault;
      }
      if (
          typeof checkoutData.payPalPaymentMethod.disableFunding ===
          'undefined'
      ) {
        disableFunding += disableFunding !== '' ? ',card' : 'card';
      } else if (checkoutData.payPalPaymentMethod.disableFunding !== '') {
        disableFunding +=
            disableFunding !== ''
                ? ',' + checkoutData.payPalPaymentMethod.disableFunding
                : checkoutData.payPalPaymentMethod.disableFunding;
      }
      if (disableFunding !== '') {
        importRequest['disable-funding'] = disableFunding;
      }
      if (checkoutData.payPalPaymentMethod.isPayLaterForNonUsCountriesEnabled) {
        importRequest['enable-funding'] = 'paylater';
      }
      if (
          typeof checkoutData.configurationData.currency !== 'undefined' &&
          checkoutData.configurationData.currency !== ''
      ) {
        importRequest.currency = checkoutData.configurationData.currency;
      }
      if (
          typeof checkoutData.configurationData.braintreeLocale !== 'undefined' &&
          checkoutData.configurationData.braintreeLocale !== ''
      ) {
        importRequest.locale = checkoutData.configurationData.braintreeLocale;
      }

      const dataAttributes = {};
      dataAttributes['data-namespace'] = 'paypalSdk';

      importRequest.dataAttributes = dataAttributes;

      paypalCheckoutInstance.loadPayPalSDK(importRequest, () => {
        onLoad();
      });
    });
  }

  createDataCollector(configObj, onLoadedDeviceData): any {
    return braintree.dataCollector.create(
        configObj,
        (error, dataCollectorInstance) => {
          if (error) {
            console.log(error);
            return;
          } else {
            onLoadedDeviceData(dataCollectorInstance.deviceData);
          }
        }
    );
  }

  getStoreInVault(storeInVault: string): boolean {
    return (
        storeInVault !== '' &&
        storeInVault !== undefined &&
        (storeInVault === ON_SUCCESS || storeInVault === TRUE)
    );
  }

  getPaypalFlow(checkoutData: CheckoutData, pageType?: PageType): string {
    if (checkoutData.configurationData.intent === INTENT_ORDER) {
      checkoutData.configurationData.storeInVault = FALSE;
    }
    if (PageType.MY_ACCOUNT === pageType) {
      return VAULT_FLOW;
    }
    if (
        this.getStoreInVault(checkoutData.configurationData.storeInVault) &&
        !checkoutData.payPalPaymentMethod.creditEnabled && !checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton
    ) {
      return VAULT_FLOW;
    }
    return CHECKOUT_FLOW;
  }

  checkIntentOption(paypalIntent: string): boolean {
    let isIntentValid = true;
    if (
        paypalIntent !== INTENT_SALE &&
        paypalIntent !== INTENT_ORDER &&
        paypalIntent !== INTENT_AUTHORIZE &&
        paypalIntent !== INTENT_CAPTURE
    ) {
      isIntentValid = false;
      return isIntentValid;
    } else {
      return isIntentValid;
    }
  }

  handleClientError(error): void {
    if (
        error != null &&
        (typeof error !== 'undefined' || error !== 'undefined') &&
        error.message !== 'User did not enter a payment method'
    ) {

      if (this.isHostedFieldsError(error)) {
        this.globalMessageService.add(
            {
              key: 'error.invalidCard',
            },
            GlobalMessageType.MSG_TYPE_ERROR
        );
      } else {
        let messageText = error.message;

        if (typeof messageText !== 'undefined' && messageText !== 'undefined') {
          this.braintreeErrorHandlerService.getErrorMessage(error.code.toLowerCase()).subscribe(res => {
            if (res !== undefined) {
              messageText = res;
            }
            this.globalMessageService.add(
                {
                  key: 'error.provider',
                  params: {reason: messageText},
                },
                GlobalMessageType.MSG_TYPE_ERROR
            );
          });
        }
      }
    }
  }

  isHostedFieldsError(error): boolean {
    if (
        error.code === HOSTED_FIELDS_FIELDS_EMPTY
        || error.code === HOSTED_FIELDS_FIELDS_INVALID
        || error.code === HOSTED_FIELDS_FAILED_TOKENIZATION
        || error.code === HOSTED_FIELDS_TOKENIZATION_NETWORK_ERROR
    ) {
      return true;
    }
    return false;
  }

  getTotalAmountBeforeRendering(): number {
    let amount: number;
    this.activeCartService
    .getActive()
    .subscribe(cart => amount = cart?.totalPrice?.value)
    .unsubscribe();
    return amount;
  }

  getSubTotalAmount(): number {
    let amount: number;
    this.activeCartService
    .getActive()
    .subscribe(cart =>
        amount = parseFloat(cart.subTotal.formattedValue.match('-?\\d*\\.?\\d+')[0])
    )
    .unsubscribe();
    return amount;
  }

  getBaseUrl(): string {
    return this.config.backend.occ.baseUrl;
  }
}
