import {BraintreePaymentMethodsErrorHandlerService} from './../errorHandling/braintree-payment-methods-error-handler.service';
import {Injectable} from '@angular/core';
import {BraintreeUtilsService} from '../utils/braintree-utils.service';
import * as CONST from '../../models/braintree.constants';

import * as braintree from 'braintree-web';
import {GlobalMessageService, GlobalMessageType} from '@spartacus/core';
import {PayPalButtonStyle} from '../../models/paypal-button-style.model';
import {PageType} from '../../models/braintree-payment-methods.model';
import {CheckoutData} from '../../models/braintree-payment-data.model';
import {BraintreeCheckoutService} from '../checkout/braintree-checkout.service';
import {BraintreeExpressCheckoutService} from '../checkout/braintree-express-checkout.service';
import {INTENT_ORDER} from '../../models/braintree.constants';

@Injectable({
  providedIn: 'root'
})
export class PaypalCheckoutService {

  protected checkoutData: CheckoutData;
  protected payPalOptionsAmount: number;
  protected buttonStyle: PayPalButtonStyle;
  protected deviceData: string;
  protected pageType: PageType;

  constructor(
      protected braintreeUtils: BraintreeUtilsService,
      protected globalMessageService: GlobalMessageService,
      protected braintreeCheckoutService: BraintreeCheckoutService,
      protected braintreeExpressCheckoutService: BraintreeExpressCheckoutService,
      protected braintreeErrorHandlerService: BraintreePaymentMethodsErrorHandlerService
  ) {
  }


  initializePayPalCheckout(buttonContainer, checkoutData: CheckoutData, buttonStyle: PayPalButtonStyle, pageType: PageType): void {
    this.buttonStyle = buttonStyle;
    this.pageType = pageType;

    this.checkoutData = checkoutData;
    let payPalOptions;
    if (PageType.CART === pageType) {
      payPalOptions = this.createPayPalOptionsExpressCheckout(checkoutData, PageType.CART);
    } else {
      payPalOptions = this.createPayPalOptions(checkoutData, pageType);
    }

    this.braintreeUtils.createClientInstance(this.checkoutData.configurationData, (client, deviceData) => {
      this.deviceData = deviceData;
      this.createPayPalCheckout(payPalOptions, client, buttonContainer, pageType);
    });
  }

  updatePayPalOptionsAmount(amount: number): void {
    this.payPalOptionsAmount = amount;
  }

  getCountyCode(): string {
    return this.checkoutData?.shippingAddressOverride?.countryCodeAlpha2;
  }

  private getPayPalOptionsAmount(): number {
    return this.payPalOptionsAmount;
  }

  public createPayPalOptionsExpressCheckout(checkoutData: CheckoutData, pageType): any {
    // const paypalIntent = this.checkoutData.configurationData.intent;
    const paypalFlow = this.braintreeUtils.getPaypalFlow(checkoutData, pageType);
    const enableEditShippingAddress = checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton
        || (!checkoutData.payPalPaymentMethod.shouldShowShippingAddressMessage && (checkoutData.configurationData.intent === INTENT_ORDER
        && checkoutData.configurationData.graphQLEnabled));

    let paypalOptions: { [k: string]: any } = {};
    paypalOptions = {
      flow: paypalFlow,
      enableShippingAddress: true,
      enableBillingAddress: true,
      locale: checkoutData.configurationData.locale,
      shippingAddressEditable: !enableEditShippingAddress
    };

    if (paypalFlow === CONST.CHECKOUT_FLOW) {
      this.configurePayPalOptions(paypalOptions, enableEditShippingAddress, checkoutData);
      // configure pay later integration
      if ((checkoutData.payPalPaymentMethod.creditEnabled || checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton)
          && this.braintreeUtils.getStoreInVault(checkoutData.configurationData.storeInVault)) {
        paypalOptions.requestBillingAgreement = true;
        paypalOptions.billingAgreementDetails = {
          description: checkoutData.payPalPaymentMethod.billingAgreementDescription
        };
      }

    } else if (paypalFlow === CONST.VAULT_FLOW) {
      paypalOptions.billingAgreementDescription = checkoutData.payPalPaymentMethod.billingAgreementDescription;
    }

    const dbaName = checkoutData.configurationData.dbaName;
    // configure display name for paypal connection
    if (typeof dbaName !== 'undefined' && dbaName !== '') {
      if (dbaName.indexOf('*') > -1) {
        paypalOptions.displayName = dbaName.substr(0, dbaName.indexOf('*'));
      }
    }
    return paypalOptions;
  }

  private createPayPalCheckout(paypalOptions, client, payPalButtonContainer, pageType: PageType): void {
    if (payPalButtonContainer != null) {
      const commit = paypalOptions.paypalIntent === CONST.INTENT_SALE && paypalOptions.userAction === 'true';
      const isIntentValid = this.braintreeUtils.checkIntentOption(this.checkoutData.configurationData.intent);
      if (isIntentValid === false) {
        this.globalMessageService.add(
            {key: 'intent.notSupported'},
            GlobalMessageType.MSG_TYPE_ERROR);
        return;
      }

      braintree.paypalCheckout.create({
        autoSetDataUserIdToken: this.checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton,
        client
      }, (paypalCheckoutErr, paypalCheckoutInstance) => {
        paypalOptions.amount = this.braintreeUtils.getTotalAmountBeforeRendering()?.toFixed(2);
        this.braintreeUtils.loadPayPalSDK(this.checkoutData, paypalCheckoutInstance,
            this.checkoutData.configurationData.intent, commit, paypalOptions.flow, pageType, () => {
              console.log('PayPal Sdk was loaded');
              payPalButtonContainer.innerHTML = '';

              this.payPalCheckoutRenderButton(paypalCheckoutInstance, payPalButtonContainer, paypalOptions, commit);
            }, paypalOptions.amount);
      });


    }

  }

  private payPalCheckoutRenderButton(paypalCheckoutInstance, payPalButtonContainer, paypalOptions, commit): void {
    const isCheckoutFlow = this.braintreeUtils.getPaypalFlow(this.checkoutData, this.pageType).localeCompare(CONST.CHECKOUT_FLOW);
    if ((this.checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton && this.checkoutData.shippingAddressOverride) ||
        (!this.checkoutData.payPalPaymentMethod.shouldShowShippingAddressMessage && (this.checkoutData.configurationData.intent === INTENT_ORDER)
        && this.checkoutData.configurationData.graphQLEnabled)) {
      const subtotal: number = this.braintreeUtils.getSubTotalAmount();
      this.braintreeExpressCheckoutService
      .countDeliveryPrice(this.checkoutData.shippingAddressOverride.countryCodeAlpha2)
      .subscribe(priceValueData => {
        this.payPalOptionsAmount = subtotal + priceValueData.value;
      });
    }
    if (isCheckoutFlow === 0) {
      if ((this.checkoutData.payPalPaymentMethod.creditEnabled
          || this.checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton)
          && this.braintreeUtils.getStoreInVault(this.checkoutData.configurationData.storeInVault)) {
        this.renderCheckoutBillingButtons(paypalCheckoutInstance, payPalButtonContainer, paypalOptions, commit);
      } else {
        this.renderCheckoutFlowPayPalButtons(paypalCheckoutInstance, payPalButtonContainer, paypalOptions, commit);
      }
    } else {
      this.renderVaultFlowPayPalButtons(paypalCheckoutInstance, payPalButtonContainer, paypalOptions, commit);
    }
  }

  private renderCheckoutFlowPayPalButtons(paypalCheckoutInstance, payPalButtonContainer, paypalOptions, commit): any {
    let fundingSource;
    const defaultShippingAddressEditable = paypalOptions.shippingAddressEditable;
    try {
      (window as any).paypalSdk.Buttons({
        style: this.buttonStyle,
        locale: this.checkoutData.configurationData.braintreeLocale,
        commit,
        onInit: (data, actions) => {
          if ((this.checkoutData.configurationData.intent === INTENT_ORDER)
              && this.checkoutData.payPalPaymentMethod.shouldShowShippingAddressMessage) {
            actions.disable();
          }
        },
        onClick: (button) => {
          if ((this.checkoutData.configurationData.intent === INTENT_ORDER)
              && this.checkoutData.payPalPaymentMethod.shouldShowShippingAddressMessage) {
            this.globalMessageService.add(
                {
                  key: 'error.braintree_message_no_shipping_address',
                },
                GlobalMessageType.MSG_TYPE_ERROR
            );
          } else {
            fundingSource = button.fundingSource;
            if ((this.checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton && this.checkoutData.shippingAddressOverride
                && this.pageType === PageType.CART) || (!this.checkoutData.payPalPaymentMethod.shouldShowShippingAddressMessage
                && (this.checkoutData.configurationData.intent === INTENT_ORDER) && this.checkoutData.configurationData.graphQLEnabled)) {
              paypalOptions.amount = this.payPalOptionsAmount.toFixed(2);
            } else {
              paypalOptions.amount = this.braintreeUtils.getTotalAmountBeforeRendering().toFixed(2);
            }
            this.updatePaypalOptions(paypalOptions, fundingSource, defaultShippingAddressEditable);
          }
        },
        createOrder(): any {
          return paypalCheckoutInstance.createPayment(paypalOptions);
        },
        onApprove: (data) => {
          data.intent = this.checkoutData.configurationData.intent;
          return paypalCheckoutInstance.tokenizePayment(data).then((payload) => {
            this.braintreeCheckoutService.processPayPalResponse(
                payload,
                this.checkoutData.payPalPaymentMethod.creditEnabled,
                this.deviceData,
                this.pageType,
                fundingSource
            );
          });
        },
        onCancel(): void {
          console.log('User cancel PayPal flow');
        },

        onError: (err) => {
          console.error('Error: ' + err, err);
          this.handlePayPalClientError(err);
        }
      }).render(payPalButtonContainer);
    } catch (err) {
      console.log(err.message);
      this.handlePayPalButtonError(err.message);
    }
  }

  private renderVaultFlowPayPalButtons(paypalCheckoutInstance, payPalButtonContainer, paypalOptions, commit): any {
    let fundingSource;
    const defaultShippingAddressEditable = paypalOptions.shippingAddressEditable;
    try {
      (window as any).paypalSdk.Buttons({
        style: this.buttonStyle,
        locale: this.checkoutData.configurationData.braintreeLocale,
        commit,
        onClick: (button) => {
          fundingSource = button.fundingSource;
          paypalOptions.amount = this.braintreeUtils.getTotalAmountBeforeRendering()?.toFixed(2);
          this.updatePaypalOptions(paypalOptions, fundingSource, defaultShippingAddressEditable);
        },
        createBillingAgreement(): any {
          return paypalCheckoutInstance.createPayment(paypalOptions);
        },
        onApprove: (data) => {
          return paypalCheckoutInstance.tokenizePayment(data).then((payload) => {
            this.braintreeCheckoutService.processPayPalResponse(
                payload,
                this.checkoutData.payPalPaymentMethod.creditEnabled,
                this.deviceData,
                this.pageType,
                fundingSource
            );
          });

        },
        onCancel(data): void {
          console.log('User cancel PayPal flow');
        },

        onError: (err) => {
          console.error('Error: ' + err, err);
          this.handlePayPalClientError(err);
        }
      }).render(payPalButtonContainer);
    } catch (err) {
      console.log(err.message);
      this.handlePayPalButtonError(err.message);
    }
  }

  private renderCheckoutBillingButtons(paypalCheckoutInstance, payPalButtonContainer, paypalOptions, commit): any {
    let fundingSource;
    const defaultShippingAddressEditable = paypalOptions.shippingAddressEditable;
    try {
      (window as any).paypalSdk.Buttons({
        style: this.buttonStyle,
        locale: this.checkoutData.configurationData.braintreeLocale,
        commit,
        onClick: (button) => {
          fundingSource = button.fundingSource;
          if (this.checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton && this.checkoutData.shippingAddressOverride
              && this.pageType === PageType.CART) {
            paypalOptions.amount = this.payPalOptionsAmount.toFixed(2);
          } else {
            paypalOptions.amount = this.braintreeUtils.getTotalAmountBeforeRendering().toFixed(2);
          }
          this.updatePaypalOptions(paypalOptions, fundingSource, defaultShippingAddressEditable);
        },
        createOrder(): any {
          return paypalCheckoutInstance.createPayment(paypalOptions);
        },
        onApprove: (data) => {
          return paypalCheckoutInstance.tokenizePayment(data).then((payload) => {
            this.braintreeCheckoutService.processPayPalResponse(
                payload,
                this.checkoutData.payPalPaymentMethod.creditEnabled,
                this.deviceData,
                this.pageType,
                fundingSource
            );
          });
        },
        onCancel(data): void {
          console.log('User cancel PayPal flow');
        },

        onError: (err) => {
          console.error('Error: ' + err, err);
          this.handlePayPalClientError(err);
        }
      }).render(payPalButtonContainer);
    } catch (err) {
      console.log(err.message);
      this.handlePayPalButtonError(err.message);
    }
  }

  private updatePaypalOptions(paypalOptions, fundingSource, defaultShippingAddressEditable): void {
    if (fundingSource === CONST.PAYPAL_FUNDING_CARD) {
      paypalOptions.shippingAddressEditable = true;
    } else {
      paypalOptions.shippingAddressEditable = defaultShippingAddressEditable;
    }
  }

  private handlePayPalClientError(error): void {
    if (typeof error !== 'undefined' || error !== 'undefined') {
      // skip validation error: use paypal method
      if ('User did not enter a payment method' !== error.message && 'PAYPAL_POPUP_CLOSED' !== error.code) {
        let messageText = error.message;
        if (typeof messageText === 'undefined' || messageText === 'undefined') {
          // Undefined error
        } else {
          this.braintreeErrorHandlerService.getErrorMessage(error.code.toLowerCase()).subscribe(res => {
            if (res !== undefined) {
              messageText = res;
            }
            this.globalMessageService.add(
                {
                  key: 'error.provider',
                  params: {reason: messageText},
                },
                GlobalMessageType.MSG_TYPE_ERROR
            );
          });
        }
      }
    }
  }

  public createPayPalOptions(checkoutData: CheckoutData, pageType: PageType): any {
    const paypalFlow = this.braintreeUtils.getPaypalFlow(checkoutData, pageType);
    const enableShippingAddress: boolean = checkoutData.enableShippingAddress;

    const payPalOptions: { [k: string]: any } = {
      flow: paypalFlow,
      enableShippingAddress,
      enableBillingAddress: true,
      locale: checkoutData.configurationData.locale,
      shippingAddressEditable: false
    };

    if (PageType.MY_ACCOUNT === pageType) {
      payPalOptions.billingAgreementDescription = checkoutData.payPalPaymentMethod.billingAgreementDescription;
    } else if (CONST.CHECKOUT_FLOW === paypalFlow) {

      this.configurePayPalOptions(payPalOptions, enableShippingAddress, checkoutData);

      if ((checkoutData.payPalPaymentMethod.creditEnabled || checkoutData.payPalPaymentMethod.shouldRenderPayPalChangePaymentButton)
          && this.braintreeUtils.getStoreInVault(checkoutData.configurationData.storeInVault)) {
        payPalOptions.requestBillingAgreement = true;
        payPalOptions.billingAgreementDetails = {
          description: checkoutData.payPalPaymentMethod.billingAgreementDescription
        };
      }

    } else if (CONST.VAULT_FLOW === paypalFlow) {

      this.configurePayPalOptions(payPalOptions, enableShippingAddress, checkoutData);

      payPalOptions.billingAgreementDescription = checkoutData.payPalPaymentMethod.billingAgreementDescription;

      const dbaName = checkoutData.configurationData.dbaName;

      if (typeof dbaName !== undefined && dbaName !== '') {
        if (dbaName.indexOf('*') > -1) {
          payPalOptions.displayName = dbaName.substr(0, dbaName.indexOf('*'));
        }
      }
    }
    return payPalOptions;
  }

  private configurePayPalOptions(payPalOptions, enableShippingAddress, checkoutData: CheckoutData): void {
    const paypalIntent = checkoutData.configurationData.intent;
    if (paypalIntent !== undefined && paypalIntent !== '') {
      payPalOptions.intent = paypalIntent;
    }
    payPalOptions.currency = checkoutData.configurationData.currency;
    // configure pay later integration
    if (checkoutData.shippingAddressOverride && enableShippingAddress) {
      payPalOptions.shippingAddressOverride = payPalOptions.shippingAddressOverride = {
        recipientName: checkoutData.shippingAddressOverride.recipientName,
        line1: checkoutData.shippingAddressOverride.streetAddress,
        line2: checkoutData.shippingAddressOverride.extendedAddress,
        city: checkoutData.shippingAddressOverride.locality,
        countryCode: checkoutData.shippingAddressOverride.countryCodeAlpha2,
        postalCode: checkoutData.shippingAddressOverride.postalCode,
        state: checkoutData.shippingAddressOverride.region,
        phone: checkoutData.shippingAddressOverride.phone
      };
    }
  }

  private handlePayPalButtonError(errorMsg): void {
    if (typeof errorMsg !== 'undefined' || errorMsg !== 'undefined') {
      this.globalMessageService.add(
          errorMsg,
          GlobalMessageType.MSG_TYPE_ERROR
      );
    }
  }
}
