import {Injectable} from '@angular/core';
import {PageType} from '../../models/braintree-payment-methods.model';
import {CheckoutData} from '../../models/braintree-payment-data.model';
import {BraintreeUtilsService} from '../utils/braintree-utils.service';
import {BraintreeCheckoutService} from '../checkout/braintree-checkout.service';
import * as braintree from 'braintree-web';

@Injectable({
  providedIn: 'root'
})
export class BraintreeVenmoService {

  protected deviceData;

  constructor(
    protected braintreeUtils: BraintreeUtilsService,
    protected braintreeCheckoutService: BraintreeCheckoutService,
  ) { }

  initializeVenmo(checkoutData: CheckoutData, buttonContainer, buttonType: PageType): void{

    let profileId;
    if (typeof checkoutData.venmoPaymentMethod.venmoProfileId !== undefined) {
      profileId = checkoutData.venmoPaymentMethod.venmoProfileId;
    }

    this.braintreeUtils.createClientInstance(checkoutData.configurationData, (client, deviceData) => {

      this.deviceData = deviceData;

      let paymentMethodUsage = 'multi_use';
      if (!checkoutData.configurationData.storeInVault) {
        paymentMethodUsage = 'single_use';
      }

      braintree.venmo.create({
        client,
        profileId,
        paymentMethodUsage,
        allowDesktop: true
      }, (venmoErr, venmoInstance) => {

        if (venmoErr) {
          console.error('Error creating Venmo:', venmoErr);
          return;
        }

        if (!venmoInstance.isBrowserSupported()) {
          console.log('Browser does not support Venmo');
          return;
        }

        this.displayVenmoButton(venmoInstance, buttonContainer, buttonType);

        if (venmoInstance.hasTokenizationResult()) {
          venmoInstance.tokenize().then(
            payload => {
              this.braintreeCheckoutService.processVenmoResponse(payload, this.deviceData, buttonType);
            }
          ).catch(this.handleVenmoError);
        }
      });
    });
  }

    displayVenmoButton(venmoInstance, buttonContainer, buttonType: PageType): void{

      // Assumes that venmoButton is initially display: none.
      buttonContainer.style.display = 'block';

      buttonContainer.addEventListener('click', () => {
        buttonContainer.disabled = true;

        venmoInstance.tokenize().then(
          payload => {
            this.braintreeCheckoutService.processVenmoResponse(payload, this.deviceData, buttonType);
          }
        ).catch(this.handleVenmoError).then( () => {
          buttonContainer.removeAttribute('disabled');
        });
      });
  }

  private handleVenmoError(tokenizeErr): void{

    if (tokenizeErr.code === 'VENMO_CANCELED') {
        console.log('App is not available or user aborted payment flow');
      } else if (tokenizeErr.code === 'VENMO_APP_CANCELED') {
        console.log('User canceled payment flow');
      } else {
        console.error('An error occurred:', tokenizeErr.message);
      }
  }

}
