import { ModalService } from '@spartacus/storefront';
import { BraintreeUtilsService } from '../utils/braintree-utils.service';
import {
  BraintreePaymentDetails,
  CheckoutData,
  CreditCardPaymentMethod,
} from '../../models/braintree-payment-data.model';
import { Injectable } from '@angular/core';
import * as braintree from 'braintree-web';
import {
  ActiveCartService,
  Address,
  Cart,
  GlobalMessageService,
  CheckoutDeliveryService,
  GlobalMessageType, TranslationService,
} from '@spartacus/core';
import { BraintreeCheckoutService } from '../checkout/braintree-checkout.service';
import {PageType} from '../../models/braintree-payment-methods.model';

@Injectable({
  providedIn: 'root',
})
export class BraintreeHostedFieldsCheckoutService {

  protected tokenizeHostedFieldsEvent: any;
  protected checkHostedFieldsValidityCallback: any;
  protected deviceData: string;
  protected billingAddress: Address;
  protected sameAsShipping: boolean;
  protected pageType: PageType;

  constructor(
    protected braintreeUtils: BraintreeUtilsService,
    protected braintreeCheckoutService: BraintreeCheckoutService,
    protected activeCartService: ActiveCartService,
    protected globalMessageService: GlobalMessageService,
    protected modalService: ModalService,
    protected checkoutDeliveryService: CheckoutDeliveryService,
    protected translationService: TranslationService
  ) {}

  initializeHostedFields(
    checkoutData: CheckoutData,
    buttonContainer: HTMLElement,
    pageType: PageType
  ): void {
    this.pageType = pageType;

    this.braintreeUtils.createClientInstance(
      checkoutData.configurationData,
      (client: any, deviceData: string) => {
        this.deviceData = deviceData;
        this.createHostedFields(
          client,
          checkoutData.creditCardPaymentMethod,
          buttonContainer
        );
      }
    );
  }

  private createHostedFields(
    client: any,
    creditCardData: CreditCardPaymentMethod,
    buttonContainer: HTMLElement
  ): void {
    this.translationService.translate('paymentForm.accountHolderName.placeholder').subscribe(accountPlaceHolder => {
      braintree.hostedFields.create(
          {
            client,
            styles: {
              // Styling element state
              ':focus': {
                color: 'blue',
              },
              '.valid': {
                color: 'green',
              },
              '.invalid': {
                color: 'red',
              },
            },
            fields: {
              cardholderName: {
                container: '#cardholderName',
                placeholder: accountPlaceHolder
              },
              number: {
                container: '#cardNumber',
              },
              expirationDate: {
                container: '#expiration-date',
                placeholder: 'MM/YY',
              },
              cvv: {
                container: '#cvv',
              },
            },
          },
          (hostedFieldsErr, hostedFieldsInstance) => {
            const ERROR_FILED_ID = 'FieldError';
            const CONTAINER_BORDER_COLOR_ERROR = 'border-color: var(--cx-color-danger)';

            hostedFieldsInstance.on('empty', function(event) {
              event.fields[event.emittedBy].container.setAttribute('style', CONTAINER_BORDER_COLOR_ERROR);
              document.getElementById(event.emittedBy + ERROR_FILED_ID).setAttribute('style', '');
            });

            hostedFieldsInstance.on('blur', function(event) {
              if (event.fields[event.emittedBy].isEmpty) {
                event.fields[event.emittedBy].container.setAttribute('style', CONTAINER_BORDER_COLOR_ERROR);
                document.getElementById(event.emittedBy + ERROR_FILED_ID).setAttribute('style', '');
              }
            });

            hostedFieldsInstance.on('notEmpty', function(event) {
              event.fields[event.emittedBy].container.setAttribute('style', '');
              document.getElementById(event.emittedBy + ERROR_FILED_ID).setAttribute('style', 'display:none');
            });

            if (hostedFieldsErr) {
              this.braintreeUtils.handleClientError(hostedFieldsErr);
              return;
            }

            // Call back for checking form validity
            this.checkHostedFieldsValidityCallback = () => {
              const state = hostedFieldsInstance.getState();
              let result = true;
              Object.keys(state.fields).forEach((key) => {
                if (!state.fields[key].isValid) {
                  state.fields[key].container.setAttribute('style', CONTAINER_BORDER_COLOR_ERROR);
                  document.getElementById(key + ERROR_FILED_ID).setAttribute('style', '');
                  result = false;

                  this.globalMessageService.add(
                      {
                        key: 'error.invalidCard',
                      },
                      GlobalMessageType.MSG_TYPE_ERROR
                  );
                }
              });

              return result;
            };

            // Add a click event listener to PayPal image
            this.tokenizeHostedFieldsEvent = () => {
              // initialize paypal authorization
              hostedFieldsInstance.tokenize((tokenizeErr: any, payload: any) => {
                if (tokenizeErr) {
                  this.braintreeUtils.handleClientError(tokenizeErr);
                } else {
                  if (
                      typeof creditCardData.secure3d !== 'undefined' &&
                      creditCardData.secure3d &&
                      payload.type === 'CreditCard' &&
                      this.pageType === PageType.BILLING
                  ) {
                    this.verify3DSecure(
                        client,
                        payload,
                        payload.details.cardholderName,
                        creditCardData.skip3dSecureLiabilityResult
                    );
                  } else {
                    this.braintreeCheckoutService.processHostedFields(
                        payload,
                        payload.details.cardholderName,
                        this.sameAsShipping,
                        this.billingAddress,
                        this.deviceData,
                        this.pageType
                    );
                  }
                }
              });
            };
          }
      );
    });

  }

  tokenizeHostedFields(sameAsShipping: boolean, billingAddress: Address): void{
    this.billingAddress = billingAddress;
    this.sameAsShipping = sameAsShipping;
    this.tokenizeHostedFieldsEvent();
  }

  isHostedFieldsValid(): boolean {
    return this.checkHostedFieldsValidityCallback();
  }

  initialise3dSecure(paymentMethod: BraintreePaymentDetails, checkoutData: CheckoutData): boolean {
    braintree.client.create(
      {
        authorization: checkoutData.configurationData.client_id,
      },
      (clientErr, clientInstance) => {
        if (clientErr) {
          this.braintreeUtils.handleClientError(clientErr);
          return;
        }
        braintree.threeDSecure.create(
          {
            version: 2,
            client: clientInstance,
          },
          (err, threeDSecure) => {
            threeDSecure.verifyCard(
              this.get3DSVerificationData(paymentMethod.paymentMethodNonce, paymentMethod.bin, paymentMethod.billingAddress),
              (error: any, response: any) => {
                if (error) {
                  this.globalMessageService.add(
                    error,
                    GlobalMessageType.MSG_TYPE_ERROR
                  );
                  return;
                } else {
                  // 3DSecure finished, add 3DSecure returned nonce
                  const liabilityShifted = response.liabilityShifted;
                  const liabilityShiftPossible = response.liabilityShiftPossible;
                  // allow process card if 3dSecureLiabilityResult is skipped by merchant
                  if (
                    liabilityShifted ||
                    checkoutData.creditCardPaymentMethod.skip3dSecureLiabilityResult || !liabilityShiftPossible
                  ) {
                    // e.currentTarget.submit();
                    this.braintreeCheckoutService.selectBraintreePaymentMethod(paymentMethod.id, response.nonce);
                  } else {
                    this.show3DSecureMessage('error.unsecuredCard');
                  }
                }
              }
            );
          }
        );
      }
    );
    return false;
  }

  private verify3DSecure(
    clientInstance: any,
    paymentResponse: any,
    accountHolderName: string,
    skip3dSecureLiabilityResult: boolean
  ): void {
    // configurePayPalAlongWithHostedFields();
    // selectPaymentMethod();
    braintree.threeDSecure.create(
      {
        version: 2,
        client: clientInstance,
      },
      (err, threeDSecure) => {
        threeDSecure.verifyCard(
          this.get3DSVerificationData(paymentResponse.nonce, paymentResponse.details.bin, null),
          (error, response) => {
            if (error) {
              this.globalMessageService.add(
                error,
                GlobalMessageType.MSG_TYPE_ERROR
              );
              return;
            } else {
              // 3DSecure finished
              // add 3DSecure returned nonce
              paymentResponse.nonce = response.nonce;
              const liabilityShifted = response.liabilityShifted;
              const liabilityShiftPossible = response.liabilityShiftPossible;
              // allow process card if 3dSecureLiabilityResult is
              // skipped by merchant
              if (liabilityShifted || skip3dSecureLiabilityResult || !liabilityShiftPossible) {
                paymentResponse.liabilityShifted = liabilityShifted;
                this.braintreeCheckoutService.processHostedFields(
                  paymentResponse,
                  accountHolderName,
                  this.sameAsShipping,
                  this.billingAddress,
                  this.deviceData,
                  this.pageType
                  );
              } else {
                this.show3DSecureMessage('error.unsecuredCard');
              }
            }
          }
        );
      }
    );
  }

  private get3DSVerificationData(nonce: string, bin: string, addressOnCard: any): any{
    let data: any;
    let address: any;
    let cart: Cart;

    this.checkoutDeliveryService
      .getDeliveryAddress()
      .subscribe((res) => address = res)
      .unsubscribe();

    this.activeCartService
      .getActive()
      .subscribe((res) => cart = res)
      .unsubscribe();

    data = {
      amount: cart.totalPrice.value,
      nonce,
      bin,
      email: cart.user.uid,
      billingAddress: this.getBillingAddressFor3DS(address, addressOnCard),
      additionalInformation: {
        workPhoneNumber: address.phone,
        shippingGivenName: address.firstName,
        shippingSurname: address.lastName,
        shippingPhone: address.phone,
        shippingAddress: {
          streetAddress: address.line1,
          extendedAddress: address.line2,
          locality: address.town,
          postalCode: address.postalCode,
          countryCodeAlpha2: address.country.isocode
        }
      },

      onLookupComplete: (data, next) => {
        next();
      },
      addFrame: (err: any, iframe: HTMLIFrameElement) => {
        const threeDSContainer = document.getElementById(
          'form-additionals'
        );
        threeDSContainer.appendChild(iframe);
        this.modalService.open(iframe, { centered: true });
      },
      removeFrame: () => {
        this.modalService.closeActiveModal();
      }
    };

    if (address.region !== undefined){
      data.additionalInformation.shippingAddress.region = address.region.isocodeShort;
    }

    return data;
  }

  private getBillingAddressFor3DS(address: any, adressOnCard: any): any{
    let billingAddress: any;

    if (adressOnCard !== null) {
      billingAddress = {
        givenName: adressOnCard.firstName,
        surname: adressOnCard.lastName,
        phoneNumber: adressOnCard.phone,
        streetAddress: adressOnCard.line1,
        extendedAddress: adressOnCard.line2,
        locality: adressOnCard.town,
        postalCode: adressOnCard.postalCode,
        countryCodeAlpha2: adressOnCard.country.isocode
      };
      if (adressOnCard.region !== undefined){
        billingAddress.region = adressOnCard.region.isocodeShort;
      }
    } else if (this.sameAsShipping){
      billingAddress = {
        givenName: address.firstName,
        surname: address.lastName,
        phoneNumber: address.phone,
        streetAddress: address.line1,
        extendedAddress: address.line2,
        locality: address.town,
        postalCode: address.postalCode,
        countryCodeAlpha2: address.country.isocode
      };
      if (address.region !== undefined){
        billingAddress.region = address.region.isocodeShort;
      }
    } else if (this.billingAddress !== undefined) {
      billingAddress = {
        givenName: this.billingAddress.firstName,
        surname: this.billingAddress.lastName,
        phoneNumber: this.billingAddress.phone,
        streetAddress: this.billingAddress.line1,
        extendedAddress: this.billingAddress.line2,
        locality: this.billingAddress.town,
        postalCode: this.billingAddress.postalCode,
        countryCodeAlpha2: this.billingAddress.country.isocode
      };
      if (this.billingAddress.region !== undefined){
        billingAddress.region = this.billingAddress.region.isocodeShort;
      }
    }

    return billingAddress;
  }

  private show3DSecureMessage(message): void{
    this.globalMessageService.add(
      {
        key: message,
      },
      GlobalMessageType.MSG_TYPE_ERROR
    );
  }

}
