import {Injectable} from '@angular/core';
import {ApplePayDropInButtonStyle, DropInButtonStyle, GooglePayDropInButtonStyle} from '../../models/paypal-button-style.model';
import {PageType} from '../../models/braintree-payment-methods.model';
import {CheckoutData} from '../../models/braintree-payment-data.model';
import {BraintreePaymentMethodsUtilsService, BraintreeUtilsService} from '../utils';
import * as dropIn from 'braintree-web-drop-in';
import {PaypalCheckoutService} from './paypal-checkout.service';
import {BraintreeCheckoutService} from '../checkout';
import {ActiveCartService, Cart, CheckoutDeliveryService, GlobalMessageService, GlobalMessageType, TranslationService, UserIdService} from '@spartacus/core';
import {combineLatest, Observable, of} from 'rxjs';
import {map} from 'rxjs/operators';
import {BraintreePaymentConnector} from '../../connectors';
import {INTENT_ORDER} from '../../models';
import {BraintreeExpressCheckoutService} from '../checkout/braintree-express-checkout.service';

@Injectable({
  providedIn: 'root'
})
export class BraintreeDropInService {

  protected checkoutData: CheckoutData;
  protected dropInButtonStyle: DropInButtonStyle;
  protected pageType: PageType;

  constructor(
      protected braintreeUtils: BraintreeUtilsService,
      protected paypalCheckoutService: PaypalCheckoutService,
      protected braintreeCheckoutService: BraintreeCheckoutService,
      protected braintreeExpressCheckoutService: BraintreeExpressCheckoutService,
      protected activeCartService: ActiveCartService,
      protected globalMessageService: GlobalMessageService,
      protected userIdService: UserIdService,
      protected paymentConnector: BraintreePaymentConnector,
      protected paymentMethodsUtilsService: BraintreePaymentMethodsUtilsService,
      protected translationService: TranslationService,
      protected checkoutDeliveryService: CheckoutDeliveryService
  ) {
  }

  initializeDropIn(
      buttonContainer,
      submitButton,
      checkoutData: CheckoutData,
      dropInButtonStyle: DropInButtonStyle,
      pageType: PageType,
      callbackAfterDropInLoaded?,
      callbackGetBillingAddress?
  ): void {
    this.dropInButtonStyle = dropInButtonStyle;
    this.pageType = pageType;
    this.checkoutData = checkoutData;

    let payPalOptions: Observable<any> = of(false);
    let payPalCreditOptions: any = false;
    let venmoOptions: any = false;
    let cardOptions: any = false;
    let googlePayOptions: any;
    let applePayOptions: any;
    let threeDSecureOptions: any = false;
    let translations: any = false;
    let vaultManager = true;
    let customerCanceled3ds = false;

    if (checkoutData.dropIn.isLimitedDeletePaymentMethods) {
      vaultManager = false;
    }

    let optionsPayPal: Observable<any>;

    if (PageType.CART === pageType) {

      optionsPayPal = this.generatePayPalOptions(checkoutData, PageType.CART);

      if (checkoutData.payPalPaymentMethod.payPalExpressEnabled) {
        payPalOptions = optionsPayPal;
      }

      googlePayOptions = this.generateGooglePayOptions(checkoutData, dropInButtonStyle.googlePayDropIn, PageType.CART);
      applePayOptions = this.generateApplePayOptions(checkoutData, dropInButtonStyle.applePayDropIn, PageType.CART);

    } else {
      if (checkoutData.creditCardPaymentMethod.hostedFieldsEnable) {
        cardOptions = {
          cardholderName: {
            required: true
          },
          vault: {
            vaultCard: (checkoutData.configurationData.storeInVault === 'true')
          }
        };
      }

      if (checkoutData.creditCardPaymentMethod.secure3d) {
        threeDSecureOptions = {
          cardinalSDKConfig: {
            version: 2
          }
        };
      }

      optionsPayPal = this.generatePayPalOptions(checkoutData, pageType);

      if (checkoutData.payPalPaymentMethod.payPalStandardEnabled) {
        payPalOptions = optionsPayPal;
      }

      if (checkoutData.venmoPaymentMethod.venmoEnabled) {
        venmoOptions = {
          allowNewBrowserTab: true,
          ignoreHistoryChanges: true,
          allowDesktop: true
        };
      }

      googlePayOptions = this.generateGooglePayOptions(checkoutData, dropInButtonStyle.googlePayDropIn, pageType);
      applePayOptions = this.generateApplePayOptions(checkoutData, dropInButtonStyle.applePayDropIn, pageType);

    }

    if (PageType.MY_ACCOUNT === pageType) {
      combineLatest(
          [
            this.translationService.translate('dropIn.myAccount.payingWith'),
            this.translationService.translate('dropIn.myAccount.chooseAWayToPay'),
            this.translationService.translate('dropIn.myAccount.chooseAnotherWayToPay'),
            this.translationService.translate('dropIn.myAccount.payWithCard')
          ]).subscribe(([payingWith, chooseAWayToPay, chooseAnotherWayToPay, payWithCard]) => {
        translations = {
          payingWith,
          chooseAWayToPay,
          chooseAnotherWayToPay,
          payWithCard
        };
      });

      payPalCreditOptions = false;
      venmoOptions = false;
      googlePayOptions = false;
      applePayOptions = false;
      threeDSecureOptions = false;
    }

    let amount: number;
    this.activeCartService
    .getActive()
    .subscribe((cart) => (amount = cart?.totalPrice?.value))
    .unsubscribe();

    payPalOptions.subscribe(options => {

      let finalPayPalOptions;
      if (options) {
        finalPayPalOptions = Object.assign({}, options);
        finalPayPalOptions.buttonStyle = dropInButtonStyle.payPalDropIn;

        if (checkoutData.payPalPaymentMethod.creditEnabled && PageType.MY_ACCOUNT !== pageType) {
          payPalCreditOptions = options;
        }
      } else {
        finalPayPalOptions = false;
      }

      dropIn.create({
        authorization: checkoutData.configurationData.client_id,
        container: buttonContainer,
        locale: checkoutData.configurationData.locale,
        card: cardOptions,
        paypal: finalPayPalOptions,
        paypalCredit: payPalCreditOptions,
        venmo: venmoOptions,
        googlePay: googlePayOptions,
        applePay: applePayOptions,
        dataCollector: true,
        threeDSecure: threeDSecureOptions,
        vaultManager,
        translations
      }, (dropInCreatingError, dropinInstance) => {

        if (dropInCreatingError) {
          // Handle any errors that might've occurred when creating Drop-in
          console.error(dropInCreatingError);
          this.showErrorMessage(dropInCreatingError.message);
          return;
        }

        if (callbackAfterDropInLoaded !== undefined) {
          callbackAfterDropInLoaded(dropinInstance, dropInCreatingError);
        }

        const processResponse = (err, payload) => {
          if (err) {
            // Handle errors in requesting payment method
            console.log(err);
            this.showErrorMessage(err.message);
            return;
          }
          // Send payload.nonce to your server
          if (payload.type === 'CreditCard') {
            if (!customerCanceled3ds) {
              this.processCreditCardResponse(payload, checkoutData, pageType, callbackGetBillingAddress);
            } else {
              // Reload page if customer canceled in 3ds
              let currentURL = window.location.href;
              window.location.href = currentURL;
            }
          } else {
            this.sendPayloadNonceToServer(payload, checkoutData, pageType);
          }

        };

        if (checkoutData.creditCardPaymentMethod.secure3d && PageType.MY_ACCOUNT !== pageType) {
          submitButton.addEventListener('click', () => {
            dropinInstance.requestPaymentMethod({
              threeDSecure: this.get3DSVerificationData(callbackGetBillingAddress)
            }, processResponse);
          });
        } else {
          submitButton.addEventListener('click', () => {
            dropinInstance.requestPaymentMethod(processResponse);
          });
        }

        dropinInstance.on('3ds:customer-canceled', function () {
          customerCanceled3ds = true;
        });

        if (dropinInstance._model !== undefined && dropinInstance._model._events !== undefined
            && this.checkoutData.dropIn.showWarningMessageAboutReplenishmentDropIn) {
          const defaultEnableEdit = dropinInstance._model._events.enableEditMode[0];
          dropinInstance._model._events.enableEditMode[0] = () => {
            this.showWarningMessage('warning.warningMessageAboutReplenishmentDropIn');
            defaultEnableEdit();
          };
        }

      });

    });

  }


  private get3DSVerificationData(callbackGetBillingAddress): any{
    let data: any;
    let address: any;
    let cart: Cart;

    this.checkoutDeliveryService
      .getDeliveryAddress()
      .subscribe((res) => address = res)
      .unsubscribe();

    this.activeCartService
      .getActive()
      .subscribe((res) => cart = res)
      .unsubscribe();

    let billingAddress = callbackGetBillingAddress();
    
    const sameAsShipping = (billingAddress === undefined);

    data = {
      amount: cart.totalPrice.value,
      email: cart.user.uid,
      billingAddress: this.getBillingAddressFor3DS(sameAsShipping? address : billingAddress),
      additionalInformation: {
        workPhoneNumber: address.phone,
        shippingGivenName: address.firstName,
        shippingSurname: address.lastName,
        shippingPhone: address.phone,
        shippingAddress: {
          streetAddress: address.line1,
          extendedAddress: address.line2,
          locality: address.town,
          postalCode: address.postalCode,
          countryCodeAlpha2: address.country.isocode
        }
      },
    }

    if(address.region !== undefined){
      data.additionalInformation.shippingAddress.region = address.region.isocodeShort;
    }
    
    return data;
  }

  private getBillingAddressFor3DS(address: any): any{
    let billingAddress: any;
    
    billingAddress = {
      givenName: address.firstName,
      surname: address.lastName,
      phoneNumber: address.phone,
      streetAddress: address.line1,
      extendedAddress: address.line2,
      locality: address.town,
      postalCode: address.postalCode,
      countryCodeAlpha2: address.country.isocode
    }
    if (address.region !== undefined){
      billingAddress.region = address.region.isocodeShort;
    }

    return billingAddress;
  }



  private showWarningMessage(message): void {
    this.globalMessageService.add(
        {
          key: message,
        },
        GlobalMessageType.MSG_TYPE_WARNING
    );
  }

  protected processCreditCardResponse(payload, checkoutData: CheckoutData, pageType: PageType, callbackGetBillingAddress): void {
    let billingAddress;
    if (PageType.MY_ACCOUNT === pageType) {
      billingAddress = this.paymentMethodsUtilsService.getBillingAddress();
      this.paymentMethodsUtilsService.getBackToPaymentMethodsEmitter().emit();
    } else {
      billingAddress = callbackGetBillingAddress();
    }
    const sameAsShipping = (billingAddress === undefined);

    if (billingAddress === false) {
      return;
    }

    if (checkoutData.creditCardPaymentMethod.secure3d && PageType.MY_ACCOUNT !== pageType) {

      const liabilityShifted = payload.liabilityShifted;
      if (liabilityShifted || checkoutData.creditCardPaymentMethod.skip3dSecureLiabilityResult || !payload.liabilityShiftPossible) {
        this.braintreeCheckoutService.processHostedFields(payload, payload.details.cardholderName, sameAsShipping,
            billingAddress, payload.deviceData, pageType);
      } else {
        this.showErrorMessage(
            {
              key: 'error.unsecuredCard',
            }
        );
      }

    } else {
      this.braintreeCheckoutService.processHostedFields(payload, payload.details.cardholderName, sameAsShipping,
          billingAddress, payload.deviceData, pageType);
    }
  }

  private showErrorMessage(message): void {
    this.globalMessageService.add(
        message,
        GlobalMessageType.MSG_TYPE_ERROR
    );
  }

  generateGooglePayOptions(checkoutData: CheckoutData, buttonStyle: GooglePayDropInButtonStyle, pageType: PageType): any {
    let googlePayOptions: any;
    if (!checkoutData.googlePayPaymentMethod.googlePayEnabled) {
      return false;
    }
    const transactionInfo: any = {
      countryCode: checkoutData.googlePayPaymentMethod.googlePayCountryCode,
      currencyCode: checkoutData.configurationData.currency,
      totalPriceStatus: 'FINAL'
    };

    if (PageType.MY_ACCOUNT === pageType) {
      transactionInfo.totalPrice = '0.0';
    } else {
      transactionInfo.totalPrice = JSON.stringify(this.braintreeUtils.getTotalAmountBeforeRendering());
    }

    googlePayOptions = {
      merchantId: checkoutData.googlePayPaymentMethod.googleMerchantId,
      googlePayVersion: 2,
      transactionInfo,
      emailRequired: true,
      allowedPaymentMethods: [{
        type: 'CARD',
        parameters: {
          billingAddressRequired: true,
          billingAddressParameters: {
            format: 'FULL',
            phoneNumberRequired: true
          }
        }
      }]
    };

    if (PageType.CART === pageType) {
      googlePayOptions.shippingAddressRequired = true;
    }

    googlePayOptions.button = buttonStyle;

    return googlePayOptions;
  }

  generateApplePayOptions(checkoutData: CheckoutData, buttonStyle: ApplePayDropInButtonStyle, pageType: PageType): any {
    let applePayOptions: any;
    if (checkoutData.applePayPaymentMethod.applePayEnabled === 'false') {
      return false;
    }
    const flow = this.braintreeUtils.getPaypalFlow(checkoutData, pageType);
    const amount = this.braintreeUtils.getTotalAmountBeforeRendering();
    let requiredShippingContactFields = [
      'postalAddress',
      'name',
      'phone',
      'email',
    ];

    if (pageType === PageType.BILLING) {
      requiredShippingContactFields = ['email'];
    }

    const paymentRequest: any = {
      total: {
        label: flow,
        amount,
      },
      requiredBillingContactFields: ['postalAddress', 'name'],
      requiredShippingContactFields
    };

    applePayOptions = {
      displayName: 'Electronics',
      paymentRequest
    };

    applePayOptions.buttonStyle = buttonStyle.buttonStyle;

    return applePayOptions;
  }

  generatePayPalOptions(checkoutData: CheckoutData, pageType: PageType): Observable<any> {
    let optionsPayPal;
    if (PageType.CART === pageType) {
      optionsPayPal = this.paypalCheckoutService.createPayPalOptionsExpressCheckout(checkoutData, pageType);
    } else {
      optionsPayPal = this.paypalCheckoutService.createPayPalOptions(checkoutData, pageType);
    }

    if (PageType.MY_ACCOUNT === pageType) {
      return of(optionsPayPal);
    }

    if (!checkoutData.payPalPaymentMethod.shouldShowShippingAddressMessage
        && (checkoutData.configurationData.intent === INTENT_ORDER) && checkoutData.configurationData.graphQLEnabled) {
      const subtotal: number = this.braintreeUtils.getSubTotalAmount();
      return this.braintreeExpressCheckoutService.countDeliveryPrice(checkoutData.shippingAddressOverride.countryCodeAlpha2)
      .pipe(
          map(price => {
            optionsPayPal.amount = subtotal + price.value;
            return optionsPayPal;
          })
      );
    } else {
      optionsPayPal.amount = this.braintreeUtils.getTotalAmountBeforeRendering()?.toFixed(2);
      return of(optionsPayPal);
    }
  }

  public getDropInButtonStyle(
      buttonType?: string
  ): Observable<DropInButtonStyle> {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();
    return this.paymentConnector.getDropInButtonStyle(userId, buttonType);
  }

  public sendPayloadNonceToServer(payload, checkoutData: CheckoutData, pageType): void {
    if (payload.type === 'PayPalAccount') {
      this.braintreeCheckoutService.processPayPalResponse(
          payload,
          checkoutData.payPalPaymentMethod.creditEnabled,
          payload.deviceData,
          pageType
      );
    } else if (payload.type === 'AndroidPayCard') {
      this.braintreeCheckoutService.processGooglePayResponse(
          payload,
          payload.details.rawPaymentData,
          payload.deviceData,
          pageType
      );
    } else if (payload.type === 'VenmoAccount') {
      this.braintreeCheckoutService.processVenmoResponse(payload, payload.deviceData, pageType);
    } else if (payload.type === 'ApplePayCard') {
      this.braintreeCheckoutService.processApplePayResponse(payload, {
        payment: {
          shippingContact: payload.details.rawPaymentData.shippingContact,
          billingContact: payload.details.rawPaymentData.billingContact
        }
      }, pageType, payload.deviceData);
    }
  }
}
