import {
  ApplePayPaymentRequest,
  BraintreeCreditCardPaymentDetails,
  Fields,
  GooglePayPaymentRequest,
  PageType, SrcPaymentRequest
} from '../../models/braintree-payment-methods.model';
import {Injectable, InjectionToken} from '@angular/core';
import {
  ActiveCartService,
  Address,
  CheckoutDeliveryService,
  CheckoutPaymentService,
  CheckoutService,
  Converter,
  ConverterService,
  EventService,
  OCC_USER_ID_ANONYMOUS,
  RoutingService,
  UserIdService,
} from '@spartacus/core';
import {PayPalTokenizePayload} from 'braintree-web';
import {BraintreeExpressCheckoutService} from './braintree-express-checkout.service';
import {
  APPLE_PAY_ADDRESS_SERIALIZER,
  GOOGLE_PAY_ADDRESS_SERIALIZER,
  PAYPAL_ADDRESS_SERIALIZER,
  SRC_ADDRESS_SERIALIZER
} from '../../braintree-checkout-store/converters/converters';
import {BraintreePaymentDetailsService} from '../../facade/braintree-payment-details.service';
import {BraintreePaymentMethodsUtilsService} from '../utils/braintree-payment-methods-utils.service';
import {BraintreeMarkCheckoutService} from './braintree-mark-checkout.service';
import {take, takeWhile} from 'rxjs/operators';
import {BraintreeChangedPaymentDetailsEvent} from '../../events/braintree-payment-details.event';

@Injectable({
  providedIn: 'root'
})
export class BraintreeCheckoutService {

  constructor(
      protected checkoutPaymentService: CheckoutPaymentService,
      protected checkoutDeliveryService: CheckoutDeliveryService,
      protected converter: ConverterService,
      protected braintreePaymentDetailsService: BraintreePaymentDetailsService,
      protected braintreeExpressCheckoutService: BraintreeExpressCheckoutService,
      protected paymentMethodsUtilsService: BraintreePaymentMethodsUtilsService,
      protected braintreeMarkCheckoutService: BraintreeMarkCheckoutService,
      protected checkoutService: CheckoutService,
      protected routingService: RoutingService,
      protected userIdService: UserIdService,
      protected eventService: EventService,
      protected activeCartService: ActiveCartService
  ) {
  }

  processPayPalResponse(
      payload: PayPalTokenizePayload,
      credit: boolean,
      deviceData: string,
      pageType: PageType,
      fundingSource?: string
  ): void {
    const isCredit = PageType.MY_ACCOUNT === pageType ? false : credit;

    if (this.isAnonymousExpressCheckout(pageType)) {
      this.activeCartService.addEmail(payload.details.email);
      this.activeCartService.getAssignedUser()
      .pipe(takeWhile(() => !this.activeCartService.isGuestCart(), true))
      .subscribe(() => {
        if (this.activeCartService.isGuestCart()) {
          this.braintreePaymentDetailsService.savePayPalPaymentDetails(isCredit, this.isShouldBeSavedPayment(pageType),
              payload, pageType, deviceData, Fields.Full, fundingSource);
        }
      });
    } else {
      this.braintreePaymentDetailsService.savePayPalPaymentDetails(isCredit, this.isShouldBeSavedPayment(pageType),
          payload, pageType, deviceData, Fields.Full, fundingSource);
    }

    this.processPaymentDetailsSetting(pageType, payload, PAYPAL_ADDRESS_SERIALIZER);
  }

  processGooglePayResponse(result, googlePayPaymentData, deviceData: string, pageType: PageType): void {
    const googlePayPaymentRequest: GooglePayPaymentRequest = {
      nonce: result.nonce,
      type: result.type,
      email: googlePayPaymentData.email,
      cardType: result.details.cardType,
      lastFour: result.details.lastFour,
      billingAddress: googlePayPaymentData.paymentMethodData.info.billingAddress
    };

    if (this.isAnonymousExpressCheckout(pageType)) {
      this.activeCartService.addEmail(googlePayPaymentRequest.email);
      this.activeCartService.getAssignedUser()
      .pipe(takeWhile(() => !this.activeCartService.isGuestCart(), true))
      .subscribe(() => {
        if (this.activeCartService.isGuestCart()) {
          this.braintreePaymentDetailsService.saveGooglePayPaymentDetails(
              googlePayPaymentRequest,
              this.isShouldBeSavedPayment(pageType),
              deviceData,
              Fields.Full
          );
        }
      });
    } else {
      this.braintreePaymentDetailsService.saveGooglePayPaymentDetails(
          googlePayPaymentRequest,
          this.isShouldBeSavedPayment(pageType),
          deviceData,
          Fields.Full
      );
    }

    this.processPaymentDetailsSetting(pageType, googlePayPaymentData.shippingAddress, GOOGLE_PAY_ADDRESS_SERIALIZER);
  }

  processVenmoResponse(payload, deviceData, pageType: PageType): void {
    let selectedAddressCode = '';
    if (PageType.MY_ACCOUNT === pageType) {
      selectedAddressCode = this.paymentMethodsUtilsService.getBillingAddress().id;
    } else {
      this.checkoutDeliveryService
      .getDeliveryAddress()
      .pipe(take(1))
      .subscribe((address) => {
        selectedAddressCode = address.id;
      });
    }
    this.braintreePaymentDetailsService.saveVenmoPaymentDetails(payload, selectedAddressCode, this.isShouldBeSavedPayment(pageType),
        deviceData, Fields.Full);

    this.braintreePaymentDetailsService.getSelectedPaymentDetails().subscribe(paymentDetails => {
      if (paymentDetails !== undefined) {
        if (PageType.BILLING === pageType) {
          this.braintreeMarkCheckoutService.processCheckout(paymentDetails);
        } else if (PageType.MY_ACCOUNT === pageType) {
          this.paymentMethodsUtilsService.getBackToPaymentMethodsEmitter().emit();
        }
      }
    });
  }

  processLocalPaymentMethodResponse(payload, deviceData): void {

    this.braintreePaymentDetailsService.processLocalPaymentMethod(payload, deviceData, Fields.Full);
  }

  processApplePayResponse(payload: any, event: any, pageType: PageType, deviceData: string): void {
    const applePayPaymentRequest: ApplePayPaymentRequest = {
      nonce: payload.nonce,
      type: payload.type,
      email: event.payment.shippingContact.emailAddress,
      cardType: payload.details.cardType,
      billingContact: event.payment.billingContact
    };
    applePayPaymentRequest.billingContact.countryCode = applePayPaymentRequest.billingContact.countryCode.toUpperCase();

    if (this.isAnonymousExpressCheckout(pageType)) {
      this.activeCartService.addEmail(applePayPaymentRequest.email);
      this.activeCartService.getAssignedUser()
      .pipe(takeWhile(() => !this.activeCartService.isGuestCart(), true))
      .subscribe(() => {
        if (this.activeCartService.isGuestCart()) {
          this.braintreePaymentDetailsService.saveApplePayPaymentDetails(
              applePayPaymentRequest,
              deviceData,
              Fields.Full
          );
        }
      });
    } else {
      this.braintreePaymentDetailsService.saveApplePayPaymentDetails(
          applePayPaymentRequest,
          deviceData,
          Fields.Full
      );
    }

    this.processPaymentDetailsSetting(pageType, event.payment.shippingContact, APPLE_PAY_ADDRESS_SERIALIZER);
  }

  processHostedFields(payload: any, accountHolderName: string, sameAsShipping: boolean, billingAddress: Address,
                      deviceData: string, pageType: PageType): void {
    let selectedAddressCode = '';
    const hostedFieldsResponse: BraintreeCreditCardPaymentDetails = {
      tokenizedCardData: payload
    };
    if (pageType === PageType.MY_ACCOUNT) {
      selectedAddressCode = billingAddress.id;
    } else {
      if (sameAsShipping) {
        this.checkoutDeliveryService.getDeliveryAddress().subscribe(address => {
          selectedAddressCode = address.id;
        }).unsubscribe();
      } else {
        hostedFieldsResponse.billingAddress = billingAddress;
        if (billingAddress.region) {
          billingAddress.region.isocode = billingAddress.country.isocode + '-' + billingAddress.region.isocodeShort;
        }
      }
    }
    hostedFieldsResponse.tokenizedCardData.details.cardholderName = accountHolderName;

    this.braintreePaymentDetailsService.saveCreditCardPaymentDetails(
        hostedFieldsResponse,
        selectedAddressCode,
        this.isShouldBeSavedPayment(pageType),
        deviceData,
        Fields.Full
    );

    this.processPaymentDetailsSetting(pageType);
  }

  processSrcResponse(payload: any, deviceData: string, pageType: PageType): void {

    const srcPaymentRequest: SrcPaymentRequest = {
      nonce: payload.nonce,
      details: payload.details,
      description: payload.description,
      type: payload.type,
      billingAddress: payload.billingAddress,
      userEmail: payload.userData.userEmail,
      userFullName: payload.userData.userFullName
    };

    if (this.isAnonymousExpressCheckout(pageType)) {
      this.activeCartService.addEmail(srcPaymentRequest.userEmail);
      this.activeCartService.getAssignedUser()
      .pipe(takeWhile(() => !this.activeCartService.isGuestCart(), true))
      .subscribe(() => {
        if (this.activeCartService.isGuestCart()) {
          this.braintreePaymentDetailsService.saveSrcPaymentDetails(
              srcPaymentRequest,
              this.isShouldBeSavedPayment(pageType),
              deviceData,
              Fields.Full
          );
        }
      });
    } else {
      this.braintreePaymentDetailsService.saveSrcPaymentDetails(
          srcPaymentRequest,
          this.isShouldBeSavedPayment(pageType),
          deviceData,
          Fields.Full
      );
    }

    this.processPaymentDetailsSetting(pageType, payload.shippingAddress, SRC_ADDRESS_SERIALIZER);
  }

  processPaymentDetailsSetting<S, T>(pageType: PageType, addressSource?: any, injectionToken?: InjectionToken<Converter<S, T>>
  ): void {
    this.braintreePaymentDetailsService.getSelectedPaymentDetails()
    .pipe(takeWhile(paymentDetails => paymentDetails === undefined, true))
    .subscribe(paymentDetails => {
      if (paymentDetails !== undefined) {
        if (PageType.CART === pageType) {
          const address = this.converter.convert(addressSource, injectionToken);
          this.braintreeExpressCheckoutService.processExpressCheckout(address, paymentDetails);
        } else if (PageType.BILLING === pageType) {
          this.braintreeMarkCheckoutService.processCheckout(paymentDetails);
        } else if (PageType.MY_ACCOUNT === pageType) {
          this.paymentMethodsUtilsService.getBackToPaymentMethodsEmitter().emit();
        }
      }
    });
  }

  isAnonymousExpressCheckout(pageType: PageType): boolean {
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();
    return PageType.CART === pageType && OCC_USER_ID_ANONYMOUS === userId && !this.activeCartService.isGuestCart();
  }

  processLPMFallback(payload, deviceData, currencyFromFallbackURL: string): void {
    this.braintreePaymentDetailsService.placeOrderThroughFallback(payload, currencyFromFallbackURL, deviceData, Fields.Full);

    this.checkoutService.getPlaceOrderSuccess().subscribe(res => {
      if (res) {
        this.routingService.go({cxRoute: 'orderConfirmation'});
      }
    });
  }

  savePaymentIdForLPM(paymentId, start): void {
    this.braintreePaymentDetailsService.savePaymentIdForLPM(paymentId).subscribe(res => {
      start();
    });

  }

  selectBraintreePaymentMethod(paymentMethodId: string, paymentMethodNonce: string): void {
    this.braintreePaymentDetailsService.selectBraintreePaymentMethod(paymentMethodId, paymentMethodNonce).subscribe(
        () => this.eventService.dispatch(new BraintreeChangedPaymentDetailsEvent())
    );
  }

  isShouldBeSavedPayment(pageType: PageType): boolean {
    if (PageType.MY_ACCOUNT === pageType) {
      return true;
    } else {
      const savePaymentInfo = this.paymentMethodsUtilsService.getSavePaymentInfo();
      return savePaymentInfo !== undefined ? savePaymentInfo : false;
    }
  }

}
