import {BraintreeAssets, BraintreePaymentDetails} from '../models/braintree-payment-data.model';
import {
  ActiveCartService,
  CheckoutDeliveryService,
  StateWithCheckout,
  StateWithProcess,
  UserIdService
} from '@spartacus/core';
import {BraintreePaymentDetailsConnector} from '../connectors/braintree-payment-details.connector';
import {BraintreePaymentConnector} from '../connectors/braintree-payment.connector';
import {Observable} from 'rxjs';
import {PayPalButtonStyle} from '../models/paypal-button-style.model';
import {CheckoutData} from '../models/braintree-payment-data.model';
import {
  ApplePayPaymentRequest,
  BraintreeCreditCardPaymentDetails,
  Fields,
  GooglePayPaymentRequest,
  LocalPaymentRequest, PageType,
  PayPalPaymentRequest, SrcPaymentRequest,
  VenmoPaymentData
} from '../models/braintree-payment-methods.model';
import {Injectable} from '@angular/core';
import {BraintreePaymentDetailsActions} from '../braintree-checkout-store/actions/index';
import {select, Store} from '@ngrx/store';
import {
  PaymentDetailsState,
  StateWithPaymentDetails
} from '../braintree-checkout-store/braintree-payment-details-state';
import {BraintreePaymentDetailsSelectors} from '../braintree-checkout-store/selectors';
import {filter, map, take} from 'rxjs/operators';
import {
  PaymentDetails,
  CheckoutSelectors
} from '@spartacus/core';

@Injectable({
  providedIn: 'root',
})
export class BraintreePaymentDetailsService {
  constructor(
    protected checkoutStore: Store<StateWithCheckout | StateWithProcess<void>>,
    protected paymentDetailsStore: Store<StateWithPaymentDetails | StateWithProcess<void>>,
    protected activeCartService: ActiveCartService,
    protected userIdService: UserIdService,
    protected connector: BraintreePaymentDetailsConnector,
    protected paymentConnector: BraintreePaymentConnector,
    protected checkoutDeliveryService: CheckoutDeliveryService,
    protected store: Store,
  ) {}

  public saveVenmoPaymentDetails(
    venmoPayment: VenmoPaymentData,
    selectedAddressCode: string,
    shouldBeSaved?: boolean,
    deviceData?: string,
    fields?: Fields
  ): void{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    let cartId;
    this.activeCartService
      .getActiveCartId()
      .subscribe((activeCartId) => (cartId = activeCartId))
      .unsubscribe();

    if (userId) {
      this.store.dispatch(
        new BraintreePaymentDetailsActions.CreateVenmoPaymentDetails({
          userId,
          selectedAddressCode,
          venmoPayment,
          cartId,
          shouldBeSaved,
          deviceData,
          fields
        })
      );
    }
  }

  public savePayPalPaymentDetails(
    credit: boolean,
    shouldBeSaved: boolean,
    payPalRequest: PayPalPaymentRequest,
    pageType: PageType,
    deviceData?: string,
    fields?: Fields,
    fundingSource?: string
  ): void{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    let cartId;
    this.activeCartService
      .getActiveCartId()
      .subscribe((activeCartId) => (cartId = activeCartId))
      .unsubscribe();

    if (userId) {
      this.store.dispatch(new BraintreePaymentDetailsActions.CreatePayPalPaymentDetails({
        credit, shouldBeSaved,
        payPalRequest, pageType, userId, cartId, deviceData, fields, fundingSource
      }));
    }
  }

  public saveGooglePayPaymentDetails(
    googlePayRequest: GooglePayPaymentRequest,
    shouldBeSaved: boolean,
    deviceData?: string,
    fields?: Fields
  ): void{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    let cartId;
    this.activeCartService
      .getActiveCartId()
      .subscribe((activeCartId) => (cartId = activeCartId))
      .unsubscribe();

    this.store.dispatch(new BraintreePaymentDetailsActions.CreateGooglePayPaymentDetails({userId, googlePayRequest,
      shouldBeSaved, cartId, deviceData, fields}));
  }

  public saveSrcPaymentDetails(
      srcRequest: SrcPaymentRequest,
      shouldBeSaved: boolean,
      deviceData?: string,
      fields?: Fields
  ): void{
    let userId;
    this.userIdService
    .getUserId()
    .subscribe((occUserId) => (userId = occUserId))
    .unsubscribe();

    let cartId;
    this.activeCartService
    .getActiveCartId()
    .subscribe((activeCartId) => (cartId = activeCartId))
    .unsubscribe();

    this.store.dispatch(new BraintreePaymentDetailsActions.CreateSrcPaymentDetails({userId, srcRequest,
      shouldBeSaved, cartId, deviceData, fields}));
  }

  public saveCreditCardPaymentDetails(
    paymentDetails: BraintreeCreditCardPaymentDetails,
    selectedAddressCode: string,
    shouldBeSaved?: boolean,
    deviceData?: string,
    fields?: Fields
  ): void{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    let cartId;
    this.activeCartService
      .getActiveCartId()
      .subscribe((activeCartId) => (cartId = activeCartId))
      .unsubscribe();

    this.store.dispatch(new BraintreePaymentDetailsActions.CreateCreditCardPaymentDetails({userId, paymentDetails,
      selectedAddressCode, cartId, shouldBeSaved, deviceData, fields}));
  }

  public saveApplePayPaymentDetails(
    applePayRequest: ApplePayPaymentRequest,
    deviceData?: string,
    fields?: Fields
  ): void{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    let cartId;
    this.activeCartService
      .getActiveCartId()
      .subscribe((activeCartId) => (cartId = activeCartId))
      .unsubscribe();

    this.store.dispatch(new BraintreePaymentDetailsActions.CreateApplePayPaymentDetails({userId, applePayRequest,
       cartId, deviceData, fields}));
  }

  public processLocalPaymentMethod(
    localPaymentRequest: LocalPaymentRequest,
    deviceData?: string,
    fields?: Fields
  ): void{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    let cartId;
    this.activeCartService
      .getActiveCartId()
      .subscribe((activeCartId) => (cartId = activeCartId))
      .unsubscribe();

    this.store.dispatch(new BraintreePaymentDetailsActions.ProcessLocalPaymentMethod({userId, cartId,
      localPaymentRequest,
       deviceData, fields}));
  }

  savePaymentIdForLPM(paymentId): Observable<{}>{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    let cartId;
    this.activeCartService
      .getActiveCartId()
      .subscribe((activeCartId) => (cartId = activeCartId))
      .unsubscribe();

    return this.connector.savePaymentIdForLPM(userId, cartId, paymentId);
  }

  public placeOrderThroughFallback(
    localPaymentRequest: LocalPaymentRequest,
    currencyFromFallbackURL: string,
    deviceData?: string,
    fields?: Fields
  ): void{
    let cartId;
    this.activeCartService
      .getActiveCartId()
      .subscribe((activeCartId) => (cartId = activeCartId))
      .unsubscribe();

    this.checkoutStore.dispatch(new BraintreePaymentDetailsActions.PlaceOrderThroughFallback(
      {localPaymentRequest, currencyFromFallbackURL, deviceData, fields, cartId}));
  }

  public getSelectedPaymentDetails(): Observable<PaymentDetails>{
    return this.paymentDetailsStore.pipe(
      select(BraintreePaymentDetailsSelectors.getPaymentDetailsState),
      filter(vl1 => !!vl1),
      map(state => state.paymentDetails)
    );
}

  public loadPaymentDetails(paymentMethod: string, pageType?: string): Observable<CheckoutData>{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();
    return this.paymentConnector.loadPaymentDetails(paymentMethod, userId, undefined, pageType);
  }

  public getBraintreePaymentDetails(saved: boolean, fields?: Fields): Observable<BraintreePaymentDetails[]>{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    return this.connector.getBraintreePaymentDetails(userId, saved, fields);
  }

  public getBraintreePaymentDetailsById(paymentId: string, fields?: Fields): Observable<BraintreePaymentDetails>{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    return this.connector.getBraintreePaymentDetailsById(userId, paymentId, fields);
  }

  public selectBraintreePaymentMethod(
    selectedPaymentMethodId: string,
    selectedPaymentMethodNonce: string
  ): Observable<{}>{

    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    let cartId;
    this.activeCartService
        .getActiveCartId()
        .subscribe((activeCartId) => (cartId = activeCartId))
        .unsubscribe();

    return this.connector.selectBraintreePaymentMethod(userId, cartId, selectedPaymentMethodId, selectedPaymentMethodNonce);
  }

  public loadPaymentDetailsForBillingPage(paymentMethod: string, pageType?: PageType): Observable<CheckoutData>{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();

    let cartId;
    this.activeCartService
      .getActiveCartId()
      .subscribe((activeCartId) => (cartId = activeCartId))
      .unsubscribe();

    if (PageType.BILLING === pageType){
      return this.paymentConnector.loadPaymentDetails(paymentMethod, userId, cartId);
    }else {
      return this.paymentConnector.loadPaymentDetails(paymentMethod, userId, undefined, pageType);
    }
  }

  public getPayPalButtonStyles(
    buttonType?: string
  ): Observable<PayPalButtonStyle>{
    let userId;
    this.userIdService
      .getUserId()
      .subscribe((occUserId) => (userId = occUserId))
      .unsubscribe();
    return this.paymentConnector.getPayPalButtonStyles(userId, buttonType);
  }

  public getBraintreeAssets(): Observable<BraintreeAssets>{
    return this.paymentConnector.getBraintreeAssets();
  }
}
