package de.hybris.platform.spartacussampledata.setup.impl;

import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.catalog.CatalogVersionService;
import de.hybris.platform.catalog.jalo.SyncItemCronJob;
import de.hybris.platform.catalog.jalo.SyncItemJob;
import de.hybris.platform.catalog.model.CatalogVersionModel;
import de.hybris.platform.catalog.model.SyncItemJobModel;
import de.hybris.platform.commerceservices.setup.AbstractSystemSetup;
import de.hybris.platform.commerceservices.setup.SetupImpexService;
import de.hybris.platform.commerceservices.util.ResponsiveUtils;
import de.hybris.platform.core.initialization.SystemSetupContext;
import de.hybris.platform.cronjob.enums.CronJobResult;
import de.hybris.platform.cronjob.enums.CronJobStatus;
import de.hybris.platform.jalo.enumeration.EnumerationValue;
import de.hybris.platform.servicelayer.config.ConfigurationService;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.commons.configuration.Configuration;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static de.hybris.platform.commerceservices.dataimport.AbstractDataImportService.IMPORT_MOBILE_DATA;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class SpaSampleDataImportServiceTest {

    private static final String IMPORT_ROOT = "importRoot";
    private static final String ELECTRONICS_CATALOG_NAME = "electronics";
    private static final String CONTENT_CATALOGS = "/contentCatalogs/";
    private static final String STAGED = "Staged";
    private static final String SPA_CONTENT_CATALOG = "-spaContentCatalog";
    private static final String CONTENT_CATALOG_CATALOG_IMPEX = "ContentCatalog/catalog.impex";
    private static final String CONTENT_CATALOG_CLEANING_IMPEX = "ContentCatalog/cleaning.impex";
    private static final String PRODUCT_CATALOG_SOLR_IMPEX = "ProductCatalog/solr.impex";
    private static final String PRODUCT_CATALOGS = "/productCatalogs/";
    private static final String CONTENT_CATALOG_SYNC_IMPEX = "ContentCatalog/sync.impex";
    private static final String CONTENT_CATALOG_EMAIL_CONTENT_IMPEX = "ContentCatalog/email-content.impex";
    private static final String ELECTRONICS_SPA_SYNC_FAIL_MESSAGE = "Catalog catalog [electronics->spa] sync has issues.";

    @Mock
    private ModelService modelService;
    @Mock
    private SystemSetupContext systemSetupContext;
    @Mock
    private SetupImpexService setupImpexService;
    @Mock
    private CatalogVersionService catalogVersionService;
    @Mock
    private CatalogVersionModel catalogVersionModel;
    @Mock
    private SyncItemJobModel syncItemJobModel;
    @Mock
    private SyncItemJob syncItemJobJalo;
    @Mock
    private SyncItemCronJob syncItemCronJob;
    @Mock
    private ConfigurationService configurationService;
    @Mock
    private Configuration configuration;
    @Mock
    private EnumerationValue cronJobResultEnumerationValue;
    @Mock
    private EnumerationValue cronJobStatusEnumerationValue;
    private List<SyncItemJobModel> incomingSync;

    @InjectMocks
    @Spy
    private SpaSampleDataImportService unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);

        when(catalogVersionService.getCatalogVersion(ELECTRONICS_CATALOG_NAME + SPA_CONTENT_CATALOG, STAGED))
                .thenReturn(catalogVersionModel);
        when(configurationService.getConfiguration()).thenReturn(configuration);
        when(configuration.getBoolean(IMPORT_MOBILE_DATA, Boolean.FALSE)).thenReturn(Boolean.FALSE);

        doNothing().when(setupImpexService).importImpexFile(IMPORT_ROOT + CONTENT_CATALOGS
                + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_CATALOG_IMPEX, false);
        doNothing().when(setupImpexService).importImpexFile(IMPORT_ROOT + CONTENT_CATALOGS
                + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_CLEANING_IMPEX, false);
        doNothing().when(setupImpexService).importImpexFile(IMPORT_ROOT + PRODUCT_CATALOGS
                + ELECTRONICS_CATALOG_NAME + PRODUCT_CATALOG_SOLR_IMPEX, false);
        doNothing().when((AbstractSystemSetup) unit).createContentCatalogSyncJob(
                systemSetupContext, ELECTRONICS_CATALOG_NAME + SPA_CONTENT_CATALOG);

        doReturn(null).when((AbstractSystemSetup) unit)
                .executeCatalogSyncJob(systemSetupContext, ELECTRONICS_CATALOG_NAME + SPA_CONTENT_CATALOG);
        doReturn(Boolean.FALSE).when((AbstractSystemSetup) unit).isSyncRerunNeeded(null);

        incomingSync = new ArrayList<>();
        incomingSync.add(syncItemJobModel);

        when(modelService.getSource(syncItemJobModel)).thenReturn(syncItemJobJalo);
        when(syncItemJobJalo.newExecution()).thenReturn(syncItemCronJob);
        doNothing().when(syncItemJobJalo).configureFullVersionSync(syncItemCronJob);
        doNothing().when(syncItemJobJalo).perform(syncItemCronJob, true);
        when(syncItemCronJob.getResult()).thenReturn(cronJobResultEnumerationValue);
        when(syncItemCronJob.getStatus()).thenReturn(cronJobStatusEnumerationValue);
        when(modelService.get(cronJobResultEnumerationValue)).thenReturn(CronJobResult.SUCCESS);
        when(modelService.get(cronJobStatusEnumerationValue)).thenReturn(CronJobStatus.FINISHED);
    }

    @Test
    public void shouldImportContentCatalogWhenCatalogHasNoIncomingSynchronizations() {
        when(catalogVersionModel.getIncomingSynchronizations()).thenReturn(Collections.emptyList());

        try (MockedStatic<ResponsiveUtils> responsiveUtilsMockedStatic = mockStatic(ResponsiveUtils.class)) {
            responsiveUtilsMockedStatic.when(ResponsiveUtils::isResponsive).thenReturn(Boolean.FALSE);

            unit.importContentCatalog(systemSetupContext, IMPORT_ROOT, ELECTRONICS_CATALOG_NAME);

            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_CATALOG_IMPEX, false);
            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_CLEANING_IMPEX, false);
            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + PRODUCT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + PRODUCT_CATALOG_SOLR_IMPEX, false);
            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_SYNC_IMPEX, false);
            verify(unit, Mockito.times(2))
                    .importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                            + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_EMAIL_CONTENT_IMPEX, false);
        }
    }

    @Test
    public void shouldImportContentCatalogWhenCatalogHasIncomingSynchronizations() {
        when(catalogVersionModel.getIncomingSynchronizations()).thenReturn(incomingSync);

        try (MockedStatic<ResponsiveUtils> responsiveUtilsMockedStatic = mockStatic(ResponsiveUtils.class)) {
            responsiveUtilsMockedStatic.when(ResponsiveUtils::isResponsive).thenReturn(Boolean.FALSE);

            unit.importContentCatalog(systemSetupContext, IMPORT_ROOT, ELECTRONICS_CATALOG_NAME);

            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_CATALOG_IMPEX, false);
            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_CLEANING_IMPEX, false);
            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + PRODUCT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + PRODUCT_CATALOG_SOLR_IMPEX, false);
            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_SYNC_IMPEX, false);
            verify(unit, Mockito.times(2))
                    .importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                            + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_EMAIL_CONTENT_IMPEX, false);
        }
    }

    @Test
    public void shouldLogInfoWhenCatalogHasIncomingSynchronizationsAndSyncCronJobFails() {
        when(catalogVersionModel.getIncomingSynchronizations()).thenReturn(incomingSync);
        when(modelService.get(cronJobResultEnumerationValue)).thenReturn(CronJobResult.FAILURE);

        try (MockedStatic<ResponsiveUtils> responsiveUtilsMockedStatic = mockStatic(ResponsiveUtils.class)) {
            responsiveUtilsMockedStatic.when(ResponsiveUtils::isResponsive).thenReturn(Boolean.FALSE);

            unit.importContentCatalog(systemSetupContext, IMPORT_ROOT, ELECTRONICS_CATALOG_NAME);

            verify(unit).logInfo(systemSetupContext, ELECTRONICS_SPA_SYNC_FAIL_MESSAGE);

            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_CATALOG_IMPEX, false);
            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_CLEANING_IMPEX, false);
            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + PRODUCT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + PRODUCT_CATALOG_SOLR_IMPEX, false);
            verify(unit).importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                    + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_SYNC_IMPEX, false);
            verify(unit, Mockito.times(2))
                    .importImpexFile(systemSetupContext, IMPORT_ROOT + CONTENT_CATALOGS
                            + ELECTRONICS_CATALOG_NAME + CONTENT_CATALOG_EMAIL_CONTENT_IMPEX, false);
        }
    }

}
