package com.paypal.filters;

import com.paypal.base.rest.APIContext;
import com.paypal.base.rest.PayPalRESTException;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.services.impl.DefaultPayPalWebservicesConfigurationService;
import de.hybris.bootstrap.annotations.UnitTest;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.util.Collections;
import java.util.Map;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_LIVE_ENVIRONMENT;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class ValidateWebhookFilterTest {

    private static final String POST_METHOD = "POST";
    private static final String GET_METHOD = "GET";
    private static final String CLIENT_ID = "clientId";
    private static final String SECRET_KEY = "secretKey";
    private static final String AUTHORIZATION_HEADER = "Authorization";
    private static final String EVENT_TYPE = "Update";
    private static final String WEBHOOK_ID = "webhookId";
    private static final String PAYPAL_SANDBOX_ENVIRONMENT = "sandbox";

    @Mock
    private HttpServletRequest request;
    @Mock
    private HttpServletResponse response;
    @Mock
    private FilterChain filterChain;
    @Mock
    private PayPalConfigurationService defaultPayPalConfigurationService;
    @Mock
    private DefaultPayPalWebservicesConfigurationService payPalWebservicesConfigurationService;

    @InjectMocks
    @Spy
    private ValidateWebhookFilter unit;

    @Before
    public void setUp() throws IOException {
        MockitoAnnotations.initMocks(this);

        when(request.getHeaderNames()).thenReturn(Collections.enumeration(Collections.singletonList(AUTHORIZATION_HEADER)));
        when(defaultPayPalConfigurationService.getEnvironmentType()).thenReturn(PAYPAL_LIVE_ENVIRONMENT);
        when(defaultPayPalConfigurationService.getClientID()).thenReturn(CLIENT_ID);
        when(defaultPayPalConfigurationService.getSecretKey()).thenReturn(SECRET_KEY);
        String json = "{\"id\":\"213213213\", \"event_type\":\"Update\"}";
        when(request.getReader()).thenReturn(new BufferedReader(new StringReader(json)));
        when(payPalWebservicesConfigurationService.getWebhookId(EVENT_TYPE)).thenReturn(WEBHOOK_ID);
    }

    @Test
    public void shouldDoFilterInternal() throws ServletException, IOException, PayPalRESTException, NoSuchAlgorithmException, SignatureException, InvalidKeyException {
        when(request.getMethod()).thenReturn(POST_METHOD);
        doReturn(true).when(unit).validateReceivedEvent(any(APIContext.class), any(Map.class), any(String.class));

        unit.doFilterInternal(request, response, filterChain);

        verify(payPalWebservicesConfigurationService).getWebhookId(EVENT_TYPE);
        verify(filterChain).doFilter(request, response);
    }

    @Test
    public void shouldDoFilterInternalWhenGetRequestMethod() throws ServletException, IOException {
        when(request.getMethod()).thenReturn(GET_METHOD);

        unit.doFilterInternal(request, response, filterChain);

        verify(payPalWebservicesConfigurationService).getWebhookId(EVENT_TYPE);
        verify(filterChain, times(0)).doFilter(request, response);
    }

    @Test
    public void shouldDoFilterInternalWhenReceivedEventNotValid() throws ServletException, IOException,
            PayPalRESTException, NoSuchAlgorithmException, SignatureException, InvalidKeyException {
        when(request.getMethod()).thenReturn(POST_METHOD);
        doReturn(false).when(unit).validateReceivedEvent(any(APIContext.class), any(Map.class), any(String.class));

        unit.doFilterInternal(request, response, filterChain);

        verify(payPalWebservicesConfigurationService).getWebhookId(EVENT_TYPE);
        verify(filterChain, times(0)).doFilter(request, response);
    }

    @Test
    public void shouldDoFilterInternalWhenValidationThrowsSignatureException() throws ServletException, IOException,
            PayPalRESTException, NoSuchAlgorithmException, SignatureException, InvalidKeyException {
        when(defaultPayPalConfigurationService.getEnvironmentType()).thenReturn(PAYPAL_SANDBOX_ENVIRONMENT);
        when(request.getMethod()).thenReturn(POST_METHOD);
        doThrow(SignatureException.class).when(unit).validateReceivedEvent(any(APIContext.class), any(Map.class), any(String.class));

        unit.doFilterInternal(request, response, filterChain);

        verify(payPalWebservicesConfigurationService).getWebhookId(EVENT_TYPE);
        verify(filterChain, times(0)).doFilter(request, response);
    }

}