package com.paypal.controllers.webhooks;

import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCartService;
import com.paypal.hybris.core.service.PayPalCommerceCheckoutService;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import com.paypal.hybris.data.PayPalCardDetailsMetadataData;
import com.paypal.hybris.data.PayPalGetCardDetailsResponseData;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import com.paypal.hybris.facade.facades.impl.DefaultPayPalCreditCardFacade;
import de.hybris.backoffice.PaypalWebhookData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.site.BaseSiteService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;

import static org.junit.Assert.assertEquals;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class WebhookListenerTest {

    private static final String CACHED_PARSED_REQUEST = "cachedParsedRequest";
    private static final String ID_KEY = "id";
    private static final String ID_VALUE = "id";
    private static final String EVENT_ID = "eventId";
    private static final String RESOURCE = "resource";
    private static final String CODE = "code";
    private static final String SUCCESS_BODY_RESPONSE = "200";
    private static final String ORDER_ID = "orderId";
    private static final String SUBSCRIPTION_ID = "subscriptionId";
    private static final String PAYMENT_AUTHORIZATION_VOIDED_EVENT_TYPE = "PAYMENT.AUTHORIZATION.VOIDED";
    private static final String PAYMENT_CAPTURE_COMPLETED_EVENT_TYPE = "PAYMENT.CAPTURE.COMPLETED";
    private static final String PAYMENT_CAPTURE_REFUNDED_EVENT_TYPE = "PAYMENT.CAPTURE.REFUNDED";
    private static final String EVENT_TYPE = "event_type";
    private static final String LINKS_KEY = "links";
    private static final String REL_KEY = "rel";
    private static final String UP = "up";
    private static final String CAPTURES = "captures/";
    private static final String HREF_KEY = "href";
    private static final String METADATA_KEY = "metadata";

    @Mock
    private HttpServletRequest request;
    @Mock
    private HttpServletResponse response;
    @Mock
    private PayPalCartService cartService;
    @Mock
    private CartModel cartModel;
    @Mock
    private BaseSiteService baseSiteService;
    @Mock
    private BaseSiteModel baseSiteModel;
    @Mock
    private PayPalAcceleratorCheckoutFacade payPalAcceleratorCheckoutFacade;
    @Mock
    private PaymentTransactionsService paymentTransactionsService;
    @Mock
    private DefaultPayPalCreditCardFacade payPalCreditCardFacade;
    @Mock
    private PayPalPaymentInfoService payPalPaymentInfoService;
    @Mock
    private PayPalCustomerAccountService payPalCustomerAccountService;
    @Mock
    private PayPalCreditCardPaymentInfoModel creditCardPaymentInfoModel;
    @Mock
    private PayPalCommerceCheckoutService commerceCheckoutService;

    @InjectMocks
    private WebhookListener unit;

    @Before
    public void setUp() {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldApproveOrder() throws InvalidCartException {
        OrderData orderData = new OrderData();
        orderData.setCode(CODE);
        CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setId(ID_VALUE);
        final Map<String, Object> parsedRequest = new HashMap<>();
        parsedRequest.put(ID_KEY, EVENT_ID);
        final Map<String, Object> resourceMap = new HashMap<>();
        resourceMap.put(ID_KEY, ORDER_ID);
        parsedRequest.put(RESOURCE, resourceMap);

        when(request.getAttribute(CACHED_PARSED_REQUEST)).thenReturn(parsedRequest);
        when(cartService.getCartByPayPalOrderIdForLocalPayments(ORDER_ID)).thenReturn(Optional.of(cartModel));
        when(payPalPaymentInfoService.getPaymentInfoByOrderId(ORDER_ID)).thenReturn(Optional.of(creditCardPaymentInfoModel));
        when(cartModel.getSite()).thenReturn(baseSiteModel);
        doNothing().when(baseSiteService).setCurrentBaseSite(baseSiteModel, true);
        when(commerceCheckoutService.setPaymentInfo(argThat(this::validateSetPaymentInfoCommerceCheckoutParameter))).thenReturn(true);
        doNothing().when(payPalAcceleratorCheckoutFacade).prepareCartForCheckout(cartModel);
        when(payPalAcceleratorCheckoutFacade.authorizePayment(ID_VALUE)).thenReturn(true);
        when(payPalAcceleratorCheckoutFacade.placeOrderByCart(cartModel)).thenReturn(orderData);

        ResponseEntity<String> result = unit.orderApproved(request, response);

        assertEquals(HttpStatus.OK, result.getStatusCode());
        assertEquals(SUCCESS_BODY_RESPONSE, result.getBody());
    }

    private boolean validateSetPaymentInfoCommerceCheckoutParameter(CommerceCheckoutParameter arg) {
        return cartModel.equals(arg.getCart())
                && creditCardPaymentInfoModel.equals(arg.getPaymentInfo())
                && arg.isEnableHooks();
    }

    @Test
    public void shouldNotApproveOrderWhenCartNotFound() throws InvalidCartException {
        OrderData orderData = new OrderData();
        orderData.setCode(CODE);
        CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setId(ID_VALUE);
        final Map<String, Object> parsedRequest = new HashMap<>();
        parsedRequest.put(ID_KEY, EVENT_ID);
        final Map<String, Object> resourceMap = new HashMap<>();
        resourceMap.put(ID_KEY, ORDER_ID);
        parsedRequest.put(RESOURCE, resourceMap);

        when(request.getAttribute(CACHED_PARSED_REQUEST)).thenReturn(parsedRequest);
        when(cartService.getCartByPayPalOrderIdForLocalPayments(ORDER_ID)).thenReturn(Optional.ofNullable(null));

        ResponseEntity<String> result = unit.orderApproved(request, response);

        assertEquals(HttpStatus.BAD_REQUEST, result.getStatusCode());
    }

    @Test
    public void shouldProcessPaymentsWhenAuthorizationVoidedEvent() throws IOException {
        OrderData orderData = new OrderData();
        orderData.setCode(CODE);
        final Map<String, Object> parsedRequest = getStringObjectMap();
        parsedRequest.put(EVENT_TYPE, PAYMENT_AUTHORIZATION_VOIDED_EVENT_TYPE);

        when(request.getAttribute(CACHED_PARSED_REQUEST)).thenReturn(parsedRequest);
        doNothing().when(paymentTransactionsService).doOnVoidPerformed(any(PaypalWebhookData.class));

        ResponseEntity<String> result = unit.processPayments(request, response);

        assertEquals(HttpStatus.OK, result.getStatusCode());
        assertEquals(SUCCESS_BODY_RESPONSE, result.getBody());
        verify(paymentTransactionsService).doOnVoidPerformed(any(PaypalWebhookData.class));
    }

    @Test
    public void shouldProcessPaymentsWhenCaptureCompletedEvent() throws IOException {
        OrderData orderData = new OrderData();
        orderData.setCode(CODE);
        final Map<String, Object> parsedRequest = getStringObjectMap();
        parsedRequest.put(EVENT_TYPE, PAYMENT_CAPTURE_COMPLETED_EVENT_TYPE);

        when(request.getAttribute(CACHED_PARSED_REQUEST)).thenReturn(parsedRequest);
        doNothing().when(paymentTransactionsService).doOnCapturePerformed(any(PaypalWebhookData.class));

        ResponseEntity<String> result = unit.processPayments(request, response);

        assertEquals(HttpStatus.OK, result.getStatusCode());
        assertEquals(SUCCESS_BODY_RESPONSE, result.getBody());
        verify(paymentTransactionsService).doOnCapturePerformed(any(PaypalWebhookData.class));
    }

    @Test
    public void shouldProcessPaymentsWhenCaptureRefundedEvent() throws IOException {
        OrderData orderData = new OrderData();
        orderData.setCode(CODE);
        final Map<String, Object> parsedRequest = getStringObjectMap();

        when(request.getAttribute(CACHED_PARSED_REQUEST)).thenReturn(parsedRequest);
        doNothing().when(paymentTransactionsService).doOnRefundPerformed(any(PaypalWebhookData.class),
                any(String.class));

        ResponseEntity<String> result = unit.processPayments(request, response);

        assertEquals(HttpStatus.OK, result.getStatusCode());
        assertEquals(SUCCESS_BODY_RESPONSE, result.getBody());
        verify(paymentTransactionsService).doOnRefundPerformed(any(PaypalWebhookData.class), any(String.class));
    }

    @Test
    public void shouldProcessPaymentsWhenCaptureRefundedEventAndEmptyLinks() throws IOException {
        OrderData orderData = new OrderData();
        orderData.setCode(CODE);
        final Map<String, Object> parsedRequest = getStringObjectMap();

        when(request.getAttribute(CACHED_PARSED_REQUEST)).thenReturn(parsedRequest);
        doNothing().when(paymentTransactionsService).doOnRefundPerformed(any(PaypalWebhookData.class),
                any(String.class));

        ResponseEntity<String> result = unit.processPayments(request, response);

        assertEquals(HttpStatus.OK, result.getStatusCode());
        assertEquals(SUCCESS_BODY_RESPONSE, result.getBody());
        verify(paymentTransactionsService).doOnRefundPerformed(any(PaypalWebhookData.class), any(String.class));
    }

    private static Map<String, Object> getStringObjectMap() {
        final Map<String, Object> parsedRequest = new HashMap<>();
        parsedRequest.put(ID_KEY, EVENT_ID);
        parsedRequest.put(EVENT_TYPE, PAYMENT_CAPTURE_REFUNDED_EVENT_TYPE);

        final Map<String, Object> resourceMap = new HashMap<>();
        resourceMap.put(ID_KEY, ORDER_ID);

        ArrayList<LinkedHashMap<String, String>> links = new ArrayList<>();
        LinkedHashMap<String, String> link = new LinkedHashMap<>();
        link.put(REL_KEY, UP);
        link.put(HREF_KEY, "/path" + CAPTURES + "123");
        links.add(link);
        resourceMap.put(LINKS_KEY, links);

        parsedRequest.put(RESOURCE, resourceMap);
        return parsedRequest;
    }

    @Test
    public void shouldDeletePaymentToken() {
        OrderData orderData = new OrderData();
        orderData.setCode(CODE);
        CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setId(ID_VALUE);
        final Map<String, Object> parsedRequest = new HashMap<>();
        parsedRequest.put(ID_KEY, EVENT_ID);
        final Map<String, Object> resourceMap = new HashMap<>();
        resourceMap.put(ID_KEY, SUBSCRIPTION_ID);
        parsedRequest.put(RESOURCE, resourceMap);

        when(request.getAttribute(CACHED_PARSED_REQUEST)).thenReturn(parsedRequest);
        doNothing().when(payPalCreditCardFacade).deleteCreditCardFromWebhookEvent(SUBSCRIPTION_ID);

        ResponseEntity<String> result = unit.paymentTokenDeleted(request, response);

        assertEquals(HttpStatus.OK, result.getStatusCode());
        assertEquals(SUCCESS_BODY_RESPONSE, result.getBody());
        verify(payPalCreditCardFacade).deleteCreditCardFromWebhookEvent(SUBSCRIPTION_ID);
    }

    @Test
    public void shouldCreateVault() throws IOException {
        OrderData orderData = new OrderData();
        orderData.setCode(CODE);
        CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setId(ID_VALUE);
        final Map<String, Object> parsedRequest = new HashMap<>();
        parsedRequest.put(ID_KEY, EVENT_ID);
        final Map<String, Object> resourceMap = new HashMap<>();
        PayPalGetCardDetailsResponseData responseData = new PayPalGetCardDetailsResponseData();
        PayPalCardDetailsMetadataData payPalCardDetailsMetadataData = new PayPalCardDetailsMetadataData();
        payPalCardDetailsMetadataData.setOrderId(ORDER_ID);
        responseData.setMetadata(payPalCardDetailsMetadataData);
        resourceMap.put(METADATA_KEY, payPalCardDetailsMetadataData);
        parsedRequest.put(RESOURCE, resourceMap);

        when(request.getAttribute(CACHED_PARSED_REQUEST)).thenReturn(parsedRequest);
        when(payPalPaymentInfoService.getPaymentInfoByOrderId(ORDER_ID))
                .thenReturn(Optional.of(creditCardPaymentInfoModel));
        doNothing().when(payPalCustomerAccountService).updateCreditCardInfo(
                any(PayPalGetCardDetailsResponseData.class), any(PayPalCreditCardPaymentInfoModel.class));

        ResponseEntity<String> result = unit.vaultCreated(request);

        assertEquals(HttpStatus.OK, result.getStatusCode());
        assertEquals(SUCCESS_BODY_RESPONSE, result.getBody());
        verify(payPalPaymentInfoService).getPaymentInfoByOrderId(ORDER_ID);
        verify(payPalCustomerAccountService).updateCreditCardInfo(any(PayPalGetCardDetailsResponseData.class),
                any(PayPalCreditCardPaymentInfoModel.class));
    }

    @Test
    public void shouldCreateVaultWhenOrderWhenNoMetadata() throws IOException {
        OrderData orderData = new OrderData();
        orderData.setCode(CODE);
        CCPaymentInfoData paymentInfoData = new CCPaymentInfoData();
        paymentInfoData.setId(ID_VALUE);
        final Map<String, Object> parsedRequest = new HashMap<>();
        parsedRequest.put(ID_KEY, EVENT_ID);
        final Map<String, Object> resourceMap = new HashMap<>();
        parsedRequest.put(RESOURCE, resourceMap);

        when(request.getAttribute(CACHED_PARSED_REQUEST)).thenReturn(parsedRequest);
        when(payPalPaymentInfoService.getPaymentInfoByOrderId(ORDER_ID))
                .thenReturn(Optional.of(creditCardPaymentInfoModel));
        doNothing().when(payPalCustomerAccountService).updateCreditCardInfo(
                any(PayPalGetCardDetailsResponseData.class), any(PayPalCreditCardPaymentInfoModel.class));

        ResponseEntity<String> result = unit.vaultCreated(request);

        assertEquals(HttpStatus.OK, result.getStatusCode());
        assertEquals(SUCCESS_BODY_RESPONSE, result.getBody());
        verify(payPalPaymentInfoService, times(0)).getPaymentInfoByOrderId(ORDER_ID);
        verify(payPalCustomerAccountService, times(0)).updateCreditCardInfo(any(PayPalGetCardDetailsResponseData.class),
                any(PayPalCreditCardPaymentInfoModel.class));
    }

}
