package com.paypal.filters;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.paypal.api.payments.Event;
import com.paypal.base.Constants;
import com.paypal.base.rest.APIContext;
import com.paypal.base.rest.PayPalRESTException;
import com.paypal.core.PayPalEnvironment;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.services.impl.DefaultPayPalWebservicesConfigurationService;

import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.http.HttpStatus;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_LIVE_ENVIRONMENT;

public class ValidateWebhookFilter extends OncePerRequestFilter {

    private static final String POST_METHOD = "POST";
    private static final String EVENT_TYPE = "event_type";
    private static final String ID = "id";
    private static final String RECEIVED_WEBHOOK_MSG = "The webhook [%s] with event type [%s] and id [%s] was received.";
    private static final String FAILED_WEBHOOK_VALIDATION = "The incoming request is incorrect! Webhook with id [%s] and with event [%s] failed validation.";
    private static final String FAILED_WEBHOOK_VALIDATION_WITH_ERROR = "Webhook with id [%s] and with event [%s] failed validation with error message: %s";
    private static final String CACHED_PARSED_REQUEST = "cachedParsedRequest";
    private static final Logger LOG = Logger.getLogger(ValidateWebhookFilter.class);
    private static final ObjectMapper objectMapper = new ObjectMapper();

    private PayPalConfigurationService defaultPayPalConfigurationService;

    private DefaultPayPalWebservicesConfigurationService payPalWebservicesConfigurationService;

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        boolean isWebhookValid = Boolean.FALSE;
        final APIContext apiContext = createAPIContext();
        Map<String, String> headers = getHeaders(request);
        String body = getBody(request);

        final Map<String, Object> parsedRequest = objectMapper.readValue(body, Map.class);
        final String eventType = (String) parsedRequest.get(EVENT_TYPE);
        final String eventId = (String) parsedRequest.get(ID);
        final String webhookId = payPalWebservicesConfigurationService.getWebhookId(eventType);
        LOG.info(String.format(RECEIVED_WEBHOOK_MSG, webhookId, eventType, eventId));
        LOG.info("Headers -> " + headers);
        LOG.info("Body -> " + body);
        try {
            apiContext.addConfiguration(Constants.PAYPAL_WEBHOOK_ID, webhookId);
            isWebhookValid = request.getMethod().equalsIgnoreCase(POST_METHOD)
                    && validateReceivedEvent(apiContext, headers, body);
        } catch (PayPalRESTException | InvalidKeyException | NoSuchAlgorithmException | SignatureException e) {
            LOG.info(e.getMessage(), e);
            LOG.error(String.format(FAILED_WEBHOOK_VALIDATION_WITH_ERROR, webhookId, eventType, e.getMessage()));
        }
        if (isWebhookValid) {
            request.setAttribute(CACHED_PARSED_REQUEST, parsedRequest);
            filterChain.doFilter(request, response);
        } else {
            LOG.info(String.format(FAILED_WEBHOOK_VALIDATION, webhookId, eventType));
            response.setStatus(HttpStatus.BAD_REQUEST.value());
        }
    }

    protected boolean validateReceivedEvent(APIContext apiContext, Map<String, String> headers, String body) throws PayPalRESTException, NoSuchAlgorithmException, SignatureException, InvalidKeyException {
        return Event.validateReceivedEvent(apiContext, headers, body);
    }

    private PayPalEnvironment createPayPalEnvironment() {
        if (StringUtils.equals(PAYPAL_LIVE_ENVIRONMENT, defaultPayPalConfigurationService.getEnvironmentType())) {
            return new PayPalEnvironment.Live(
                    defaultPayPalConfigurationService.getClientID(),
                    defaultPayPalConfigurationService.getSecretKey());
        }
        return new PayPalEnvironment.Sandbox(
                defaultPayPalConfigurationService.getClientID(),
                defaultPayPalConfigurationService.getSecretKey());
    }

    private APIContext createAPIContext() {
        final PayPalEnvironment payPalEnvironment = createPayPalEnvironment();
        return new APIContext(payPalEnvironment.clientId(), payPalEnvironment.clientSecret(),
                defaultPayPalConfigurationService.getEnvironmentType());
    }

    private static Map<String, String> getHeaders(HttpServletRequest request) {
        Map <String, String> map = new HashMap<>();
        @SuppressWarnings("rawtypes")
        Enumeration headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String key = (String) headerNames.nextElement();
            String value = request.getHeader(key);
            map.put(key, value);
        }
        return map;
    }

    private String getBody(HttpServletRequest request) throws IOException {
        StringBuilder sb = new StringBuilder();
        BufferedReader reader = request.getReader();
        String line;
        while ((line = reader.readLine()) != null) {
            sb.append(line);
        }
        return sb.toString();
    }

    public void setDefaultPayPalConfigurationService(PayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    public void setPayPalWebservicesConfigurationService(DefaultPayPalWebservicesConfigurationService payPalWebservicesConfigurationService) {
        this.payPalWebservicesConfigurationService = payPalWebservicesConfigurationService;
    }
}
