package com.paypal.controllers.webhooks;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCartService;
import com.paypal.hybris.core.service.PayPalCommerceCheckoutService;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import com.paypal.hybris.data.PayPalGetCardDetailsResponseData;
import com.paypal.hybris.facade.facades.PayPalAcceleratorCheckoutFacade;
import com.paypal.hybris.facade.facades.PayPalCreditCardFacade;
import de.hybris.backoffice.PaypalWebhookData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.order.InvalidCartException;
import de.hybris.platform.site.BaseSiteService;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;

@Controller
@RequestMapping(value = "/paypal/webhook")
public class WebhookListener {

    private static final String PAYMENT_AUTHORIZATION_VOIDED_EVENT_TYPE = "PAYMENT.AUTHORIZATION.VOIDED";
    private static final String PAYMENT_CAPTURE_COMPLETED_EVENT_TYPE = "PAYMENT.CAPTURE.COMPLETED";
    private static final String PAYMENT_CAPTURE_REFUNDED_EVENT_TYPE = "PAYMENT.CAPTURE.REFUNDED";
    private static final String RESOURCE = "resource";
    private static final String ID = "id";
    private static final String CAPTURES = "captures/";
    private static final String REL = "rel";
    private static final String UP = "up";
    private static final String HREF = "href";
    private static final String SUCCESS_BODY_RESPONSE = "200";
    private static final String CACHED_PARSED_REQUEST = "cachedParsedRequest";
    private static final Logger LOG = Logger.getLogger(WebhookListener.class);
    private static final ObjectMapper mapper = new ObjectMapper();

    @Resource(name = "defaultPayPalAcceleratorCheckoutFacade")
    private PayPalAcceleratorCheckoutFacade payPalAcceleratorCheckoutFacade;

    @Resource(name = "payPalCartService")
    private PayPalCartService cartService;

    @Resource(name = "payPalPaymentTransactionsService")
    private PaymentTransactionsService paymentTransactionsService;

    @Resource(name = "baseSiteService")
    private BaseSiteService baseSiteService;

    @Resource(name = "paymentInfoService")
    private PayPalPaymentInfoService payPalPaymentInfoService;

    @Resource(name = "payPalCreditCardFacade")
    private PayPalCreditCardFacade payPalCreditCardFacade;
    @Resource
    private PayPalCustomerAccountService payPalCustomerAccountService;
    @Resource(name = "commerceCheckoutService")
    private PayPalCommerceCheckoutService payPalCommerceCheckoutService;

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @RequestMapping(value = "/order/approved")
    public ResponseEntity<String> orderApproved(final HttpServletRequest request,
                                                final HttpServletResponse response) throws InvalidCartException {
        final Map<String, Object> parsedRequest = (Map<String, Object>) request.getAttribute(CACHED_PARSED_REQUEST);
        final String eventId = (String) parsedRequest.get(ID);
        final String orderId = getIdFromResource(parsedRequest);
        Optional<CartModel> foundCart = cartService.getCartByPayPalOrderIdForLocalPayments(orderId);
        Optional<PayPalCreditCardPaymentInfoModel> foundPaymentInfo = payPalPaymentInfoService.getPaymentInfoByOrderId(orderId);
        if (foundCart.isPresent() && foundPaymentInfo.isPresent()) {
            CartModel cart = foundCart.get();
            PaymentInfoModel paymentInfo = foundPaymentInfo.get();

            baseSiteService.setCurrentBaseSite(cart.getSite(), true);
            setPaymentInfoToCart(cart, paymentInfo);
            payPalAcceleratorCheckoutFacade.prepareCartForCheckout(cart);
            payPalAcceleratorCheckoutFacade.authorizePayment(cart);
            final OrderData orderData = payPalAcceleratorCheckoutFacade.placeOrderByCart(cart);
            LOG.info(orderData.getCode() + " was placed due to webhook event: " + eventId);
            return ResponseEntity.ok(SUCCESS_BODY_RESPONSE);
        }
        return ResponseEntity.badRequest().build();
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @RequestMapping(value = "/payments")
    public ResponseEntity<String> processPayments(final HttpServletRequest request, final HttpServletResponse response) throws IOException {
        final Map<String, Object> parsedRequest = (Map<String, Object>) request.getAttribute(CACHED_PARSED_REQUEST);
        final PaypalWebhookData paypalWebhookData = mapper.readValue(mapper.writeValueAsString(parsedRequest), PaypalWebhookData.class);
        processEventType(paypalWebhookData.getEventType(), paypalWebhookData);
        return ResponseEntity.ok(SUCCESS_BODY_RESPONSE);
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @RequestMapping(value = "/payments/deleted")
    public ResponseEntity<String> paymentTokenDeleted(final HttpServletRequest request, final HttpServletResponse response) {
        final Map<String, Object> parsedRequest = (Map<String, Object>) request.getAttribute(CACHED_PARSED_REQUEST);
        final String subscriptionId = getIdFromResource(parsedRequest);
        payPalCreditCardFacade.deleteCreditCardFromWebhookEvent(subscriptionId);
        LOG.info("Payment with subscription id: " + subscriptionId +
                " has been deleted by vault payment token deleted webhook");
        return ResponseEntity.ok(SUCCESS_BODY_RESPONSE);
    }

    @Secured({"ROLE_CLIENT", "ROLE_CUSTOMERGROUP", "ROLE_GUEST", "ROLE_CUSTOMERMANAGERGROUP", "ROLE_TRUSTED_CLIENT", "ROLE_ANONYMOUS"})
    @RequestMapping(value = "/vault/created")
    public ResponseEntity<String> vaultCreated(HttpServletRequest request) throws IOException {
        final Map<String, Object> parsedRequest = (Map<String, Object>) request.getAttribute(CACHED_PARSED_REQUEST);
        LOG.info("[VAULT.PAYMENT-TOKEN.CREATED] Request: " + parsedRequest.toString());

        final Map<String, Object> resource = (Map<String, Object>) parsedRequest.get(RESOURCE);
        ObjectMapper mapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        PayPalGetCardDetailsResponseData payPalGetCardDetails = mapper.readValue(mapper.writeValueAsString(resource),
                PayPalGetCardDetailsResponseData.class);

        String orderId = Optional.ofNullable(payPalGetCardDetails.getMetadata())
                .map(metadataData -> metadataData.getOrderId())
                .orElse(StringUtils.EMPTY);
        if (StringUtils.isNotEmpty(orderId)) {
            payPalPaymentInfoService.getPaymentInfoByOrderId(orderId).ifPresent(foundPaymentInfo -> {
                payPalCustomerAccountService.updateCreditCardInfo(payPalGetCardDetails, foundPaymentInfo);
            });
        }

        return ResponseEntity.ok(SUCCESS_BODY_RESPONSE);
    }

    private void setPaymentInfoToCart(CartModel cart, PaymentInfoModel paymentInfo) {
        CommerceCheckoutParameter parameter = new CommerceCheckoutParameter();
        parameter.setEnableHooks(true);
        parameter.setCart(cart);
        parameter.setPaymentInfo(paymentInfo);
        payPalCommerceCheckoutService.setPaymentInfo(parameter);
    }

    private void processEventType(String eventType, PaypalWebhookData paypalWebhookData) {
        if (eventType.equalsIgnoreCase(PAYMENT_AUTHORIZATION_VOIDED_EVENT_TYPE)) {
            paymentTransactionsService.doOnVoidPerformed(paypalWebhookData);
        } else if (eventType.equalsIgnoreCase(PAYMENT_CAPTURE_COMPLETED_EVENT_TYPE)) {
            paymentTransactionsService.doOnCapturePerformed(paypalWebhookData);
        } else if (eventType.equalsIgnoreCase(PAYMENT_CAPTURE_REFUNDED_EVENT_TYPE)) {
            paymentTransactionsService.doOnRefundPerformed(paypalWebhookData, getCaptureIdFromLink(paypalWebhookData));
        }
    }

    private static String getIdFromResource(Map<String, Object> parsedRequest) {
        final Map<String, Object> resource = (Map<String, Object>) parsedRequest.get(RESOURCE);
        return (String) resource.get(ID);
    }

    private String getCaptureIdFromLink(PaypalWebhookData paypalWebhookData) {
        String captureId = StringUtils.EMPTY;
        final ArrayList<LinkedHashMap<String, String>> linksArray = paypalWebhookData.getResource().getLinks();
        for (final LinkedHashMap<String, String> entry : linksArray) {
            if (entry.get(REL).equals(UP)) {
                final String href = entry.get(HREF);
                captureId = href
                        .substring(href.lastIndexOf(CAPTURES) + CAPTURES.length());
            }
        }
        return captureId;
    }

}
